// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief


#ifndef INCLUDED_core_scoring_constraints_Constraints_HH
#define INCLUDED_core_scoring_constraints_Constraints_HH

// Unit headers
#include <core/scoring/constraints/Constraints.fwd.hh>

#include <core/types.hh>

#include <core/scoring/constraints/Constraint.hh>

#include <core/scoring/EnergyMap.fwd.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Conformation.hh>
#include <core/id/AtomID.hh>

// Utility Headers
#include <numeric/xyzVector.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++ Headers

namespace core {
namespace scoring {
namespace constraints {

class Constraints : public utility::pointer::ReferenceCount {
public:
	typedef id::AtomID AtomID;
	typedef conformation::Residue Residue;
	typedef conformation::Conformation Conformation;
	typedef ConstraintCOPs::const_iterator const_iterator;

public:

	Constraints();
	Constraints( Constraints const & );
	ConstraintsOP clone() const;
	Constraints const & operator = ( Constraints const & );

	// call the setup_for_derivatives for each constraint
 	void
 	setup_for_scoring( pose::Pose &pose, ScoreFunction const &scfxn ) const;

	// call the setup_for_derivatives for each constraint
 	void
 	setup_for_derivatives( pose::Pose &pose, ScoreFunction const &scfxn ) const;

	/// will fail if Residues dont contain all the necessary atoms
	void
	residue_pair_energy(
		Residue const & rsd1,
		Residue const & rsd2,
		EnergyMap const & weights,
 		EnergyMap & emap
	) const;

	/// will fail if Residue doesnt contain all the necessary atoms
	void
	intra_residue_energy(
		Residue const & rsd,
		EnergyMap const & weights,
		EnergyMap & emap
	) const;

	void
	conformation_energy(
		Conformation const & conformation,
		EnergyMap const & weights,
		EnergyMap & emap
	) const;

	///
 	void
 	eval_atom_derivative(
 		AtomID const & atom_id,
 		Conformation const & conformation,
 		EnergyMap const & weights,
 		Vector & F1,
 		Vector & F2
 	) const;


	///
	void
	add_constraint( ConstraintCOP cst );

	/// @brief Returns true iff the constraint was successfully found and removed.
	bool
	remove_constraint( ConstraintCOP cst );

	const_iterator begin() const { return constraints_.begin(); }
	const_iterator end() const { return constraints_.end(); }

	void
	show( std::ostream& out );

	void
	show_definition( std::ostream& out, pose::Pose const& pose ) const;

	virtual Size
	show_violations( std::ostream& out, pose::Pose const& pose, Size verbose_level, core::Real threshold = 1  );

	Size
	size() const
	{
		return constraints_.size();
	}

	void
	clear()
	{
		constraints_.clear();
	}

private:
	void
	energy( XYZ_Func const & xyz_func, EnergyMap const & weights, EnergyMap & emap ) const;

	// There's no implementation for this method in the .cc file ...
	//void
	//add_residue_pair_constraint( Size const pos1, Size const pos2, ConstraintCOP cst );

	void
	copy_from( Constraints const & );

private:

	ConstraintCOPs constraints_;

};


} // constraints
} // scoring
} // core

#endif
