// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/constraints/ConstraintsEnergy.hh
/// @brief  Constraints Energy Method declaration
/// @author Andrew Leaver-Fay

// Unit headers
#include <core/scoring/constraints/ConstraintsEnergy.hh>
#include <core/scoring/constraints/ConstraintsEnergyCreator.hh>

// Package headers
#include <core/scoring/constraints/ConstraintEnergyContainer.hh>
#include <core/scoring/constraints/ConstraintSet.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
//#include <core/scoring/ScoringManager.hh>

// utility headers
#include <core/util/Tracer.hh>
#include <core/util/prof.hh>


namespace core {
namespace scoring {
namespace constraints {

/// @details This must return a fresh instance of the ConstraintsEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
ConstraintsEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new ConstraintsEnergy;
}

ScoreTypes
ConstraintsEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( atom_pair_constraint );
	sts.push_back( angle_constraint );
	sts.push_back( dihedral_constraint );
	sts.push_back( constant_constraint );
	sts.push_back( coordinate_constraint );
	sts.push_back( dof_constraint );
	sts.push_back( res_type_constraint );
	sts.push_back( backbone_stub_constraint );
	sts.push_back( big_bin_constraint );
	sts.push_back( rna_bond_geometry );
	return sts;
}


static util::Tracer tr( "core.scoring.ConstraintsEnergy");

ConstraintsEnergy::ConstraintsEnergy() : parent( new ConstraintsEnergyCreator ) {}

ConstraintsEnergy::~ConstraintsEnergy() {}

methods::EnergyMethodOP
ConstraintsEnergy::clone() const
{
	return new ConstraintsEnergy;
}

methods::LongRangeEnergyType
ConstraintsEnergy::long_range_type() const { return methods::constraints_lr; }


bool
ConstraintsEnergy::defines_residue_pair_energy(
	pose::Pose const & pose,
	Size res1,
	Size res2
) const
{
	return pose.constraint_set()->residue_pair_constraint_exists( res1, res2 );
}


/////////////////////////////////////////////////////////////////////////////
// methods for ContextIndependentTwoBodyEnergies
/////////////////////////////////////////////////////////////////////////////

/// @brief Intraresidue constraints can exist; the ConstraintsEnergy class
/// cannot tell whether or not any intraresidue constraints exist when only
/// given the weight set, so for safety it says "yes, I define intraresidue
/// energies" and guarantees they will be evaluated properly if they do exist.
bool
ConstraintsEnergy::defines_intrares_energy( EnergyMap const & ) const
{
	return true;
}

/// @brief Evaluate the intra-residue constraint energy for a given residue
void
ConstraintsEnergy::eval_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap & emap
) const
{
	if ( pose.constraint_set() == 0 ) return;
	PROF_START( util::CONSTRAINT_SCORE );
	pose.constraint_set()->eval_intrares_energy( rsd, pose, sfxn, emap );
	PROF_STOP( util::CONSTRAINT_SCORE );
}


// @brief store a constraint graph in the pose.energies object, or if one is
// already present, make sure that the constraint graph accurately reflects
// the constraint set.  The constraint energy container keeps the revision id
// for a particular constraint set.  If a pose constraint set changes, either
// in identity (i.e. a new object) or in composition (i.e. an addition/removal
// of a constraint) then thene ConstraintsEnergy object throws away the old
// ConstraintsEnergyContainer and creats a new one... somewhat wasteful.
//
// In the future, either the CE or the ConstraintSet will have to intelligently
// decide between a constraint graph or a constraint table.  In the case of
// dense residue pair constraints (i.e. all pairs), a table would be more efficient.
void
ConstraintsEnergy::setup_for_scoring( pose::Pose & pose, ScoreFunction const & sfxn ) const
{
	using namespace methods;

	//do nothing if there are no constraints
	if ( pose.constraint_set() == 0 ) return; //this will never happen: since empty constraint_set is created as soon as method constraint_set() is called
	// however, if one jumps out here after asking !has_constraints() it breaks in ScoringFunction.cc:604 because the
	// long_rang_container is not setup for constraints.

	PROF_START( util::CONSTRAINT_SCORE );
	Energies & energies( pose.energies() );
	bool create_new_cstcontainer( false );
	if ( energies.long_range_container( constraints_lr ) == 0 ) {
		create_new_cstcontainer = true; // pbmod
	} else {
		LREnergyContainerOP lrc = energies.nonconst_long_range_container( constraints_lr );
		CstEnergyContainerOP cec( static_cast< CstEnergyContainer * > ( lrc.get() ) );
		if ( ! cec->matches( pose.constraint_set()) ) {
			create_new_cstcontainer = true;
		}
	}

	if ( create_new_cstcontainer ) {
		CstEnergyContainerOP new_cec = new CstEnergyContainer( pose );
		energies.set_long_range_container( constraints_lr, new_cec );
	}

	pose.constraint_set()->setup_for_scoring( pose, sfxn );  //fpd

	PROF_STOP( util::CONSTRAINT_SCORE );
}

///@brief Setup constraint-set specific derivatives
void
ConstraintsEnergy::setup_for_derivatives( pose::Pose & pose, ScoreFunction const & sfxn ) const
{
	PROF_START( util::CONSTRAINT_SCORE );
	pose.constraint_set()->setup_for_derivatives( pose, sfxn );  //fpd
	PROF_STOP( util::CONSTRAINT_SCORE );
}


/// @brief Evaluate constraint residue_pair_energy.  Defers entirely to cst_set.
void
ConstraintsEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap & emap
) const
{
	if ( pose.constraint_set() == 0 ) return;
	PROF_START( util::CONSTRAINT_SCORE );
	pose.constraint_set()->residue_pair_energy( rsd1, rsd2, pose, sfxn, emap );
	PROF_STOP( util::CONSTRAINT_SCORE );
}

/// @brief called at the end of energy evaluation; allows
/// for the evaluation of constraints that cannot be
/// decomposed into residue pairs.  Defers entirely to cst_set.
void
ConstraintsEnergy::finalize_total_energy(
	pose::Pose & pose,
	ScoreFunction const & sfxn,
	EnergyMap & totals
) const
{
	if ( pose.constraint_set() == 0 ) return;
	PROF_START( util::CONSTRAINT_SCORE );
	pose.constraint_set()->eval_non_residue_pair_energy( pose, sfxn, totals );
	PROF_STOP( util::CONSTRAINT_SCORE );
}


/// called during gradient-based minimization inside dfunc
/**
	 F1 and F2 are not zeroed -- contributions from this atom are
	 just summed in
**/
void
ConstraintsEnergy::eval_atom_derivative(
	id::AtomID const & id,
	pose::Pose const & pose,
	kinematics::DomainMap const &, // domain_map,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	if ( !pose.constraint_set()->has_constraints() ) return;
	PROF_START( util::CONSTRAINT_SCORE );
	pose.constraint_set()->eval_atom_derivative( id, pose, sfxn, weights, F1, F2 );
	PROF_STOP( util::CONSTRAINT_SCORE );
}

/// uses the dof constraints
Real
ConstraintsEnergy::eval_dof_derivative(
	id::DOF_ID const & id,
	id::TorsionID const & tor,
	pose::Pose const & pose,
	ScoreFunction const & scorefxn,
	EnergyMap const & weights
) const
{
	// if ( pose.constraint_set() == 0 ) return 0.0;
	//this -- oddly -- never happens, since the pose will make an empty set as soon as you ask.

	// instead...
	if ( ! pose.constraint_set()->has_constraints() ) return 0.0;
	PROF_START( util::CONSTRAINT_SCORE );
	Real result ( pose.constraint_set()->eval_dof_derivative( id, tor, pose, scorefxn, weights ) );
	PROF_STOP( util::CONSTRAINT_SCORE );
	return result;
}


/// @brief constraints are context independent
void
ConstraintsEnergy::indicate_required_context_graphs(
	utility::vector1< bool > &
) const
{}





} // constraints
} // scoring
} // core
