// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

// Package Headers
#include <core/scoring/constraints/FuncFactory.hh>
#include <core/scoring/constraints/CoordinateConstraint.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/pose/Pose.hh>
#include <core/id/AtomID.hh>

#include <core/conformation/Conformation.hh>

// Utility Headers
#include <core/util/Tracer.hh>


// C++ Headers
#include <cstdlib>
#include <iostream>
#include <utility>


static core::util::Tracer tr("core.io.constraints");

namespace core {
namespace scoring {
namespace constraints {

Size
CoordinateConstraint::show_violations(
	std::ostream & out,
	pose::Pose const & pose,
	Size verbose_level,
	Real threshold
) const{
  if ( verbose_level > 80 ) {
    out << "CoordConstraint ("
		<< pose.residue_type(atom_.rsd() ).atom_name( atom_.atomno() )
		<< ":" << atom_.atomno() << "," << atom_.rsd() << " ) ";
  }
  return func_->show_violations( out, dist( pose ), verbose_level, threshold );
}


/// @brief Copies the data from this Constraint into a new object and returns an OP
/// atoms are mapped to atoms with the same name in dest pose ( e.g. for switch from centroid to fullatom )
/// if a sequence_mapping is present it is used to map residue numbers .. NULL = identity mapping
/// to the new object. Intended to be implemented by derived classes.
ConstraintOP CoordinateConstraint::remapped_clone( pose::Pose const& src, pose::Pose const& dest, sequence::SequenceMappingCOP smap ) const {
	id::NamedAtomID atom1( atom(1), src );
  id::NamedAtomID atom2( atom(2), src );

	if ( smap ) {
		atom1.rsd() = (*smap)[ atom(1).rsd() ];
		atom2.rsd() = (*smap)[ atom(2).rsd() ];
	}

	//get AtomIDs for target pose
	id::AtomID id1( atom1, dest );
	id::AtomID id2( atom2, dest );
	if ( id1.valid() && id2.valid() ) {
		return new CoordinateConstraint( id1, id2, xyz_target_, func_, score_type() );
	} else {
		return NULL;
	}
}

Real
CoordinateConstraint::dist( pose::Pose const & pose ) const {
  conformation::Conformation const & conformation( pose.conformation() );
  Vector const & xyz( conformation.xyz( atom_ ) );
  return xyz.distance( xyz_target_ );
}

ConstraintOP
CoordinateConstraint::remap_resid( core::sequence::SequenceMapping const &seqmap ) const
{

	if ( seqmap[atom_.rsd()] != 0 && seqmap[fixed_atom_.rsd()] != 0 ) {
    AtomID remap_a( atom_.atomno(), seqmap[atom_.rsd()] ),
      remap_fa( fixed_atom_.atomno(), seqmap[fixed_atom_.rsd()] );
    return ConstraintOP( new CoordinateConstraint( remap_a, remap_fa, xyz_target_, this->func_ ) );
  } else {
    return NULL;
  }

}

void
CoordinateConstraint::steal_def( pose::Pose const& pose ) {
	conformation::Conformation const & conformation( pose.conformation() );
  xyz_target_ = conformation.xyz( atom_ );
}

///@details one line definition "Coordinates atom1 res1 atom2 res2 function_type function_definition"
void
CoordinateConstraint::read_def(
	std::istream& data,
	core::pose::Pose const& pose,
	FuncFactory const& func_factory
) {
	Size res1, res2;
	std::string name1, name2;
	std::string func_type;
	std::string type;

	data
		>> name1 >> res1
		>> name2 >> res2
		>> xyz_target_.x()
		>> xyz_target_.y()
		>> xyz_target_.z()
		>> func_type;

	tr.Debug << "read: " << name1 << " " << name2 << " " << res1 << " " << res2 << " func: " << func_type << std::endl;
	if ( res1 > pose.total_residue() || res2 > pose.total_residue() ) {
		tr.Warning 	<< "ignored constraint (no such atom in pose!)"
			<< name1 << " " << name2 << " " << res1 << " " << res2 << std::endl;
		data.setstate( std::ios_base::failbit );
		return;
	}

	atom_ = id::AtomID( id::NamedAtomID( name1, res1 ), pose );
	fixed_atom_ = id::AtomID( id::NamedAtomID( name2, res2 ), pose );
	if ( atom_.atomno() == 0 || fixed_atom_.atomno() == 0 ) {
		tr.Warning << "Error reading atoms: read in atom names("
			<< name1 << "," << name2 << "), "
			<< "and found AtomIDs (" << atom_ << "," << fixed_atom_ << ")" << std::endl;
			data.setstate( std::ios_base::failbit );
			return;
	}

	func_ = func_factory.new_func( func_type );
	func_->read_data( data );

	if ( data.good() ) {
	//chu skip the rest of line since this is a single line defintion.
		while( data.good() && (data.get() != '\n') ) {}
		if ( !data.good() ) data.setstate( std::ios_base::eofbit );
	}

	if ( tr.Debug.visible() ) {
		func_->show_definition( std::cout );
		std::cout << std::endl;
	}
} // read_def


} // constraints
} // scoring
} // core
