// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

#ifndef INCLUDED_core_scoring_constraints_CoordinateConstraint_HH
#define INCLUDED_core_scoring_constraints_CoordinateConstraint_HH

#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/id/AtomID.hh>

#include <core/conformation/Conformation.hh>


// C++ Headers
#include <cstdlib>
#include <iostream>
//#include <map>
#include <utility>

namespace core {
namespace scoring {
namespace constraints {


///

class CoordinateConstraint : public Constraint {
public:


	CoordinateConstraint() :
		Constraint( coordinate_constraint ),
		atom_( id::BOGUS_ATOM_ID ),
		fixed_atom_( id::BOGUS_ATOM_ID ),
		func_( NULL ) {}

	///c-tor
	CoordinateConstraint(
		AtomID const & a1,
		AtomID const & fixed_atom_in,
		Vector const & xyz_target_in,
	 	FuncOP func,
		ScoreType scotype = coordinate_constraint
	):
		Constraint( scotype ),
		atom_(a1),
		fixed_atom_( fixed_atom_in ),
		xyz_target_( xyz_target_in ),
		func_( func )
	{}


	virtual std::string type() const {
		return "CoordinateConstraint";
	}

	virtual ConstraintOP clone() const {
		return new CoordinateConstraint( *this );
	}

	/// @brief Copies the data from this Constraint into a new object and returns an OP
	/// atoms are mapped to atoms with the same name in dest pose ( e.g. for switch from centroid to fullatom )
	/// if a sequence_mapping is present it is used to map residue numbers .. NULL = identity mapping
	/// to the new object. Intended to be implemented by derived classes.
	virtual ConstraintOP remapped_clone( pose::Pose const& src, pose::Pose const& dest, sequence::SequenceMappingCOP map=NULL ) const;


 	///
	void show( std::ostream& out ) const
	{
		out << "CoordinateConstraint ("
				<< atom_.atomno() << "," << atom_.rsd() << "-"
				<< fixed_atom_.atomno() << "," << fixed_atom_.rsd() << ")" << std::endl;
		func_->show( out );
	}

	// @brief Reads the definition of a Constraint from the given std::istream,
	// using the given Pose, and the given FuncFactory. This method is intended
	// to be overridden by derived classes if they'd like to use the
	// ConstraintIO machinery.
	virtual void read_def( std::istream &, pose::Pose const &, FuncFactory const & );

	///
	void show_def( std::ostream& out, pose::Pose const& pose ) const
	{
		out << type() << " " << id::NamedAtomID( atom_, pose ) << " " << id::NamedAtomID(fixed_atom_, pose ) << " ";
		out << xyz_target_.x() << " " << xyz_target_.y() << " " << xyz_target_.z() << " " ;
		func_->show_definition( out );
	}

	// @brief take coordinates, distances, angles, etc from given pose
	///
	virtual void steal_def( pose::Pose const& );

	///
	Real
	score(
		Vector const & xyz
	) const
	{
		return func( xyz.distance( xyz_target_ ) );
	}

	///
	void
	score( XYZ_Func const & xyz, EnergyMap const &, EnergyMap & emap ) const
	{
		emap[ this->score_type() ] += score( xyz( atom_ ) );
	}

	// atom deriv
	void
	fill_f1_f2(
		AtomID const & atom,
		conformation::Conformation const & conformation,
		Vector & F1,
 		Vector & F2,
		EnergyMap const & weights
	) const
	{
		if ( atom != atom_ ) return;

		Vector const & xyz1( conformation.xyz( atom_ ) ), xyz2( xyz_target_ );

		Vector const f2( xyz1 - xyz2 );
		Real const dist( f2.length() ), deriv( dfunc( dist ) );
		if ( deriv != 0.0 && dist != 0.0 ) {
			Vector const f1( xyz1.cross( xyz2 ) );
			// jk: double F1 and F2 because the target is fixed
			// (matches deriv_check, and minimizes faster)
			// rhiju: No, JK, this isn't working...
			F1 += ( ( deriv / dist ) * f1 ) * weights[ this->score_type() ];
			F2 += ( ( deriv / dist ) * f2 ) * weights[ this->score_type() ];
		}
	}



	///
	Size
	natoms() const
	{
		return 2;
	}

	virtual
	ConstraintOP
	remap_resid( core::sequence::SequenceMapping const &seqmap ) const;

	///
	AtomID const &
	atom( Size const n ) const
	{
		switch( n ) {
		case 1:
			return atom_;
		case 2:
			return fixed_atom_;
		default:
			utility_exit_with_message( "CoordinateConstraint::atom() bad argument" );
		}
		return atom_;
	}

	Real
	dist( pose::Pose const & pose ) const;


	virtual Size show_violations(
		std::ostream& out,
		pose::Pose const& pose,
		Size verbose_level,
		Real threshold = 1
	) const;

private:

	// functions
	Real
	func( Real const theta ) const
	{
		return func_->func( theta );
	}

	// deriv
	Real
	dfunc( Real const theta ) const
	{
		return func_->dfunc( theta );
	}

private:
	// data
	AtomID atom_;
	AtomID fixed_atom_;
	Vector xyz_target_;
	FuncOP func_;
};

}
}
}

#endif
