// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

#ifndef INCLUDED_core_scoring_constraints_DihedralConstraint_HH
#define INCLUDED_core_scoring_constraints_DihedralConstraint_HH


#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/ScoreType.hh>

#include <core/id/AtomID.hh>

//Utility Headers

// C++ Headers

namespace core {
namespace scoring {
namespace constraints {


/// constraint on dihedral angle formed by 4 points

class DihedralConstraint : public Constraint {
public:

	///
	Real
	score(
		Vector const & xyz1,
		Vector const & xyz2,
		Vector const & xyz3,
		Vector const & xyz4
	) const;

	virtual std::string type() const {
		return "Dihedral";
	}

	virtual ConstraintOP clone() const {
		return new DihedralConstraint( atom1_, atom2_, atom3_, atom4_, func_, score_type() );
	}

	void read_def( std::istream& in, pose::Pose const& pose, FuncFactory const& func_factory );

	///
	void
	score( XYZ_Func const & xyz, EnergyMap const &, EnergyMap & emap ) const
	{
		emap[ this->score_type() ] += score( xyz( atom1_ ), xyz( atom2_ ), xyz( atom3_ ), xyz( atom4_ ) );
	}

	///
	Real
	score( conformation::Conformation const & conformation 	) const;


	// atom deriv
	void
	fill_f1_f2(
		AtomID const & atom,
		conformation::Conformation const & conformation,
		Vector & F1,
		Vector & F2,
		EnergyMap const & weights
	) const;


	///c-tor
	DihedralConstraint(
		AtomID const & a1,
		AtomID const & a2,
		AtomID const & a3,
		AtomID const & a4,
		FuncOP func,
		ScoreType scotype = dihedral_constraint
	):
		Constraint( scotype ),
		atom1_(a1),
		atom2_(a2),
		atom3_(a3),
		atom4_(a4),
		func_( func )
	{}

	///
	Size
	natoms() const
	{
		return 4;
	}

	virtual
	ConstraintOP
	remap_resid( core::sequence::SequenceMapping const &seqmap ) const;

	///
	AtomID const &
	atom( Size const n ) const
	{
		switch( n ) {
		case 1:
			return atom1_;
		case 2:
			return atom2_;
		case 3:
			return atom3_;
		case 4:
			return atom4_;
		default:
			utility_exit_with_message( "DihedralConstraint::atom() bad argument" );
		}
		return atom1_;
	}

	virtual void show(std::ostream& out ) const;

private:

	// functions
	Real
	func( Real const theta ) const
	{
		return func_->func( theta );
	}

	// deriv
	Real
	dfunc( Real const theta ) const
	{
		return func_->dfunc( theta );
	}

	// this guy doesnt need member data --> static
	static
	void
	helper(
		Vector const & M,
		Vector const & v,
		Vector const & w,
		Vector & F1,
		Vector & F2
	);

	static
	void
	p1_cosine_deriv(
		Vector const & p1,
		Vector const & p2,
		Vector const & p3,
		Vector const & p4,
		Real & x,
		Vector & F1,
		Vector & F2
	);

	static
	void
	p2_cosine_deriv(
		Vector const & p1,
		Vector const & p2,
		Vector const & p3,
		Vector const & p4,
		Real & x,
		Vector & F1,
		Vector & F2
	);

private:
	// data
	AtomID atom1_, atom2_, atom3_, atom4_;
	FuncOP func_;
};

} // constraints
} // scoring
} // core

#endif
