// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/scoring/constraints/HarmonicFunc.hh
/// @brief Definition for functions used in definition of constraints.
/// @author James Thompson

#ifndef INCLUDED_core_scoring_constraints_HarmonicFunc_HH
#define INCLUDED_core_scoring_constraints_HarmonicFunc_HH

#include <core/scoring/constraints/HarmonicFunc.fwd.hh>

#include <core/scoring/constraints/Func.hh>

#include <core/types.hh>

#include <utility/pointer/ReferenceCount.hh>

#include <numeric/angle.functions.hh>
#include <numeric/random.functions.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <core/util/Tracer.hh>

// C++ Headers

namespace core {
namespace scoring {
namespace constraints {

class HarmonicFunc : public Func {
public:
	HarmonicFunc( Real const x0_in, Real const sd_in ): x0_( x0_in ), sd_( sd_in ){}

	FuncOP
	clone() const { return new HarmonicFunc( *this ); }

	Real func( Real const x ) const;
	Real dfunc( Real const x ) const;

	void read_data( std::istream& in );

	void show_definition( std::ostream &out ) const;

	Real x0() const { return x0_; }
	Real sd() const { return sd_; }
	void x0( Real x ) { x0_ = x; }
	void sd( Real sd ){ sd_ = sd;}

	Size
	show_violations( std::ostream& out, Real x, Size verbose_level, core::Real threshold = 1 ) const;

private:
	Real x0_;
	Real sd_;
};


/// @brief Function that operates in radians, for applications like DihedralConstraint.
/// Prevents discontinuities at 0/360 or -180/180 degrees for dihedral constraints.
class CircularHarmonicFunc : public Func {
public:
	CircularHarmonicFunc(
		Real const x0_radians, Real const sd_radians
	): x0_( x0_radians ), sd_( sd_radians ) {}

	FuncOP clone() const { return new CircularHarmonicFunc( *this ); }

	Real func( Real const x ) const;
	Real dfunc( Real const x ) const;

	virtual void read_data( std::istream& in );
	virtual void show_definition( std::ostream &out ) const;

private:
	Real x0_;
	Real sd_;
};

/// @brief Generalization of CircularHarmonicFunc -- other exponents allowed.
/// @brief Operates in radians, like DihedralConstraint.
/// @brief Prevents discontinuities at 0/360 or -180/180 degrees for dihedral constraints.

class CircularPowerFunc : public Func {
public:
	CircularPowerFunc( Real const x0_radians, Real const sd_radians, int const power, Real const weight ):
		x0_( x0_radians ),
		sd_( sd_radians ),
		power_( power ),
		weight_( weight )
	{
		assert( std::abs( std::pow(3.0,2) - 9.0 ) < 1e-3 );
		assert( power_ != 1 && power_ != 0 );
	}


	FuncOP
	clone() const
	{
		return new CircularPowerFunc( *this );
	}

	Real
	func( Real const x ) const
	{
		Real const z = ( numeric::nearest_angle_radians(x,x0_)-x0_ )/sd_;
		return weight_ * std::pow( z, power_ );
	}

	Real
	dfunc( Real const x ) const
	{
		Real const z = ( numeric::nearest_angle_radians(x,x0_)-x0_ )/sd_;
		return weight_ * power_ * std::pow( z, power_ - 1 ) / sd_;
	}

private:
	Real const x0_;
	Real const sd_;
	int const power_;
	Real const weight_;

};


} // constraints
} // scoring
} // core

#endif
