// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief

// Package Headers
#include <core/scoring/constraints/FuncFactory.hh>
#include <core/scoring/constraints/LocalCoordinateConstraint.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/Func.hh>
#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/pose/Pose.hh>
#include <core/id/AtomID.hh>

#include <core/conformation/Conformation.hh>

// Utility Headers
#include <core/util/Tracer.hh>


// C++ Headers
#include <cstdlib>
#include <iostream>
#include <utility>


static core::util::Tracer tr("core.LocalCoordinateConstraint");

namespace core {
namespace scoring {
namespace constraints {

Size
LocalCoordinateConstraint::show_violations(
	std::ostream & out,
	pose::Pose const & pose,
	Size verbose_level,
	Real threshold
) const{
  if ( verbose_level > 80 ) {
    out << "CoordConstraint ("
		<< pose.residue_type(atom_.rsd() ).atom_name( atom_.atomno() )
		<< ":" << atom_.atomno() << "," << atom_.rsd() << " ) ";
  }
  return func_->show_violations( out, dist( pose ), verbose_level, threshold );
}


/// @brief Copies the data from this Constraint into a new object and returns an OP
/// atoms are mapped to atoms with the same name in dest pose ( e.g. for switch from centroid to fullatom )
/// if a sequence_mapping is present it is used to map residue numbers .. NULL = identity mapping
/// to the new object. Intended to be implemented by derived classes.
ConstraintOP LocalCoordinateConstraint::remapped_clone( pose::Pose const& src, pose::Pose const& dest, sequence::SequenceMappingCOP smap ) const {
	id::NamedAtomID atom1( atom(1), src );
  id::NamedAtomID atom2( atom(2), src );
	id::NamedAtomID atom3( atom(3), src );
	id::NamedAtomID atom4( atom(4), src );

	if ( smap ) {
		atom1.rsd() = (*smap)[ atom(1).rsd() ];
		atom2.rsd() = (*smap)[ atom(2).rsd() ];
		atom3.rsd() = (*smap)[ atom(3).rsd() ];
		atom4.rsd() = (*smap)[ atom(4).rsd() ];
	}

	//get AtomIDs for target pose
	id::AtomID id1( atom1, dest );
	id::AtomID id2( atom2, dest );
	id::AtomID id3( atom3, dest );
	id::AtomID id4( atom4, dest );
	if ( id1.valid() && id2.valid() && id3.valid() && id4.valid() ) {
		return new LocalCoordinateConstraint( id1, id::StubID( id2, id3, id4) , xyz_target_, func_, score_type() );
	} else {
		return NULL;
	}
}

Real
LocalCoordinateConstraint::dist( pose::Pose const & pose ) const {
  conformation::Conformation const & conformation( pose.conformation() );
  Vector const & xyz( conformation.xyz( atom_ ) );
	kinematics::Stub my_stub( conformation.stub_from_id( fixed_stub_ ) );
	return func( xyz_target_.distance( my_stub.global2local( xyz ) ) );
  return xyz.distance( xyz_target_ );
}

void
LocalCoordinateConstraint::steal_def( pose::Pose const& pose ) {
	conformation::Conformation const & conformation( pose.conformation() );
	kinematics::Stub my_stub( conformation.stub_from_id( fixed_stub_ ) );
  xyz_target_ = my_stub.global2local( conformation.xyz( atom_ ) );
	//tr.Trace << "get local xyz for " << atom_ << " " << xyz_target_.x() << " " << xyz_target_.y() << " " << xyz_target_.z() << " vs global: " //	Vector bla( conformation.xyz( atom_ ) );
	//tr.Trace << bla.x() << " " << bla.y() << " " << bla.z() << " " << std::endl;
}

Vector LocalCoordinateConstraint::xyz_target( core::pose::Pose const& local_frame_pose ) const {
	tr.Trace << "local target: " <<  xyz_target_.x() << " " << xyz_target_.y() << " " << xyz_target_.z() << std::endl;
	conformation::Conformation const & conformation( local_frame_pose.conformation() );
	kinematics::Stub my_stub( conformation.stub_from_id( fixed_stub_ ) );
	tr.Trace << "stub: " << my_stub << std::endl;
	return my_stub.local2global( xyz_target_ );
}

void LocalCoordinateConstraint::set_xyz_target( Vector const& xyz_in, core::pose::Pose const& local_frame_pose ) {
	conformation::Conformation const & conformation( local_frame_pose.conformation() );
	kinematics::Stub my_stub( conformation.stub_from_id( fixed_stub_ ) );
	xyz_target_=my_stub.global2local( xyz_in );
}

///@details one line definition "LocalCoordinates atom1 res1 atom2 res2 function_type function_definition"
void
LocalCoordinateConstraint::read_def(
	std::istream& data,
	core::pose::Pose const& pose,
	FuncFactory const& func_factory
) {
	Size res1, res2;
	std::string name1, name2, name3, name4;
	std::string func_type;
	std::string type;

	data
		>> name1 >> res1
		>> name2 >> name3 >> name4 >> res2
		>> xyz_target_.x()
		>> xyz_target_.y()
		>> xyz_target_.z()
		>> func_type;

	tr.Debug << "read: " << name1 << " " << res1 << " " << name2 << " " << name3
					 <<	" " << name4 << " " << res2 << " func: " << func_type << std::endl;
	if ( res1 > pose.total_residue() || res2 > pose.total_residue() ) {
		tr.Warning 	<< "ignored constraint (no such atom in pose!)"
			<< name1 << " " << name2 << " " << res1 << " " << res2 << std::endl;
		data.setstate( std::ios_base::failbit );
		return;
	}

	atom_ = id::AtomID( id::NamedAtomID( name1, res1 ), pose );
	fixed_stub_ = id::StubID( id::NamedStubID( name2, name3, name4, res2 ), pose );
	if ( !atom_.valid() || !fixed_stub_.valid() ) {
		tr.Warning << "Error reading atoms: read in atom names("
							 << name1 << "," << name2 << "), "
							 << "and found AtomIDs (" << atom_ << "," << fixed_stub_ << ")" << std::endl;
		data.setstate( std::ios_base::failbit );
		return;
	}

	func_ = func_factory.new_func( func_type );
	func_->read_data( data );

	if ( data.good() ) {
	//chu skip the rest of line since this is a single line defintion.
		while( data.good() && (data.get() != '\n') ) {}
		if ( !data.good() ) data.setstate( std::ios_base::eofbit );
	}

	if ( tr.Debug.visible() ) {
		func_->show_definition( std::cout );
		std::cout << std::endl;
	}
} // read_def

void
LocalCoordinateConstraint::show_def(
	std::ostream& out,
	core::pose::Pose const& pose ) const
{
	out << type() << " " << id::NamedAtomID( atom_, pose ) << " ";
	id::NamedStubID stubid(fixed_stub_, pose );
	out << stubid.atom( 1 ).atom() << " " <<
		stubid.atom( 2 ).atom() << " " <<
		stubid.atom( 3 ).atom() << " " << stubid.atom(1).rsd() << " ";

	out << xyz_target_.x() << " " << xyz_target_.y() << " " << xyz_target_.z() << " " ;
	func_->show_definition( out );
}

} // constraints
} // scoring
} // core
