// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file core/scoring/constraints/MultiConstraint.hh
/// @brief contains declarations for a type of constraint that holds multiple
/// other constrains that belong to each other and are all evaluate at once
/// @author Florian Richter (floric@u.washington.edu, march 2008)


#ifndef INCLUDED_core_scoring_constraints_MultiConstraint_HH
#define INCLUDED_core_scoring_constraints_MultiConstraint_HH

#include <core/scoring/constraints/MultiConstraint.fwd.hh>

// Unit header
#include <core/scoring/constraints/Constraint.hh>

#include <core/scoring/constraints/XYZ_Func.hh>
#include <core/scoring/ScoreType.hh>

#include <core/id/AtomID.hh>
#include <core/scoring/EnergyMap.fwd.hh>
#include <core/conformation/Conformation.fwd.hh>
#include <core/pose/Pose.fwd.hh>

//Utility Headers
#include <numeric/xyzVector.fwd.hh>

// STL Headers
#include <map>

namespace core {
namespace scoring {
namespace constraints {


class MultiConstraint : public Constraint {
public:

	/// @brief default Constructor
	MultiConstraint():
		Constraint( dof_constraint )
	{}

	/// @brief Constructor
	MultiConstraint( const ConstraintCOPs & cst_in );

	///
	virtual
	ConstraintOP clone() const {
		if ( member_constraints_.size() > 0 ) {
			return new MultiConstraint( member_constraints_ );
		} else {
			return new MultiConstraint();
		}
	}

	/// @brief number of atoms involved in this MultiConstraint container
  virtual Size natoms() const
	{
		return member_atoms_.size();
	}
	/// @brief number of constraints data
	virtual Size size() const { return member_constraints_.size(); }

	virtual std::string type() const {
		return "MultiConstraint";
	}

	/// @brief read in constraint defiinition
	virtual
	void read_def( std::istream& data, pose::Pose const& pose, FuncFactory const& func_factory );

	/// @brief compute score
	virtual
	void
	score( XYZ_Func const & xyz_func, EnergyMap const & weights, EnergyMap & emap ) const;

	virtual
	AtomID const & atom( Size const n ) const{
		assert( n <= member_atoms_.size() );
		return member_atoms_[n];
	}


	///@brief add individual constraint into MultiConstraint
	virtual
	void
	add_individual_constraint( ConstraintCOP cst_in );

	virtual
	ConstraintOP
	remap_resid( core::sequence::SequenceMapping const &seqmap ) const;

	/// @brief compute atom deriv
	virtual
	void
	fill_f1_f2(
		AtomID const & atom,
		conformation::Conformation const & conformation,
		Vector & F1,
 		Vector & F2,
		EnergyMap const & weights
	) const;

	virtual
	void show( std::ostream& out ) const;

	virtual
	void show_def( std::ostream& out, pose::Pose const& pose ) const;

	virtual
	Size show_violations( std::ostream & out, pose::Pose const & pose, Size verbose_level, Real threshold = 1.0 ) const;

	ConstraintCOPs const &
	member_constraints() const {
		return member_constraints_;
	}


protected:

	//vector that holds the constraints
	ConstraintCOPs member_constraints_;

private:

	//data structure that holds the atoms and atom numbers
	utility::vector1< AtomID > member_atoms_;
	std::map< AtomID, ConstraintCOPs > AtomID_to_Csts_;

}; //MultiConstraint

} //constraints
} //scoring
} //core

#endif
