// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/scoring/constraints/SOGFunc.hh
/// @brief Definition for functions used in definition of constraints.
/// @author James Thompson


#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/scoring/constraints/SOGFunc.hh>

#include <core/scoring/constraints/util.hh>

#include <core/types.hh>

#include <utility/pointer/ReferenceCount.hh>

#include <numeric/util.hh>
#include <numeric/angle.functions.hh>
#include <numeric/random.functions.hh>
#include <ObjexxFCL/formatted.o.hh>
#include <core/util/Tracer.hh>

#include <core/options/option.hh>
#include <core/options/keys/james.OptionKeys.gen.hh>

// C++ Headers

#include <iostream>

namespace core {
namespace scoring {
namespace constraints {

void
SOGFunc::read_data( std::istream & in ) {
	Size n_funcs;

	clear_(); // don't forget to remove old data!

	//bool skip(false);

	using namespace core::options;
	using namespace core::options::OptionKeys;

	Real lowest_sdev(100);
	in >> n_funcs;
	for ( Size i = 1; i <= n_funcs; ++i ) {
		Real mean(0.0), sdev(0.0), weight(0.0);
		in >> mean >> sdev >> weight;
		means_.  push_back( mean );
		sdevs_.  push_back( sdev );
		weights_.push_back( weight );

		//std::cout << "read sdev of " << mean;
		if ( sdev < lowest_sdev ) lowest_sdev = sdev;
	}

	if ( lowest_sdev > option[ james::sog_cutoff ]() ) {
		means_  .clear();
		sdevs_  .clear();
		weights_.clear();
		//std::cout << "cleared after sdev of " << lowest_sdev;
	} else {
		//std::cout << " not skipped after standard deviation " << lowest_sdev;
	}
	//std::cout << std::endl;
}

void
SOGFunc::clear_() {
	means_.  clear();
	sdevs_.  clear();
	weights_.clear();
}

core::Real
SOGFunc::get_alt_score_( Real const x ) const {
	Real const alt_mean( 18.991 );
	Real const alt_sdev(  7.353 );
	return -1 * std::log( dgaussian( x, alt_mean, alt_sdev, 1.0 ) );
}

Real
SOGFunc::func( Real const x ) const	{
	//if ( use_log_score_ ) return - logdgaussian( x, mean_, sd_, 1 );
	//else                  return dgaussian( x, mean_, sd_, 1 );
	Real score( 0.0 );
	//Real const alt_sc( get_alt_score_(x) );
	for ( utility::vector1< Real >::const_iterator
				w = weights_.begin(), w_end = weights_.end(),
				m = means_.begin(), m_end = means_.end(),
				s = sdevs_.begin(), s_end = sdevs_.end(); // iterators
				w != w_end && m != m_end && s != s_end; // condition
				++w, ++m, ++s  // iteration
	) {
		Real temp_sc = dgaussian( x, *m, *s, *w );
		//score += std::min( alt_sc, temp_sc );
		score += temp_sc;
	}

	check_bounds( x, score );
	if ( score == 0 ) {
		return 0.0;
	} else {
		Real const sc( -1 * std::log( score ) );
		return sc;
	}
} // func

Real
SOGFunc::dfunc( Real const x ) const {
	// wrong! f / f'
	//Real deriv( 0.0 );
	//for ( utility::vector1< Real >::const_iterator
	//			w = weights_.begin(), w_end = weights_.end(),
	//			m = means_.begin(), m_end = means_.end(),
	//			s = sdevs_.begin(), s_end = sdevs_.end(); // iterator def
	//			w != w_end && m != m_end && s != s_end; // condition
	//			++w, ++m, ++s // iteration
	//) {
	//	deriv += gaussian_deriv( x, *m, *s, *w );
	//}
	Real const df( estimate_dfunc( x ) );
	check_bounds( x, df );
	return df;
} // dfunc

void SOGFunc::check_bounds( Real const x, Real const val ) const {
	if ( numeric::isinf( val ) || numeric::isnan( val ) ) {
		std::cerr << "bounds error (radius =" << x << "), def = ";
		show_definition( std::cerr );
	}
}


void SOGFunc::show_definition( std::ostream & out ) const {
	assert( weights_.size() == means_.size() );
	assert( weights_.size() == sdevs_.size() );

	out << "SOGFUNC " << weights_.size();
	for ( Size i = 1; i <= weights_.size(); ++i ) {
		out << " " << means_[i] << " " << sdevs_[i] << " " << weights_[i];
	}
	out << '\n';
} // show_definition

} // namespace constraints
} // namespace scoring
} // namespace core
