// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file src/core/scoring/constraints/util.cc
/// @brief utility functions for defining constraints. Maybe better placed in src/numeric?
/// @author James Thompson

#include <core/types.hh>

#include <utility/pointer/ReferenceCount.hh>

#include <numeric/angle.functions.hh>
#include <numeric/random.functions.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/constraints/util.hh>
#include <core/scoring/constraints/Constraint.hh>
#include <core/scoring/constraints/ConstraintIO.hh>
#include <core/options/option.hh>
#include <core/scoring/constraints/ConstraintSet.hh>

#include <core/util/Tracer.hh>

#include <iostream>

// option key includes

#include <core/options/keys/constraints.OptionKeys.gen.hh>



// C++ Headers
static core::util::Tracer tr("core.scoring.constraints");
static numeric::random::RandomGenerator RG(420);  // <- Magic number, do not change it (and dont try and use it anywhere else) !!! what a retarded system ... %-|

namespace core {
namespace scoring {
namespace constraints {

/// @brief Returns the weighted value of a normal distribution evaluated
///  with the given mean, sd, and x values. Returns zero if the weight is
///  less than 1e-10.
Real logdgaussian( Real x, Real mean, Real sd, Real weight )
{
	using namespace std;
	//		Real r = abs( x - mean );//expensive call to abs is not needed since you square later
	Real r = x - mean;
	static Real sqrt_2pi = 2.50662721600161;

	Real answer = log(weight / (sd * sqrt_2pi)) - ( r * r / (2*sd*sd) );
	return answer;
}

/// @brief Returns the weighted value of a normal distribution evaluated
///  with the given mean, sd, and x values. Returns zero if the weight is
///  less than 1e-10.
Real logdgaussian_deriv( Real x, Real mean, Real sd, Real )
{
	using namespace std;
	Real r = x - mean;
	Real answer =  - (r / (sd*sd) );
	return answer;
}

/// @brief Returns the weighted value of a normal distribution evaluated
///  with the given mean, sd, and x values. Returns zero if the weight is
///  less than 1e-10.
Real dgaussian( Real x, Real mean, Real sd, Real weight ) {
	if ( weight < 1e-10 ) {
		return 0;
	}

	using namespace std;
	Real r = x - mean;
	static Real sqrt_2pi = 2.50662721600161;
	Real answer = weight * (1 / (sd * sqrt_2pi)) * exp( -1 * r * r / (2*sd*sd) );
	return answer;
}

/// @brief Returns the weighted derivative of a normal distribution evaluated
/// with the given mean, sd, and x values. Returns zero if the weight is less
/// than 1e-10.
Real gaussian_deriv( Real x, Real mean, Real sd, Real weight ) {
	assert( weight >= 0.0 && weight <= 1.0 );

	if ( weight < 1e-10 ) {
		return 0;
	}

	using namespace std;
	Real r = abs( x - mean );
	Real answer = dgaussian( x, mean, sd , weight ) * ( 1 / ( sd * sd ) ) * r;
	return answer;
}

/// @brief Returns the weighted value of an exponential distribution
/// evaluated with the given anchor, rate, and x values. Returns zero if the
/// weight is less than 1e-10.
Real dexponential( Real x, Real anchor, Real rate, Real weight ) {
	assert( weight >= 0.0 && weight <= 1.0 );
	if ( weight < 1e-10 ) {
		return 0;
	}

	using namespace std;
	Real r = abs( x - anchor );
	Real answer = weight * rate * exp( -1 * rate * r );
	return answer;
}

/// @brief Returns the weighted derivative of a log-exponential distribution
/// evaluated with the given anchor, rate, and x values. Returns zero if the
/// weight is less than 1e-10.
Real exponential_deriv( Real x, Real anchor, Real rate, Real weight ) {
	if ( weight < 1e-10 ) {
		return 0;
	}

	using namespace std;
	Real r = abs( x - anchor );
	return weight  * rate * rate * exp( -1 * rate * r );
}

Real linear_interpolate(
	Real const x_val,
	Real const x1,
	Real const x2,
	Real const y1,
	Real const y2
) {

	if ( x_val == x1 ) return y1;
	if ( x_val == x2 ) return y2;

	// calculate slope
	Real slope = ( y2 - y1 ) / ( x2 - x1 );
	// walk along line
	return (x_val - x1) * slope;
}

void
cull_violators(
	ConstraintCOPs const& target_list,
	ConstraintCOPs &culled_list,
	pose::Pose const& filter_pose,
	core::Real threshold
) {
	culled_list.clear();
	for ( ConstraintCOPs::const_iterator it = target_list.begin(),
					eit = target_list.end(); it != eit; ++it ) {
		if ( (*it)->show_violations( tr.Debug, filter_pose, 1, threshold ) == 0 ) {
			culled_list.push_back( *it );
		}
	}
}


////////// Centroid constraints
// this entire system is really dumb. We shouldn't hard-code

std::string get_cst_file_option(){
	using namespace core::options;
	utility::vector1< std::string>  cst_files = option[ OptionKeys::constraints::cst_file ]();
	core::Size choice=core::Size( RG.random_range(1,(cst_files.size())  ));
	tr.Info << "Constraint choice: " << cst_files[choice] << std::endl;
	return cst_files[choice];
}

//// @brief	add constraints if specified by user.
void add_constraints_from_cmdline_to_pose( core::pose::Pose & pose ) {
	using namespace core::options;
	using namespace core::scoring::constraints;
	if ( option[ OptionKeys::constraints::cst_file ].user() ) {
		ConstraintSetOP cstset_ = ConstraintIO::get_instance()->read_constraints( get_cst_file_option() ,new ConstraintSet, pose	);
		pose.constraint_set( cstset_ );
	}
}

//// @brief	add constraints if specified by user.
void add_constraints_from_cmdline_to_scorefxn( core::scoring::ScoreFunction &scorefxn_  ) {
	using namespace core::options;
	if ( option[ OptionKeys::constraints::cst_weight ].user() ) {
		scorefxn_.set_weight(	atom_pair_constraint,  option[ OptionKeys::constraints::cst_weight ]() );
		scorefxn_.set_weight(	angle_constraint,      option[ OptionKeys::constraints::cst_weight ]() );
		scorefxn_.set_weight(	dihedral_constraint,   option[ OptionKeys::constraints::cst_weight ]() );
		scorefxn_.set_weight( coordinate_constraint, option[ OptionKeys::constraints::cst_weight ]() );
	}
}

//// @brief 	add constraints if specified by user.
void add_constraints_from_cmdline( core::pose::Pose & pose, core::scoring::ScoreFunction &scorefxn_  ) {
	add_constraints_from_cmdline_to_pose( pose );
	add_constraints_from_cmdline_to_scorefxn( scorefxn_ );
}





////////// FA constraints


std::string get_cst_fa_file_option() {
	using namespace core::options;
	utility::vector1< std::string>  cst_files = option[ OptionKeys::constraints::cst_fa_file ]();
	core::Size choice=core::Size( RG.random_range(1,(cst_files.size())  ));
	tr.Info << "Constraint choice: " << cst_files[choice] << std::endl;
	return cst_files[choice];
}

/// @brief add constraints if specified by user using the
/// -constraints::cst_fa_file flag. Setting appropriate weights for
/// ScoreFunction is done elsewhere.
void add_fa_constraints_from_cmdline_to_pose( core::pose::Pose & pose ) {
	using namespace core::options;
	using namespace core::scoring::constraints;
	if ( option[ OptionKeys::constraints::cst_fa_file ].user() ) {
		ConstraintSetOP cstset_ = ConstraintIO::get_instance()->read_constraints(get_cst_fa_file_option(),	new ConstraintSet, pose	);
		pose.constraint_set( cstset_ );
	}
}

/// @brief	add constraints if specified by user.
void add_fa_constraints_from_cmdline_to_scorefxn( core::scoring::ScoreFunction &scorefxn_  ) {
	using namespace core::options;
	if ( option[ OptionKeys::constraints::cst_fa_weight ].user() ) {
		scorefxn_.set_weight(	atom_pair_constraint,  option[ OptionKeys::constraints::cst_fa_weight ]()  );
		scorefxn_.set_weight(	angle_constraint,	     option[ OptionKeys::constraints::cst_fa_weight ]()	);
		scorefxn_.set_weight(	dihedral_constraint,   option[ OptionKeys::constraints::cst_fa_weight ]()	);
		scorefxn_.set_weight( coordinate_constraint, option[ OptionKeys::constraints::cst_fa_weight ]()  );
	}
}


//// @brief 	add constraints if specified by user.
void add_fa_constraints_from_cmdline(
	core::pose::Pose & pose,
	core::scoring::ScoreFunction & scorefxn_
) {
	add_fa_constraints_from_cmdline_to_pose( pose );
	add_fa_constraints_from_cmdline_to_scorefxn( scorefxn_ );
}

} // namespace constraints
} // namespace scoring
} // namespace core
