// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/disulfides/FullatomDisulfideEnergy.cc
/// @brief  Disulfide Energy class implementation
/// @author Andrew Leaver-Fay

// Unit headers
#include <core/scoring/disulfides/FullatomDisulfideEnergy.hh>
#include <core/scoring/disulfides/FullatomDisulfideEnergyCreator.hh>

// Package headers
#include <core/scoring/disulfides/FullatomDisulfidePotential.hh>
#include <core/scoring/disulfides/FullatomDisulfideEnergyContainer.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/conformation/Conformation.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/methods/Methods.hh>

namespace core {
namespace scoring {
namespace disulfides {


/// @details This must return a fresh instance of the FullatomDisulfideEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
FullatomDisulfideEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new FullatomDisulfideEnergy( ScoringManager::get_instance()->get_FullatomDisulfidePotential() );
}

ScoreTypes
FullatomDisulfideEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( dslf_ss_dst );
	sts.push_back( dslf_cs_ang );
	sts.push_back( dslf_ss_dih );
	sts.push_back( dslf_ca_dih );
	sts.push_back( dslf_cbs_ds );
	return sts;
}



FullatomDisulfideEnergy::FullatomDisulfideEnergy( FullatomDisulfidePotential const & potential )
:
	parent( new FullatomDisulfideEnergyCreator ),
	potential_( potential )
{}

FullatomDisulfideEnergy::~FullatomDisulfideEnergy()
{}

// EnergyMethod Methods:

methods:: EnergyMethodOP
FullatomDisulfideEnergy::clone() const
{
	return new FullatomDisulfideEnergy( potential_ );
}


void
FullatomDisulfideEnergy::setup_for_scoring(
	pose::Pose & pose,
	ScoreFunction const & ) const
{

	using namespace methods;

	if ( pose.energies().long_range_container( fa_disulfide_energy ) == 0 ) {
		FullatomDisulfideEnergyContainerOP dec = new FullatomDisulfideEnergyContainer( pose );
		pose.energies().set_long_range_container( fa_disulfide_energy, dec );
	} else {
		FullatomDisulfideEnergyContainerOP dec = FullatomDisulfideEnergyContainerOP (
			static_cast< FullatomDisulfideEnergyContainer * > (
			pose.energies().nonconst_long_range_container( fa_disulfide_energy ).get() ));
		dec->update( pose );
		if ( dec->num_residues() != pose.conformation().size() ) {
			FullatomDisulfideEnergyContainerOP dec = new FullatomDisulfideEnergyContainer( pose );
			pose.energies().set_long_range_container( fa_disulfide_energy, dec );
		}
	}

}


void
FullatomDisulfideEnergy::eval_atom_derivative(
	id::AtomID const & atomid,
	pose::Pose const & pose,
	kinematics::DomainMap const &,
	ScoreFunction const &,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{

	FullatomDisulfideEnergyContainerCOP dec = FullatomDisulfideEnergyContainerCOP (
		static_cast< FullatomDisulfideEnergyContainer const * > (
		pose.energies().long_range_container( methods::fa_disulfide_energy ).get() ));
	if ( ! dec->residue_forms_disulfide( atomid.rsd() ) ) return;

	if ( dec->disulfide_atom_indices( atomid.rsd() ).atom_gets_derivatives( atomid.atomno() ) ) {
		conformation::Residue const & res( pose.residue( atomid.rsd() ));
	//if ( res.atom_name( atomid.atomno() ) == " CA " ||
	//		res.atom_name( atomid.atomno() ) == " CB "  ||
	//		res.atom_name( atomid.atomno() ) == " SG " ) {
		Vector f1( 0.0 ), f2( 0.0 );
		potential_.get_disulfide_derivatives(
			pose,
			res,
			pose.residue( dec->other_neighbor_id( atomid.rsd()) ),
			dec->disulfide_atom_indices( atomid.rsd() ),
			dec->other_neighbor_atom_indices( atomid.rsd() ),
			atomid.atomno(),
			weights,
			f1, f2 );
		F1 += f1;
		F2 += f2;
	}

}



Real
FullatomDisulfideEnergy::eval_dof_derivative(
	id::DOF_ID const &,
	id::TorsionID const &,
	pose::Pose const &,
	ScoreFunction const &,
	EnergyMap const &
) const
{
	return 0.0;
}

void
FullatomDisulfideEnergy::indicate_required_context_graphs( utility::vector1< bool > & ) const
{}


// TwoBodyEnergy Methods:

void
FullatomDisulfideEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const &,
	EnergyMap & emap
) const
{
	Energy distance_score_this_disulfide;
	Energy csangles_score_this_disulfide;
	Energy dihedral_score_this_disulfide;
	Energy ca_dihedral_sc_this_disulf;
	bool truefalse_fa_disulf;

	if ( rsd1.aa() != chemical::aa_cys || rsd2.aa() != chemical::aa_cys ) return;

	FullatomDisulfideEnergyContainerCOP dec = FullatomDisulfideEnergyContainerCOP (
		static_cast< FullatomDisulfideEnergyContainer const * > (
		pose.energies().long_range_container( methods::fa_disulfide_energy ).get() ));

	if ( ! dec->residue_forms_disulfide( rsd1.seqpos() ) ||
			dec->other_neighbor_id( rsd1.seqpos() ) != (Size) rsd2.seqpos() ){
		return;
	}

	potential_.score_this_disulfide(
		rsd1, rsd2,
		dec->disulfide_atom_indices( rsd1.seqpos() ),
		dec->other_neighbor_atom_indices( rsd2.seqpos() ),
		distance_score_this_disulfide,
		csangles_score_this_disulfide,
		dihedral_score_this_disulfide,
		ca_dihedral_sc_this_disulf,
		truefalse_fa_disulf
	);

	/*
	Energy cbs_sc_this_disulf( 0.0 );

	if ( ! sfxn.has_zero_weight( dslf_cbs_ds ) ) {
		potential_.get_cbs_sc_this_disulf(
			rsd1, rsd2, cbs_sc_this_disulf
		);
	}
	*/

	//if ( truefalse_fa_disulf ) { // this just allows the packer to unwittingly break a disulfide bond, what is its point?
	emap[ dslf_ss_dst ] = distance_score_this_disulfide;
	emap[ dslf_cs_ang ] = csangles_score_this_disulfide;
	emap[ dslf_ss_dih ] = dihedral_score_this_disulfide;
	emap[ dslf_ca_dih ] = ca_dihedral_sc_this_disulf;

	//emap[ dslf_cbs_ds ] = cbs_sc_this_disulf;
	//}
}


bool
FullatomDisulfideEnergy::defines_intrares_energy( EnergyMap const & ) const
{
	return false;
}


void
FullatomDisulfideEnergy::eval_intrares_energy(
	conformation::Residue const &,
	pose::Pose const &,
	ScoreFunction const &,
	EnergyMap &
) const
{
}

// LongRangeTwoBodyEnergy methods
methods::LongRangeEnergyType
FullatomDisulfideEnergy::long_range_type() const
{
	return methods::fa_disulfide_energy;
}


bool
FullatomDisulfideEnergy::defines_residue_pair_energy(
	pose::Pose const & pose,
	Size res1,
	Size res2
) const
{
	using namespace methods;
	if ( ! pose.energies().long_range_container( fa_disulfide_energy )) return false;

	FullatomDisulfideEnergyContainerCOP dec = FullatomDisulfideEnergyContainerCOP (
		static_cast< FullatomDisulfideEnergyContainer const * > (
		pose.energies().long_range_container( fa_disulfide_energy ).get() ));
	return dec->disulfide_bonded( res1, res2 );
}


} // namespace disulfides
} // namespace scoring
} // namespace core

