// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/disulfides/FullatomDisulfidePotential.cc
/// @brief  Fullatom Disulfide potential class definition
/// @author Bill Schief
/// @author blindly ported by Andrew Leaver-Fay

// Unit Headers
#include <core/scoring/disulfides/FullatomDisulfidePotential.hh>

// Package Headers
#include <core/scoring/disulfides/DisulfideAtomIndices.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/conformation/Residue.hh>
#include <core/scoring/constraints/AtomPairConstraint.hh>
#include <core/scoring/constraints/AngleConstraint.hh>
#include <core/scoring/constraints/DihedralConstraint.hh>
#include <core/io/database/open.hh>

// Project Headers
#include <core/pose/Pose.hh>

// Utility Headers
#include <numeric/xyz.functions.hh>
#include <utility/exit.hh>

// Numeric headers
#include <numeric/constants.hh>
#include <numeric/interpolation/Histogram.hh>

#include <core/util/Tracer.hh>
using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR("core.scoring.disulfides.FullatomDisulfidePotential");

namespace core {
namespace scoring {
namespace disulfides {

using namespace core;

//-----------------------------------------
// physical parameters of disulfide bonds
//-----------------------------------------

//
//	core::Real const ideal_ss_dist_in_disulfide = { 2.02 };
//	 // mean sulfur-sulfur distance in natives // angstroms
//
//	core::Real const disulf_ss_dist_stdev = { 0.35 };
//	 // standard dev. of s-s dist in natives // degrees 0.35
//
//	core::Real const ideal_cs_angle_in_disulfide = { 103.4 };
//	 // mean cbeta-sulfur-sulfur angle in natives // degrees
//
//	core::Real const disulf_cs_angle_stdev = { 5.0 };
//	 // standard dev. of cbeta-s-s angle in natives // degrees  2.6
//
//	core::Real const mean_pos_dihedral_in_disulf = { 87.9 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	core::Real const disulf_pos_dihedral_stdev = { 21.8 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	core::Real const mean_pos_dihedral_gauss1 = { 87.4 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	core::Real const stdev_pos_dihedral_gauss1 = { 20.9 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	core::Real const mean_pos_dihedral_gauss2 = { 95.6 };
//	 // mean positive cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	core::Real const stdev_pos_dihedral_gauss2 = { 3.0 };
//	 // standard dev. of pos. dihedral angle in natives // degrees
//
//	core::Real const mean_neg_dihedral_in_disulf = { -86.2 };
//	 // mean negative cbeta-sulfur-sulfur-cbeta dihedral angle // degrees
//
//	core::Real const disulf_neg_dihedral_stdev = { 11.1 };
//	 // standard dev. of neg. dihedral angle in natives // degrees
//
//	core::Real const ideal_ca_dihedral_in_disulf = { 74.0 };
//	 // ideal calpha-cbeta-sulfur-sulfur dihedral (abs val) // degrees


FullatomDisulfidePotential::FullatomDisulfidePotential() :
	parent(),
	disulf_ssdist_cutoff_( 3.0 ),
	cbsg_dihedral_func_( new CBSG_Dihedral_Func ),
	sgsg_dihedral_func_( new SGSG_Dihedral_Func ),
	cb_angle_func_( new CB_Angle_Func ),
	sg_dist_func_( new SG_Dist_Func )
{}

FullatomDisulfidePotential::~FullatomDisulfidePotential() {}

/**
 * @brief Calculated scores for a disulfide bond between two residues
 *
 * @param[in] res1 The lower residue of the disulfide
 * @param[in] res2 The upper residue of the disulfide. Assumed to be bonded to res1
 * @param[out] distance_score_this_disulfide A score based on S-S distance
 * @param[out] csangles_score_this_disulfide A score based on the Cb-S angles
 * @param[out] dihedral_score_this_disulfide A score based on the S-S dihedral
 * @param[out] ca_dihedral_sc_this_disulf A score based on the Cb-S dihedrals
 * @param[out] truefalse_fa_disulf True if these residues should be disulfide bonded
 *
 * @details Scores are interpolated from the histograms created by the
 *  farlx_*_initializer methods.
 *  The distance score has such a histogram as its core with two linear
 *  functions at either side so that the score increases to infinity
 */
void
FullatomDisulfidePotential::score_this_disulfide(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	DisulfideAtomIndices const & res1_atom_indices,
	DisulfideAtomIndices const & res2_atom_indices,
	Energy & distance_score_this_disulfide,
	Energy & csangles_score_this_disulfide,
	Energy & dihedral_score_this_disulfide,
	Energy & ca_dihedral_sc_this_disulf,
	bool & truefalse_fa_disulf
	) const
{
	// Allocate memory for disulf params. Values are set by get_disulfide_params
	// dist between cys sulfurs
	Real ssdist(-1);
	// Cb-S-S bond angles
	Real csang_1(-1),csang_2(-1);
	// dihedral (torsion) angle, cbeta1-s1-s2-cbeta2
	Real dihed(360);
	// dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	Real disulf_ca_dihedral_angle_1(360);
	// dihedral (torsion) angle, calpha2-cbeta2-s2-s1
	Real disulf_ca_dihedral_angle_2(360);

	//Assume res1 and res2 are bonded and score them
	get_disulfide_params(res1,res2,res1_atom_indices,res2_atom_indices,
		ssdist,csang_1,csang_2,dihed,disulf_ca_dihedral_angle_1,disulf_ca_dihedral_angle_2);

	distance_score_this_disulfide = 0;
	csangles_score_this_disulfide = 0;
	dihedral_score_this_disulfide = 0;
	ca_dihedral_sc_this_disulf = 0;
	truefalse_fa_disulf = false;

	// ssdist score:

	distance_score_this_disulfide = sg_dist_func_->func(ssdist);

	// csangle score:

	//Check that csang is in the right domain
	runtime_assert_msg( ! ( csang_1 > 180. || csang_2 > 180. ) , "Error csang > 180" );

	using namespace numeric::constants::d;
	Energy csang_1_score = cb_angle_func_->func(csang_1*degrees_to_radians);
	Energy csang_2_score = cb_angle_func_->func(csang_2*degrees_to_radians);
	csangles_score_this_disulfide = (csang_1_score+csang_2_score)*0.5;

	// dihedral score:

	dihedral_score_this_disulfide = sgsg_dihedral_func_->func(dihed*degrees_to_radians);

	// ca_dihedral score: (this is just the chi_2 angle, rotation about cbeta-sulfur)

	// chi_2 of cys_1
	ca_dihedral_sc_this_disulf = cbsg_dihedral_func_->func(disulf_ca_dihedral_angle_1*degrees_to_radians);
	ca_dihedral_sc_this_disulf += cbsg_dihedral_func_->func(disulf_ca_dihedral_angle_2*degrees_to_radians);
	//Compensate for adding the scores together
	ca_dihedral_sc_this_disulf *= .5 ;

	// Call it a disulfide or not!
	if ( ssdist < disulf_ssdist_cutoff_ ) {
		truefalse_fa_disulf = true;
	}
}


void
FullatomDisulfidePotential::get_disulfide_derivatives(
	pose::Pose const & pose,
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	DisulfideAtomIndices const & res1_atom_indices,
	DisulfideAtomIndices const & res2_atom_indices,
	Size const at1,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	using namespace id;
	using namespace constraints;

	Vector f1( 0.0 ), f2( 0.0 );
	if ( res1_atom_indices.derivative_atom( at1 ) == CYS_C_ALPHA ) {

		f1 = f2 = 0;
		DihedralConstraint dihedral_ang_cst = DihedralConstraint(
			AtomID( at1, res1.seqpos() ),
			AtomID( res1_atom_indices.c_beta_index(), res1.seqpos() ),
			AtomID( res1_atom_indices.disulf_atom_index(), res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
				cbsg_dihedral_func_, dslf_ca_dih );
		dihedral_ang_cst.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

	} else if ( res1_atom_indices.derivative_atom( at1 ) == CYS_C_BETA  ) {

		f1 = f2 = 0;
		AngleConstraint ang_cst = AngleConstraint(
			AtomID( at1, res1.seqpos() ),
			AtomID( res1_atom_indices.disulf_atom_index(), res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
				cb_angle_func_, dslf_cs_ang );
		ang_cst.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		DihedralConstraint dihedral_ang_cst = DihedralConstraint(
			AtomID( res1_atom_indices.c_alpha_index(), res1.seqpos() ),
			AtomID( at1, res1.seqpos() ),
			AtomID( res1_atom_indices.disulf_atom_index(), res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
				cbsg_dihedral_func_, dslf_ca_dih );
		dihedral_ang_cst.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		DihedralConstraint ss_dihedral_ang_cst = DihedralConstraint(
			AtomID( at1, res1.seqpos() ),
			AtomID( res1_atom_indices.disulf_atom_index(), res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
			AtomID( res2_atom_indices.c_beta_index(), res2.seqpos() ),
				sgsg_dihedral_func_, dslf_ss_dih );
		ss_dihedral_ang_cst.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += f1;
		F2 += f2;

	} else if ( res1_atom_indices.derivative_atom( at1 ) == CYS_S_GAMMA ) {

		AtomPairConstraint apc( AtomID( at1, res1.seqpos() ), AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ), sg_dist_func_, dslf_ss_dst );
		apc.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += f1;
		F2 += f2;

		f1 = f2 = 0;
		AngleConstraint ang_cst1 = AngleConstraint(
			AtomID( res1_atom_indices.c_beta_index(), res1.seqpos() ),
			AtomID( at1, res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
				cb_angle_func_, dslf_cs_ang );
		ang_cst1.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		AngleConstraint ang_cst2 = AngleConstraint(
			AtomID( at1, res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
			AtomID( res2_atom_indices.c_beta_index(), res2.seqpos() ),
				cb_angle_func_, dslf_cs_ang );
		ang_cst2.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		DihedralConstraint dihedral_ang_cst1 = DihedralConstraint(
			AtomID( res1_atom_indices.c_alpha_index(), res1.seqpos() ),
			AtomID( res1_atom_indices.c_beta_index(), res1.seqpos() ),
			AtomID( at1, res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
				cbsg_dihedral_func_, dslf_ca_dih );
		dihedral_ang_cst1.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		DihedralConstraint dihedral_ang_cst2 = DihedralConstraint(
			AtomID( res2.atom_index("CA"), res2.seqpos() ),
			AtomID( res2_atom_indices.c_beta_index(), res2.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
			AtomID( at1, res1.seqpos() ),
				cbsg_dihedral_func_, dslf_ca_dih );
		dihedral_ang_cst2.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += 0.5 * f1;
		F2 += 0.5 * f2;

		f1 = f2 = 0;
		DihedralConstraint ss_dihedral_ang_cst = DihedralConstraint(
			AtomID( res1_atom_indices.c_beta_index(), res1.seqpos() ),
			AtomID( at1, res1.seqpos() ),
			AtomID( res2_atom_indices.disulf_atom_index(), res2.seqpos() ),
			AtomID( res2_atom_indices.c_beta_index(), res2.seqpos() ),
				sgsg_dihedral_func_, dslf_ss_dih );
		ss_dihedral_ang_cst.fill_f1_f2( AtomID( at1, res1.seqpos() ), pose.conformation(), f1, f2, weights );
		F1 += f1;
		F2 += f2;

	}
}

///////////////////////////////////////////
/// Private: Methods, Data Initializers ///
///////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
/// @brief Calculates several geometrical parameters of a fullatom disulfide bond
/// @details
///   given residue num for 2 cys involved in disulfide bond, returns
///   four quantities:
///   sulf-sulf dist, 2 carb-sulf bond angles, and a 4-atom dihedral angle.
///   Angles are returned in degrees.
///
/// @param[in]   coord1 - in - fullatom coords of cys 1
/// @param[in]   coord2 - in - fullatom coords of cys 2
/// @param[out]   dist_between_sulfurs - out - s1-s2 distance
/// @param[out]   cs_bond_angle_1 - out - cb1-s1-s2 bond angle
/// @param[out]   cs_bond_angle_2 - out - cb2-s1-s2 bond angle
/// @param[out]   disulf_dihedral_angle - out - cb1-s1-s2-cb2 dihedral
/// @param[out]   disulf_ca_dihedral_angle_1 - out - ca1-cb1-s1-s2 dihedral
/// @param[out]   disulf_ca_dihedral_angle_2 - out - ca2-cb2-s2-s1 dihedral
///
/// @author Bill Schief
/////////////////////////////////////////////////////////////////////////////////
void
FullatomDisulfidePotential::get_disulfide_params(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	DisulfideAtomIndices const & res1_atom_indices,
	DisulfideAtomIndices const & res2_atom_indices,
	Distance & dist_between_sulfurs, // dist between cys sulfurs
	Real & cs_bond_angle_1,
	Real & cs_bond_angle_2,
	Real & disulf_dihedral_angle, // dihedral (torsion) angle, cbeta-s-s-cbeta
	Real & disulf_ca_dihedral_angle_1,
	 // dihedral (torsion) angle, calpha1-cbeta1-s1-s2
	Real & disulf_ca_dihedral_angle_2 // dihedral (torsion) angle, calpha2-cbeta2-s2-s1
) const
{
	using namespace numeric::constants::d;

	Vector calpha_1( res1.atom( res1_atom_indices.c_alpha_index() ).xyz() );
	Vector cbeta_1 ( res1.atom( res1_atom_indices.c_beta_index( ) ).xyz() );
	Vector sulfur_1( res1.atom( res1_atom_indices.disulf_atom_index() ).xyz() );
	Vector calpha_2( res2.atom( res2_atom_indices.c_alpha_index() ).xyz() );
	Vector cbeta_2 ( res2.atom( res2_atom_indices.c_beta_index( ) ).xyz() );
	Vector sulfur_2( res2.atom( res2_atom_indices.disulf_atom_index() ).xyz() );

	dist_between_sulfurs = sulfur_1.distance( sulfur_2 );
	cs_bond_angle_1 = angle_of( cbeta_1, sulfur_1, sulfur_2);
	cs_bond_angle_2 = angle_of( cbeta_2, sulfur_2, sulfur_1);
	cs_bond_angle_1 *= radians_to_degrees; // convert
	cs_bond_angle_2 *= radians_to_degrees; // convert
	disulf_dihedral_angle      = dihedral_degrees(cbeta_1,sulfur_1,sulfur_2,cbeta_2);
	disulf_ca_dihedral_angle_1 = dihedral_degrees(calpha_1,cbeta_1,sulfur_1,sulfur_2);
	disulf_ca_dihedral_angle_2 = dihedral_degrees(calpha_2,cbeta_2,sulfur_2,sulfur_1);
}


//------------------------------------------------------------------------------

CBSG_Dihedral_Func::CBSG_Dihedral_Func() :
	cbsg_pos_peak_( 86.0 ),
	cbsg_pos_sd_( 72.0 ),
	cbsg_neg_peak_( -74.0 ),
	cbsg_neg_sd_( 39.0 )
{}

CBSG_Dihedral_Func::~CBSG_Dihedral_Func() {}

/// @param ang[in] Dihedral angle in radians
/// @note This function is not continuous across 0 or 180 degrees. This is bad,
///  and is only acceptable because it occurs at the maximums so we minimize
///  away from the discontinuity. -Spencer
Real
CBSG_Dihedral_Func::func( Real const ang ) const
{
	using namespace numeric::constants::d;

	Real ang_deg = ang * radians_to_degrees;
	Real const ang_peak  = ang_deg > 0.0 ? cbsg_pos_peak_ : cbsg_neg_peak_;
	Real const ang_sd    = ang_deg > 0.0 ? cbsg_pos_sd_   : cbsg_neg_sd_;
	Real const delta_ang = ang_deg - ang_peak;
	Real const ang_frac  = delta_ang / ang_sd;
	return -std::exp(-(ang_frac*ang_frac));
}

/// @param ang[in] Dihedral angle in radians
Real
CBSG_Dihedral_Func::dfunc( Real const ang ) const {
	using namespace numeric::constants::d;

	Real ang_deg = ang * radians_to_degrees;
	Real const ang_peak  = ang_deg > 0.0 ? cbsg_pos_peak_ : cbsg_neg_peak_;
	Real const ang_sd    = ang_deg > 0.0 ? cbsg_pos_sd_   : cbsg_neg_sd_;
	Real const delta_ang = ang_deg - ang_peak;
	Real const ang_frac  = delta_ang / ang_sd;

	// d/d ang_deg  of -1 * e^( -1 * ((ang_deg - ang_peak) / ang_sd )^2)
	// =
	// -1 * e^( -1 * (ang_deg - ang_peak / ang_sd )^2) * [ d/d ang_deg of -1 * ( ang_deg^2 - 2*ang_peak*ang_deg + ang_peak^2 ) /ang_sd^2 ]
	// =
	// e^( -1 * (ang_deg - ang_peak / ang_sd )^2) * ( 2*ang_deg - 2*ang_peak ) / (ang_sg * ang_sd )

	//std::cout << "CBSG_Dihedral_Func::dfunc, ang_deg = " << ang_deg << " ang_peak: " << ang_peak << " delta_ang: " << delta_ang << " deriv: " << radians_to_degrees * std::exp(-(ang_frac*ang_frac)) * 2 * ( ang_deg - ang_peak ) / (ang_sd * ang_sd ) << std::endl;

	return radians_to_degrees * std::exp(-(ang_frac*ang_frac)) * ( 2*ang_deg - 2*ang_peak ) / ( ang_sd * ang_sd );
}

///////////////

SGSG_Dihedral_Func::SGSG_Dihedral_Func() {}

SGSG_Dihedral_Func::~SGSG_Dihedral_Func() {}

/// @param ang[in] Dihedral angle in radians
Real
SGSG_Dihedral_Func::func( Real const ang) const
{
	using namespace numeric::constants::d;

	Real ang_deg = ang * radians_to_degrees;
	Real score(0);
	fa_sgsg_dihedral_scores()->interpolate(ang_deg,score);
	return score;
}

/// @param ang[in] Dihedral angle in radians
Real
SGSG_Dihedral_Func::dfunc( Real const ang ) const
{
	using namespace numeric::constants::d;

	Real ang_deg = ang * radians_to_degrees;
	Real deriv;
	fa_sgsg_dihedral_scores()->derivative(ang_deg, deriv); // units/degree
	return deriv*radians_to_degrees; // return units/radian
}

/// Access the histogram for this Func (in degrees)
numeric::interpolation::HistogramCOP<core::Real,core::Real>::Type
SGSG_Dihedral_Func::fa_sgsg_dihedral_scores()
{
	using namespace numeric::interpolation;
	static HistogramCOP<Real,Real>::Type scores(0);
	if(scores == 0) {
		utility::io::izstream scores_stream;
		io::database::open( scores_stream, "scoring/score_functions/disulfides/fa_CbSSCb_dihedral_score");
		scores = new Histogram<Real,Real>( scores_stream() );
		scores_stream.close();
	}
	return scores;
}

///////////////

CB_Angle_Func::CB_Angle_Func() {}

CB_Angle_Func::~CB_Angle_Func() {}

/// @param csang[in] Dihedral angle in radians
Real
CB_Angle_Func::func( Real const ang) const
{
	using namespace numeric::constants::d;

	Real ang_deg = ang*radians_to_degrees;
	//Check that csang is in the right domain
	runtime_assert_msg(  0 <= ang_deg && ang_deg <= 180., "Error csang > 180" );

	Real score(0);
	// Will be out of the histogram's range sometimes, but we like the default
	// behavior of using the boundary scores for extreme ang_deg
	CB_Angle_Func::fa_csang_scores()->interpolate(ang_deg,score);
	return score;
}

/// @param ang[in] Dihedral angle in radians
Real
CB_Angle_Func::dfunc( Real const ang ) const
{
	using namespace numeric::constants::d;
	Real ang_deg = ang * radians_to_degrees;
	//std::cout << "ang_deg: " << ang_deg << std::endl;

	Real d_csang_score_dang( 0.0 );
	// Should be zero when ang_deg is not in the hist range
	CB_Angle_Func::fa_csang_scores()->derivative(ang_deg,d_csang_score_dang);
	return d_csang_score_dang * radians_to_degrees;
}

/// Access the histogram for this Func (in degrees)
numeric::interpolation::HistogramCOP<core::Real,core::Real>::Type
CB_Angle_Func::fa_csang_scores()
{
	using namespace numeric::interpolation;
	static HistogramCOP<Real,Real>::Type scores(0);
	if(scores == 0) {
		utility::io::izstream scores_stream;
		io::database::open( scores_stream, "scoring/score_functions/disulfides/fa_CaCbS_angle_score");
		scores = new Histogram<Real,Real>( scores_stream() );
		scores_stream.close();
	}
	return scores;
}

//////////////////

SG_Dist_Func::SG_Dist_Func() {}

SG_Dist_Func::~SG_Dist_Func() {}

/// @param ssdist[in] S-S distance, in Angstroms
Real
SG_Dist_Func::func( Real const ssdist ) const
{
	Real distance_score_this_disulfide( 0.0 );

	numeric::interpolation::HistogramCOP<Real,Real>::Type scores =
		SG_Dist_Func::fa_ssdist_scores();
	Real ssdist_min_range( scores->minimum() );
	Real ssdist_max_range( scores->maximum() );

	if ( ssdist >= ssdist_max_range ) { // max tail
		Real ssdist_score_at_max(0);
		scores->interpolate(ssdist_max_range,ssdist_score_at_max);
		Real ssdist_max_tail_slope(0);
		scores->derivative(ssdist_max_range-scores->step_size(),ssdist_max_tail_slope);

		distance_score_this_disulfide = ssdist_score_at_max + ( ssdist - ssdist_max_range ) * ssdist_max_tail_slope;
	} else if ( ssdist <= ssdist_min_range ) { // min tail
		Real ssdist_score_at_min(0);
		scores->interpolate(ssdist_min_range,ssdist_score_at_min);
		Real ssdist_min_tail_slope(0);
		scores->derivative(ssdist_min_range,ssdist_min_tail_slope);

		distance_score_this_disulfide = ssdist_score_at_min + ( ssdist - ssdist_min_range ) * ssdist_min_tail_slope;
	} else {
		SG_Dist_Func::fa_ssdist_scores()->interpolate(ssdist, distance_score_this_disulfide);
	}
	return distance_score_this_disulfide;
}

/// @param ssdist[in] S-S distance, in Angstroms
Real
SG_Dist_Func::dfunc( Real const ssdist ) const {
	Real d_distance_score_this_disulfide_ddis( 0.0 );

	numeric::interpolation::HistogramCOP<Real,Real>::Type scores =
		SG_Dist_Func::fa_ssdist_scores();
	Real ssdist_min_range( scores->minimum() );
	Real ssdist_max_range( scores->maximum() );
	if ( ssdist >= ssdist_max_range ) { // max tail
		scores->derivative(ssdist_max_range-scores->step_size(),d_distance_score_this_disulfide_ddis);
	} else if ( ssdist <= ssdist_min_range ) { // min tail
		scores->derivative(ssdist_min_range,d_distance_score_this_disulfide_ddis);
	} else {
		SG_Dist_Func::fa_ssdist_scores()->derivative(ssdist, d_distance_score_this_disulfide_ddis);
	}
	return d_distance_score_this_disulfide_ddis;
}

/// Access the histogram for this Func
numeric::interpolation::HistogramCOP<core::Real,core::Real>::Type
SG_Dist_Func::fa_ssdist_scores()
{
	using namespace numeric::interpolation;
	static HistogramCOP<Real,Real>::Type scores(0);
	if(scores == 0) {
		utility::io::izstream scores_stream;
		io::database::open( scores_stream, "scoring/score_functions/disulfides/fa_SS_distance_score");
		scores = new Histogram<Real,Real>( scores_stream() );
		scores_stream.close();
	}
	return scores;
}

}
}
}

