// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/electron_density/ElectronDensity.cc
/// @brief  Scoring a structure against an electron density map
/// @author Frank DiMaio

// Unit Headers
#include <core/scoring/electron_density/ElectronDensity.hh>
#include <core/scoring/electron_density/SplineInterp.hh>
#include <core/scoring/electron_density/util.hh>
#ifdef WIN32
	#define _USE_MATH_DEFINES
	#include <math.h>
#endif
// Project headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/symmetry/SymmetryInfo.hh>
#include <core/conformation/symmetry/util.hh>

#include <core/options/option.hh>
#include <core/options/after_opts.hh>
#include <core/options/util.hh>

// Utility headers
#include <numeric/fourier/FFT.hh>
#include <numeric/xyzMatrix.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/statistics.functions.hh>
#include <core/util/Tracer.hh>

//
#include <core/options/keys/edensity.OptionKeys.gen.hh>

// C++ headers
#include <fstream>


namespace core {
namespace scoring {
namespace electron_density {

using core::util::T;
using core::util::Tracer;
static core::util::Tracer TR("core.scoring.electron_density.ElectronDensity");

using namespace core;
using namespace core::options;

#ifndef WIN32
const core::Real FLT_MAX=1e37;
#endif

const int CCP4HDSIZE = 1024;  // size of CCP4/MRC header

//const int DEFAULT_HARMONIC_BW=32;
//const core::Real DEFAULT_HARMONIC_STEPSIZE=1.0;

//
// weight from scattering factors
// should be encoded in database
core::Real get_a( std::string elt ) {
	if (elt == "C") return 6;
	if (elt == "N") return 7;
	if (elt == "O") return 8;
	if (elt == "P") return 15;
	if (elt == "S") return 16;
	if (elt == "X") return 0;  // centroid

	// default to C
	TR.Warning << "[ WARNING ] Unknown atom " << elt << std::endl;
	return 6;
}


//
// one-liners
inline float d2r(float d) { return (d*M_PI/180.0); }
inline double d2r(double d) { return (d*M_PI/180.0); }
inline float  square(float  x) { return (x*x); }
inline double square(double x) { return (x*x); }

// x mod y, returns z in [0,y-1]
inline int pos_mod(int x,int y) {
	int r=x%y; if (r<0) r+=y;
	return r;
}
inline float pos_mod(float x,float y) {
	float r=std::fmod(x,y); if (r<0) r+=y;
	return r;
}
inline double pos_mod(double x,double y) {
	double r=std::fmod(x,y); if (r<0) r+=y;
	return r;
}

// x mod y, returns z in [-y/2,y/2]
inline int min_mod(int x,int y) {
	int r=x%y; if (r<-y/2) r+=y;if (r>=y/2) r-=y;
	return r;
}
inline float min_mod(float x,float y) {
	float r=std::fmod(x,y); if (r<-0.5*y) r+=y;if (r>=0.5*y) r-=y;
	return r;
}
inline double min_mod(double x,double y) {
	double r=std::fmod(x,y); if (r<-0.5*y) r+=y;if (r>=0.5*y) r-=y;
	return r;
}

// missing density test
// pose_from_pdb randomizing missing density with:
// 		ai.x = ai.x + 900.000 + RG.uniform()*100.000;
// 		ai.y = ai.y + 900.000 + RG.uniform()*100.000;
// 		ai.z = ai.z + 900.000 + RG.uniform()*100.000;
inline bool is_missing_density( numeric::xyzVector< core::Real > const &X ) {
	if (  X[0] >= 900.000 && X[0] <= 1000.000
		&& X[1] >= 900.000 && X[1] <= 1000.000
		&& X[2] >= 900.000 && X[2] <= 1000.000 ) {
		return true;
	}
	return false;
}

// Endianness swap
// Only works with aligned 4-byte quantities
static void swap4_aligned(void *v, long ndata) {
	int *data = (int *) v;
	long i;
	int *N;
	for (i=0; i<ndata; i++) {
		N = data + i;
		*N=(((*N>>24)&0xff) | ((*N&0xff)<<24) | ((*N>>8)&0xff00) | ((*N&0xff00)<<8));
	}
}

///////////////////////////////  ///////////////////////////////
///////////////////////////////  ///////////////////////////////
///////////////////////////////  ///////////////////////////////

//
ElectronDensity &getDensityMap() {
	static ElectronDensity theDensityMap;

	if (!theDensityMap.isMapLoaded()) {
		// load map from disk
		TR << "Loading Density Map" << std::endl;
		if (!core::options::option[ core::options::OptionKeys::edensity::mapfile ].user()) {
			TR.Warning << "[ Warning ] No density map specified." << std::endl;
			//TR << "[ ERROR ] Density map score will not be used.\n";
			//exit(1);
		} else {
			std::string mapfile = core::options::option[ core::options::OptionKeys::edensity::mapfile ]();
			core::Real mapreso = core::options::option[ core::options::OptionKeys::edensity::mapreso ]();
			core::Real mapsampling = core::options::option[ core::options::OptionKeys::edensity::grid_spacing ]();

			// Initialize ElectronDensity object
			bool map_loaded = theDensityMap.readMRCandResize( mapfile , mapreso , mapsampling );

			if (!map_loaded) {
				TR << "[ ERROR ] Error loading density map named '" << mapfile << "'" << std::endl;
				//TR << "[ ERROR ] Density map score will not be used.\n";
				exit(1);
			}
		}
	}

	return theDensityMap;
}


///////////////////////////////  ///////////////////////////////
///////////////////////////////  ///////////////////////////////
///////////////////////////////  ///////////////////////////////

// trilinear interpolation of electron density
template <class S>
core::Real ElectronDensity::interp_linear(
                       ObjexxFCL::FArray3D< S > const & data ,
                       numeric::xyzVector< core::Real > const & idxX) const {
	int pt000[3], pt111[3];
	core::Real fpart[3],neg_fpart[3];
	core::Real retval = 0.0;

	// find bounding grid points
	pt000[0] = (int)(floor(idxX[0])) % grid[0]; if (pt000[0] <= 0) pt000[0]+= grid[0];
	pt000[1] = (int)(floor(idxX[1])) % grid[1]; if (pt000[1] <= 0) pt000[1]+= grid[1];
	pt000[2] = (int)(floor(idxX[2])) % grid[2]; if (pt000[2] <= 0) pt000[2]+= grid[2];
	pt111[0] = (pt000[0]+1); if (pt111[0]>grid[0]) pt111[0] = 1;
	pt111[1] = (pt000[1]+1); if (pt111[1]>grid[0]) pt111[1] = 1;
	pt111[2] = (pt000[2]+1); if (pt111[2]>grid[0]) pt111[2] = 1;

	// interpolation coeffs
	fpart[0] = idxX[0]-floor(idxX[0]); neg_fpart[0] = 1-fpart[0];
	fpart[1] = idxX[1]-floor(idxX[1]); neg_fpart[1] = 1-fpart[1];
	fpart[2] = idxX[2]-floor(idxX[2]); neg_fpart[2] = 1-fpart[2];

	// take care of the case when density map does not cover the entire unit cell
	if (pt000[0] <= density.u1())
		if (pt000[1] <= density.u2())
			if (pt000[2] <= density.u3())
				retval+= neg_fpart[0]*neg_fpart[1]*neg_fpart[2] * data(pt000[0],pt000[1],pt000[2]);
			if (pt111[2] <= density.u3())
			retval+= neg_fpart[0]*neg_fpart[1]*    fpart[2] * data(pt000[0],pt000[1],pt111[2]);
		if (pt111[1] <= density.u2())
			if (pt000[2] <= density.u3())
			retval+= neg_fpart[0]*    fpart[1]*neg_fpart[2] * data(pt000[0],pt111[1],pt000[2]);
			if (pt111[2] <= density.u3())
			retval+= neg_fpart[0]*    fpart[1]*    fpart[2] * data(pt000[0],pt111[1],pt111[2]);

	if (pt111[0] <= density.u1())
		if (pt000[1] <= density.u2())
			if (pt000[2] <= density.u3())
			retval+= fpart[0]*neg_fpart[1]*neg_fpart[2] * data(pt111[0],pt000[1],pt000[2]);
			if (pt111[2] <= density.u3())
			retval+= fpart[0]*neg_fpart[1]*    fpart[2] * data(pt111[0],pt000[1],pt111[2]);
		if (pt111[1] <= density.u2())
			if (pt000[2] <= density.u3())
			retval+= fpart[0]*    fpart[1]*neg_fpart[2] * data(pt111[0],pt111[1],pt000[2]);
			if (pt111[2] <= density.u3())
			retval+= fpart[0]*    fpart[1]*    fpart[2] * data(pt111[0],pt111[1],pt111[2]);

		return retval;
}

// trilinear "inverse" interpolation of electron density
void ElectronDensity::inc_linear(
                         ObjexxFCL::FArray3D< double > & data ,
                         numeric::xyzVector< core::Real > const & idxX,
                         core::Real value) {
	int pt000[3], pt111[3];
	core::Real fpart[3],neg_fpart[3];

	// find bounding grid points
	pt000[0] = (int)(floor(idxX[0])) % grid[0]; if (pt000[0] <= 0) pt000[0]+= grid[0];
	pt000[1] = (int)(floor(idxX[1])) % grid[1]; if (pt000[1] <= 0) pt000[1]+= grid[1];
	pt000[2] = (int)(floor(idxX[2])) % grid[2]; if (pt000[2] <= 0) pt000[2]+= grid[2];
	pt111[0] = (pt000[0]+1); if (pt111[0]>grid[0]) pt111[0] = 1;
	pt111[1] = (pt000[1]+1); if (pt111[1]>grid[0]) pt111[1] = 1;
	pt111[2] = (pt000[2]+1); if (pt111[2]>grid[0]) pt111[2] = 1;

	// interpolation coeffs
	fpart[0] = idxX[0]-floor(idxX[0]); neg_fpart[0] = 1-fpart[0];
	fpart[1] = idxX[1]-floor(idxX[1]); neg_fpart[1] = 1-fpart[1];
	fpart[2] = idxX[2]-floor(idxX[2]); neg_fpart[2] = 1-fpart[2];

	// take care of the case when density map does not cover the entire unit cell
	if (pt000[0] <= density.u1())
		if (pt000[1] <= density.u2())
			if (pt000[2] <= density.u3())
				data(pt000[0],pt000[1],pt000[2]) += neg_fpart[0]*neg_fpart[1]*neg_fpart[2] * value;
			if (pt111[2] <= density.u3())
			data(pt000[0],pt000[1],pt111[2]) += neg_fpart[0]*neg_fpart[1]*    fpart[2] * value;
		if (pt111[1] <= density.u2())
			if (pt000[2] <= density.u3())
			data(pt000[0],pt111[1],pt000[2]) += neg_fpart[0]*    fpart[1]*neg_fpart[2] * value;
			if (pt111[2] <= density.u3())
			data(pt000[0],pt111[1],pt111[2]) += neg_fpart[0]*    fpart[1]*    fpart[2] * value;

	if (pt111[0] <= density.u1())
		if (pt000[1] <= density.u2())
			if (pt000[2] <= density.u3())
			data(pt111[0],pt000[1],pt000[2]) += fpart[0]*neg_fpart[1]*neg_fpart[2] * value;
			if (pt111[2] <= density.u3())
			data(pt111[0],pt000[1],pt111[2]) += fpart[0]*neg_fpart[1]*    fpart[2] * value;
		if (pt111[1] <= density.u2())
			if (pt000[2] <= density.u3())
			data(pt111[0],pt111[1],pt000[2]) += fpart[0]*    fpart[1]*neg_fpart[2] * value;
			if (pt111[2] <= density.u3())
			data(pt111[0],pt111[1],pt111[2]) += fpart[0]*    fpart[1]*    fpart[2] * value;
}

core::Real ElectronDensity::interp_spline(
                         ObjexxFCL::FArray3D< double > & coeffs ,
                         numeric::xyzVector< core::Real > const & idxX ) const {
	int dims[3] = { coeffs.u3(), coeffs.u2(), coeffs.u1() };
	core::Real pt[3] = { idxX[2]-1.0 , idxX[1]-1.0, idxX[0]-1.0 };
	core::Real retval = SplineInterp::interp3(&coeffs[0], dims, pt, 3);
	return retval;
}

// gradient of density
numeric::xyzVector<core::Real> ElectronDensity::dens_grad (
			numeric::xyzVector<core::Real> const & idxX ) const {
	numeric::xyzVector< core::Real > dx;
	dx[0] = interp_spline( coeff_grad_x, idxX );
	dx[1] = interp_spline( coeff_grad_y, idxX );
	dx[2] = interp_spline( coeff_grad_z, idxX );
	return dx;
}


void ElectronDensity::spline_coeffs(
                         ObjexxFCL::FArray3D< double > & data ,
                         ObjexxFCL::FArray3D< double > & coeffs) {
	int dims[3] = { data.u3(), data.u2(), data.u1() };
	coeffs = data;
	SplineInterp::compute_coefficients( &coeffs[0] , dims , 3 ); // cubic spline interpolation
}

void ElectronDensity::spline_coeffs(
                         ObjexxFCL::FArray3D< float > & data ,
                         ObjexxFCL::FArray3D< double > & coeffs) {
	int N = data.u3()*data.u2()*data.u1();
	ObjexxFCL::FArray3D< double > data_d(data.u1(),data.u2(),data.u3()) ;
	for (int i=0; i<N; ++i)
		data_d[i] = (double)data[i];
	spline_coeffs( data_d, coeffs ); // cubic spline interpolation
}


// NN interpolation of electron density
template<class T>
void ElectronDensity::set_nearest(
                   ObjexxFCL::FArray3D< T > & data ,
                   numeric::xyzVector<double> const & idxX,
                   T value) {
	int pt000[3];

	// find bounding grid points
	pt000[0] = (int)(floor(idxX[0]+0.5)) % grid[0]; if (pt000[0] <= 0) pt000[0]+= grid[0];
	pt000[1] = (int)(floor(idxX[1]+0.5)) % grid[1]; if (pt000[1] <= 0) pt000[1]+= grid[1];
	pt000[2] = (int)(floor(idxX[2]+0.5)) % grid[2]; if (pt000[2] <= 0) pt000[2]+= grid[2];

	// take care of the case when density map does not cover the entire unit cell
	if (pt000[0] <= density.u1())
		if (pt000[1] <= density.u2())
			if (pt000[2] <= density.u3())
				data(pt000[0],pt000[1],pt000[2]) = value;
}



/// null constructor
ElectronDensity::ElectronDensity() {
	isLoaded = false;

	grid = numeric::xyzVector< int >(0,0,0);
	efforigin = origin = numeric::xyzVector< int >(0,0,0);
	cellDimensions = numeric::xyzVector< float >(1,1,1);
	cellAngles = numeric::xyzVector< float >(90,90,90);

	// resolution-dependent ... these values are set in readMRCandResize
	reso = -1.0;
	ATOM_MASK = -1.0;
	CA_MASK   = -1.0;

	// more defaults
	DensScoreInMinimizer = true;
	ExactDerivatives = false;

	// only used when computing exact derivatives
	NUM_DERIV_H = 0.0001;
	NUM_DERIV_H_CEN = 0.0001;
}


int ElectronDensity::suggestRadius( ) {
	core::Real dens_cutoff = dens_mean + dens_stdev;  // threshold at 1*sigma

	double maxLen = (f2c*numeric::xyzVector<core::Real>(1.0,1.0,1.0)).length();
	int maxDist = (int) ceil( maxLen+1.5 );
	              // super-high upper bound
	utility::vector1<int> aboveThreshCtr(maxDist,0);
	int aboveThreshSum=0;

	numeric::xyzVector<core::Real> fracCenter(centerOfMass[0]/grid[0] , centerOfMass[1]/grid[1] , centerOfMass[2]/grid[2]);
	//numeric::xyzVector<double> fracCenter( 0.0 , 0.0 , 0.0 );

	TR << "ElectronDensity::suggestRadius( )\n";
	TR << "  com       : " << centerOfMass[0] << " , " << centerOfMass[1] << " , " << centerOfMass[2] << std::endl;
	TR << "  grid      : " << grid[0] << " , " << grid[1] << " , " << grid[2] << std::endl;
	TR << "  fracCenter: " << fracCenter[0] << " , " << fracCenter[1] << " , " << fracCenter[2] << std::endl;

	int maxI = 0;
	for (int x=1; x<density.u1(); ++x)
	for (int y=1; y<density.u2(); ++y)
	for (int z=1; z<density.u3(); ++z) {
		numeric::xyzVector<core::Real> fracX( ((core::Real)x ) / grid[0] ,
		                                      ((core::Real)y ) / grid[1] ,
		                                      ((core::Real)z ) / grid[2] );
		double len = (f2c*(fracX - fracCenter)).length();
		int lenI = (int)floor( len+1.5 );

		// err chk
		if (lenI > maxDist) {
			TR.Error << "ERROR! " << lenI << " = || [" << x << "," << y << "," << z << "] || > " <<  maxDist << std::endl;
			exit(1);
		}

		if ( density(x,y,z) > dens_cutoff ) {
			aboveThreshCtr[lenI]++;
			aboveThreshSum++;
			maxI = std::max(maxI,lenI);
		}
	}

	//for (int i=2; i<=maxI; ++i)
	//	std::cerr << i << "   ->   " << aboveThreshCtr[i] << std::endl;
	for (int i=2; i<=maxI; ++i) {
		aboveThreshCtr[i] += aboveThreshCtr[i-1];
		if ( aboveThreshCtr[i] > 0.95*aboveThreshSum)
			return (i-1);
	}
	return maxI; // get rid of a compiler warning
}


/////////////////////////////////////
/////////////////////////////////////
// Match a centroid pose to the density map, returning correlation coefficient between
//    map and pose.
core::Real ElectronDensity::matchCentroidPose(
		core::pose::Pose const &pose,
 		const core::conformation::symmetry::SymmetryInfo *symmInfo /*=NULL*/,
 		bool cacheCCs /* = false */) {
	using namespace numeric::statistics;

	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::matchCentroidPose called but no map is loaded!\n";
		return 0.0;
	}

	if (!DensScoreInMinimizer) cacheCCs = false;

	ObjexxFCL::FArray3D< double >  rho_calc, inv_rho_mask;
	rho_calc.dimension(density.u1() , density.u2() , density.u3());
	inv_rho_mask.dimension(density.u1() , density.u2() , density.u3());
	for (int i=0; i<density.u1()*density.u2()*density.u3(); ++i) {
		rho_calc[i]=0.0;
		inv_rho_mask[i]=1.0;
	}

	int nres = pose.total_residue(); //reses.size();
	numeric::xyzVector< core::Real > cartX, fracX;
	numeric::xyzVector< core::Real > atm_i, atm_j, del_ij;

	// compute RHO_C --> a gaussian at each CA
	core::Real effReso = std::max( 2.4+0.8*reso , reso );
	core::Real k=square(M_PI/effReso);
	core::Real C=pow(k/M_PI,1.5);
	core::Real a=33.0;  // treat everything as ALA
	                    //    maybe change this???

	// per-atom derivs
	utility::vector1< numeric::xyzVector<core::Real> >                     atm_idx(nres);
	utility::vector1< utility::vector1< int > >                            rho_dx_pt(nres);
	utility::vector1< utility::vector1< numeric::xyzVector<core::Real> > > rho_dx_mask(nres), rho_dx_atm(nres);

	// symmetry
	bool isSymm = (symmInfo != NULL);
	bool remapSymm = core::options::option[ core::options::OptionKeys::edensity::score_symm_complex ]();

	///////////////////////////
	/// 1 COMPUTE RHO_C, MASK
	for (int i=1 ; i<=nres; ++i) {
		conformation::Residue const &rsd_i (pose.residue(i));

		// skip vrts & masked reses
		if ( rsd_i.aa() == core::chemical::aa_vrt ) continue;
		if ( scoring_mask_.find(i) != scoring_mask_.end() ) continue;

		// symm
		if (isSymm && !symmInfo->bb_is_independent(i) && !remapSymm) {  // should this be fa_...??
			continue; // only score the independent monomer
		}

		conformation::Atom const &atm_i( rsd_i.atom("CA") );

		// skip randomized residues
		if ( is_missing_density( atm_i.xyz() ) ) continue;

		cartX = atm_i.xyz() - getTransform();
		fracX = c2f*cartX;
		atm_idx[i][0] = pos_mod (fracX[0]*grid[0] - origin[0] + 1 , (double)grid[0]);
		atm_idx[i][1] = pos_mod (fracX[1]*grid[1] - origin[1] + 1 , (double)grid[1]);
		atm_idx[i][2] = pos_mod (fracX[2]*grid[2] - origin[2] + 1 , (double)grid[2]);


		for (int z=1; z<=density.u3(); ++z) {
			atm_j[2] = z;
			del_ij[2] = (atm_idx[i][2] - atm_j[2]) / grid[2];
			// wrap-around??
			if (del_ij[2] > 0.5) del_ij[2]-=1.0;
			if (del_ij[2] < -0.5) del_ij[2]+=1.0;

			del_ij[0] = del_ij[1] = 0.0;
			if ((f2c*del_ij).length_squared() > (CA_MASK+1)*(CA_MASK+1)) continue;

			for (int y=1; y<=density.u2(); ++y) {
				atm_j[1] = y;

				// early exit?
				del_ij[1] = (atm_idx[i][1] - atm_j[1]) / grid[1] ;
				// wrap-around??
				if (del_ij[1] > 0.5) del_ij[1]-=1.0;
				if (del_ij[1] < -0.5) del_ij[1]+=1.0;
				del_ij[0] = 0.0;
				if ((f2c*del_ij).length_squared() > (CA_MASK+1)*(CA_MASK+1)) continue;

				for (int x=1; x<=density.u1(); ++x) {
					atm_j[0] = x;

					// early exit?
					del_ij[0] = (atm_idx[i][0] - atm_j[0]) / grid[0];
					// wrap-around??
					if (del_ij[0] > 0.5) del_ij[0]-=1.0;
					if (del_ij[0] < -0.5) del_ij[0]+=1.0;

					numeric::xyzVector< core::Real > cart_del_ij = (f2c*del_ij);  // cartesian offset from (x,y,z) to atom_i
					core::Real d2 = (cart_del_ij).length_squared();

					if (d2 <= (CA_MASK+1)*(CA_MASK+1)) {
						// v1 ...
						core::Real atm = C*a*exp(-k*d2);
						core::Real inv_msk = 1/(1+exp( d2 - (CA_MASK)*(CA_MASK)  ));

						rho_calc(x,y,z) += atm;
						inv_rho_mask(x,y,z) *= (1 - inv_msk);

						if (cacheCCs) {
							int idx = (z-1)*density.u2()*density.u1() + (y-1)*density.u1() + x-1;
							rho_dx_pt[i].push_back  ( idx );
							rho_dx_mask[i].push_back( -inv_msk*cart_del_ij );
							rho_dx_atm[i].push_back ( atm*cart_del_ij );
						}
					}
				}
			}
		}
	}

	//////////////////////////
	/// 2 COMPUTE SUMMARY STATISTICS
	core::Real sumC_i=0, sumO_i=0, sumCO_i=0, vol_i=0, CC_i=0;
 	core::Real sumO2_i=0.0, sumC2_i=0.0, varC_i=0, varO_i=0;
	core::Real clc_x, obs_x, eps_x;

	for (int x=0; x<density.u1()*density.u2()*density.u3(); ++x) {
		// fetch this point
		clc_x = rho_calc[x];
		obs_x = density[x];
		eps_x = 1-inv_rho_mask[x]; //1/(1+exp( (0.01-rho_calc(x,y,z)) * 1000 ));  // sigmoidal

		// SMOOTHED
		sumCO_i += eps_x*clc_x*obs_x;
		sumO_i  += eps_x*obs_x;
		sumO2_i += eps_x*obs_x*obs_x;
		sumC_i  += eps_x*clc_x;
		sumC2_i += eps_x*clc_x*clc_x;
		vol_i   += eps_x;
	}
	varC_i = (sumC2_i - sumC_i*sumC_i / vol_i );
	varO_i = (sumO2_i - sumO_i*sumO_i / vol_i ) ;
	if (varC_i == 0 || varO_i == 0)
		CC_i = 0;
	else
		CC_i = (sumCO_i - sumC_i*sumO_i/ vol_i) / sqrt( varC_i * varO_i );

	if (cacheCCs)
		CC_cen = CC_i;


	///////////////////////////
	/// 3  CALCULATE SYMMETRIC ROTATION MATRICES + SYMM MAPPING at each level
	if (isSymm && remapSymm && cacheCCs) {
		compute_symm_rotations( pose, symmInfo );
	}

	///////////////////////////
	/// 4  CALCULATE PER-CA DERIVATIVES
	if (cacheCCs) {
		std::map< core::Size , numeric::xyzMatrix< core::Real > > symmRots;
		for (int i=1 ; i<=nres; ++i) {
			if (isSymm && !symmInfo->bb_is_independent(i) && !remapSymm) {  // should this be fa_...??
				continue; // only score the monomer
			}

			conformation::Residue const &rsd_i (pose.residue(i)); //( *reses[i] );

			if ( rsd_i.aa() == core::chemical::aa_vrt ) continue;
			if ( scoring_mask_.find(i) != scoring_mask_.end() ) continue;

			numeric::xyzVector< core::Real > dVdx_ij(0,0,0), dOdx_ij(0,0,0), dO2dx_ij(0,0,0), dCOdx_ij(0,0,0), dC2dx_ij(0,0,0);

			conformation::Atom const &atm_i( rsd_i.atom("CA") );
	 		if ( is_missing_density( atm_i.xyz() ) ) continue;

			utility::vector1< int > const &rho_dx_pt_ij   = rho_dx_pt[i];
			utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_mask_ij = rho_dx_mask[i];
			utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_atm_ij  = rho_dx_atm[i];

			int npoints = rho_dx_pt_ij.size();
			for (int n=1; n<=npoints; ++n) {
				const int x(rho_dx_pt_ij[n]);
				clc_x = rho_calc[x];
				obs_x = density[x];
				core::Real inv_eps_x = inv_rho_mask[x];

				numeric::xyzVector<double> del_mask = 2.0*inv_eps_x*rho_dx_mask_ij[n];
				dVdx_ij  += del_mask;
				dOdx_ij  += del_mask*obs_x;
				dO2dx_ij += del_mask*obs_x*obs_x;
				numeric::xyzVector<double> del_rhoc =-2.0*k*rho_dx_atm_ij[n];
				dCOdx_ij += del_rhoc*obs_x;
				dC2dx_ij += 2.0*del_rhoc*clc_x;
			}

			// finally compute dCC/dx_ij
			core::Real f = ( sumCO_i - sumC_i*sumO_i / vol_i );
			core::Real g = sqrt ( varO_i * varC_i );

			numeric::xyzVector<core::Real> fprime = dCOdx_ij - 1/(vol_i*vol_i) * ( dOdx_ij*sumC_i*vol_i - sumO_i*sumC_i*dVdx_ij);
			numeric::xyzVector<core::Real> gprime = 0.5 * (
					sqrt(varO_i)/sqrt(varC_i) * ( dC2dx_ij + ( sumC_i*sumC_i*dVdx_ij/(vol_i*vol_i) ) )  +
					sqrt(varC_i)/sqrt(varO_i) * ( dO2dx_ij - ( 1/(vol_i*vol_i) * ( 2*vol_i*sumO_i*dOdx_ij - sumO_i*sumO_i*dVdx_ij ) ) ) );

			dCCdxs_cen[i] = (g*fprime - f*gprime) / (g*g);
		}
	}
	// >> debugging <<
	//ElectronDensity(rho_calc, 2.0).writeMRC( "rho_calc.mrc" );
	//ElectronDensity(inv_rho_mask, 2.0).writeMRC( "inv_rho_mask.mrc" );
	//exit(1);

	//std::cerr << "ElectronDensity::matchCentroidPose() returning CC = " << CC_i << "   vol = " << vol_i << std::endl;

	return CC_i;
}


/////////////////////////////////////
/////////////////////////////////////
/// Match a residue to the density map, returning correlation coefficient between
///    map and pose
core::Real ElectronDensity::matchPose(
		core::pose::Pose const &pose,
		const core::conformation::symmetry::SymmetryInfo *symmInfo /*=NULL*/,
		bool cacheCCs/*=false*/ ) {
	using namespace numeric::statistics;

	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::matchPose called but no map is loaded!\n";
		return 0.0;
	}

	if (!DensScoreInMinimizer) cacheCCs = false;

	ObjexxFCL::FArray3D< double >  rho_calc, inv_rho_mask;
	rho_calc.dimension(density.u1() , density.u2() , density.u3());
	inv_rho_mask.dimension(density.u1() , density.u2() , density.u3());
	for (int i=0; i<density.u1()*density.u2()*density.u3(); ++i) {
		rho_calc[i]=0.0;
		inv_rho_mask[i]=1.0;
	}

	int nres = pose.total_residue(); //reses.size();
	numeric::xyzVector< core::Real > cartX, fracX;
	numeric::xyzVector< core::Real > atm_i, atm_j, del_ij;

	// compute RHO_C --> a gaussian at each CA
	core::Real k=square(M_PI/reso);
	core::Real C=pow(k/M_PI,1.5);
	//core::Real a=6.0;  // treat everything as C
	//                   //    maybe change this???
	core::Real SC_scaling = core::options::option[ core::options::OptionKeys::edensity::sc_scaling ]();

	// per-atom derivs
	utility::vector1< utility::vector1< numeric::xyzVector<core::Real> > >                     atm_idx(nres);
	utility::vector1< utility::vector1< utility::vector1< int > > >                            rho_dx_pt(nres);
	utility::vector1< utility::vector1< utility::vector1< numeric::xyzVector<core::Real> > > > rho_dx_mask(nres), rho_dx_atm(nres);

	// symmetry
	bool isSymm = (symmInfo != NULL);
	bool remapSymm = core::options::option[ core::options::OptionKeys::edensity::score_symm_complex ]();

	///////////////////////////
	/// 1 COMPUTE RHO_C, MASK
	for (int i=1 ; i<=nres; ++i) {
		conformation::Residue const &rsd_i (pose.residue(i)); //( *reses[i] );

		// skip vrts & masked reses
		if ( rsd_i.aa() == core::chemical::aa_vrt ) continue;
		if ( scoring_mask_.find(i) != scoring_mask_.end() ) continue;

		// symm
		if (isSymm && !symmInfo->bb_is_independent(i) && !remapSymm) {  // should this be fa_...??
			continue; // only score the independent monomer
		}

		int nheavyatoms = rsd_i.nheavyatoms();
		//int nheavyatoms = rsd_i.last_backbone_atom();

		atm_idx[i].resize(nheavyatoms);
		rho_dx_pt[i].resize(nheavyatoms);
		rho_dx_mask[i].resize(nheavyatoms);
		rho_dx_atm[i].resize(nheavyatoms);

		for (int j=1 ; j<=nheavyatoms; ++j) {
			conformation::Atom const &atm_i( rsd_i.atom(j) ); //(pose.residue(i).atom("CA"));

			chemical::AtomTypeSet const & atom_type_set( rsd_i.atom_type_set() );
			std::string elt_i = atom_type_set[ rsd_i.atom_type_index( j ) ].element();
			core::Real a = get_a( elt_i );

			// sidechain weight
			if (j > rsd_i.last_backbone_atom())
				a *= SC_scaling;

			// skip randomized residues
			if ( is_missing_density( atm_i.xyz() ) ) continue;

			// if this atom's weight is 0 continue
			if ( a < 1e-6 ) continue;

			cartX = atm_i.xyz() - getTransform();
			fracX = c2f*cartX;
			atm_idx[i][j][0] = pos_mod (fracX[0]*grid[0] - origin[0] + 1 , (double)grid[0]);
			atm_idx[i][j][1] = pos_mod (fracX[1]*grid[1] - origin[1] + 1 , (double)grid[1]);
			atm_idx[i][j][2] = pos_mod (fracX[2]*grid[2] - origin[2] + 1 , (double)grid[2]);


			for (int z=1; z<=density.u3(); ++z) {
				atm_j[2] = z;
				del_ij[2] = (atm_idx[i][j][2] - atm_j[2]) / grid[2];
				// wrap-around??
				if (del_ij[2] > 0.5) del_ij[2]-=1.0;
				if (del_ij[2] < -0.5) del_ij[2]+=1.0;

				del_ij[0] = del_ij[1] = 0.0;
				if ((f2c*del_ij).length_squared() > (ATOM_MASK+1)*(ATOM_MASK+1)) continue;

				for (int y=1; y<=density.u2(); ++y) {
					atm_j[1] = y;

					// early exit?
					del_ij[1] = (atm_idx[i][j][1] - atm_j[1]) / grid[1] ;
					// wrap-around??
					if (del_ij[1] > 0.5) del_ij[1]-=1.0;
					if (del_ij[1] < -0.5) del_ij[1]+=1.0;
					del_ij[0] = 0.0;
					if ((f2c*del_ij).length_squared() > (ATOM_MASK+1)*(ATOM_MASK+1)) continue;

					for (int x=1; x<=density.u1(); ++x) {
						atm_j[0] = x;

						// early exit?
						del_ij[0] = (atm_idx[i][j][0] - atm_j[0]) / grid[0];
						// wrap-around??
						if (del_ij[0] > 0.5) del_ij[0]-=1.0;
						if (del_ij[0] < -0.5) del_ij[0]+=1.0;

						numeric::xyzVector< core::Real > cart_del_ij = (f2c*del_ij);  // cartesian offset from (x,y,z) to atom_i
						core::Real d2 = (cart_del_ij).length_squared();

						if (d2 <= (ATOM_MASK+1)*(ATOM_MASK+1)) {
							// v1 ...
							core::Real atm = C*a*exp(-k*d2);
							// v2 ... is this ok for non-orthogonal???
							//core::Real atm = C*a*exp(-k)*grid[0]*grid[1]*grid[2]*
							//	( (errf( cart_del_ij[0]+0.5/grid[0] )-errf( cart_del_ij[0]-0.5/grid[0] ))*
							//	  (errf( cart_del_ij[1]+0.5/grid[1] )-errf( cart_del_ij[1]-0.5/grid[1] ))*
							//	  (errf( cart_del_ij[2]+0.5/grid[2] )-errf( cart_del_ij[2]-0.5/grid[2] ))  );
							core::Real inv_msk = 1/(1+exp( d2 - (ATOM_MASK)*(ATOM_MASK)  ));

							rho_calc(x,y,z) += atm;
							inv_rho_mask(x,y,z) *= (1 - inv_msk);

							if (cacheCCs) {
								int idx = (z-1)*density.u2()*density.u1() + (y-1)*density.u1() + x-1;
								rho_dx_pt[i][j].push_back  ( idx );
								rho_dx_mask[i][j].push_back( -inv_msk*cart_del_ij );
								rho_dx_atm[i][j].push_back ( atm*cart_del_ij );
							}
						}
					}
				}
			}
		}
	}

	//////////////////////////
	/// 2 COMPUTE SUMMARY STATISTICS
	core::Real sumC_i=0, sumO_i=0, sumCO_i=0, vol_i=0, CC_i=0;
 	core::Real sumO2_i=0.0, sumC2_i=0.0, varC_i=0, varO_i=0;
	core::Real clc_x, obs_x, eps_x;

	for (int x=0; x<density.u1()*density.u2()*density.u3(); ++x) {
		// fetch this point
		clc_x = rho_calc[x];
		obs_x = density[x];
		eps_x = 1-inv_rho_mask[x]; //1/(1+exp( (0.01-rho_calc(x,y,z)) * 1000 ));  // sigmoidal

		// SMOOTHED
		sumCO_i += eps_x*clc_x*obs_x;
		sumO_i  += eps_x*obs_x;
		sumO2_i += eps_x*obs_x*obs_x;
		sumC_i  += eps_x*clc_x;
		sumC2_i += eps_x*clc_x*clc_x;
		vol_i   += eps_x;
	}
	varC_i = (sumC2_i - sumC_i*sumC_i / vol_i );
	varO_i = (sumO2_i - sumO_i*sumO_i / vol_i ) ;
	if (varC_i == 0 || varO_i == 0)
		CC_i = 0;
	else
		CC_i = (sumCO_i - sumC_i*sumO_i/ vol_i) / sqrt( varC_i * varO_i );

	if (cacheCCs)
		CC_aacen = CC_i;


	///////////////////////////
	/// 3  CALCULATE SYMMETRIC ROTATION MATRICES + SYMM MAPPING at each level
	if (isSymm && remapSymm && cacheCCs) {
		compute_symm_rotations( pose, symmInfo );
	}


	///////////////////////////
	/// 4  CALCULATE PER-ATOM DERIVATIVES
	if (cacheCCs) {
		std::map< core::Size , numeric::xyzMatrix< core::Real > > symmRots;
		for (int i=1 ; i<=nres; ++i) {
			if (isSymm && !symmInfo->bb_is_independent(i) && !remapSymm) {  // should this be fa_...??
				continue; // only score the monomer
			}

			conformation::Residue const &rsd_i (pose.residue(i)); //( *reses[i] );

			if ( rsd_i.aa() == core::chemical::aa_vrt ) continue;
			if ( scoring_mask_.find(i) != scoring_mask_.end() ) continue;

			int nheavyatoms = atm_idx[i].size();
			dCCdxs_aacen[i].resize( nheavyatoms, numeric::xyzVector< core::Real >(0,0,0) );

			for (int j=1 ; j<=nheavyatoms; ++j) {
				numeric::xyzVector< core::Real > dVdx_ij(0,0,0), dOdx_ij(0,0,0), dO2dx_ij(0,0,0), dCOdx_ij(0,0,0), dC2dx_ij(0,0,0);

				conformation::Atom const &atm_i( rsd_i.atom(j) );
		 		if ( is_missing_density( atm_i.xyz() ) ) continue;

				utility::vector1< int > const &rho_dx_pt_ij   = rho_dx_pt[i][j];
				utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_mask_ij = rho_dx_mask[i][j];
				utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_atm_ij  = rho_dx_atm[i][j];

				int npoints = rho_dx_pt_ij.size();
				for (int n=1; n<=npoints; ++n) {
					const int x(rho_dx_pt_ij[n]);
					clc_x = rho_calc[x];
					obs_x = density[x];
					core::Real inv_eps_x = inv_rho_mask[x];

					numeric::xyzVector<double> del_mask = 2.0*inv_eps_x*rho_dx_mask_ij[n];
					dVdx_ij  += del_mask;
					dOdx_ij  += del_mask*obs_x;
					dO2dx_ij += del_mask*obs_x*obs_x;
					numeric::xyzVector<double> del_rhoc =-2.0*k*rho_dx_atm_ij[n];
					dCOdx_ij += del_rhoc*obs_x;
					dC2dx_ij += 2.0*del_rhoc*clc_x;
				}

				// finally compute dCC/dx_ij
				core::Real f = ( sumCO_i - sumC_i*sumO_i / vol_i );
				core::Real g = sqrt ( varO_i * varC_i );

				numeric::xyzVector<core::Real> fprime = dCOdx_ij - 1/(vol_i*vol_i) * ( dOdx_ij*sumC_i*vol_i - sumO_i*sumC_i*dVdx_ij);
				numeric::xyzVector<core::Real> gprime = 0.5 * (
						sqrt(varO_i)/sqrt(varC_i) * ( dC2dx_ij + ( sumC_i*sumC_i*dVdx_ij/(vol_i*vol_i) ) )  +
						sqrt(varC_i)/sqrt(varO_i) * ( dO2dx_ij - ( 1/(vol_i*vol_i) * ( 2*vol_i*sumO_i*dOdx_ij - sumO_i*sumO_i*dVdx_ij ) ) ) );

				dCCdxs_aacen[i][j] = (g*fprime - f*gprime) / (g*g);
			}
		}
	}
	// >> debugging <<
	//ElectronDensity(rho_calc, 2.0).writeMRC( "rho_calc.mrc" );
	//ElectronDensity(inv_rho_mask, 2.0).writeMRC( "inv_rho_mask.mrc" );
	//exit(1);

	//std::cerr << "ElectronDensity::matchPose() returning CC = " << CC_i << "   vol = " << vol_i << std::endl;

	return CC_i;
}


/// @details Computes the symmatric rotation matrices.  Stores mapping in 'symmap'.
void ElectronDensity::compute_symm_rotations( core::pose::Pose const &pose,
                                              const core::conformation::symmetry::SymmetryInfo *symmInfo /*=NULL*/ ) {
	// symmetry
	bool isSymm = (symmInfo != NULL);
	bool remapSymm = core::options::option[ core::options::OptionKeys::edensity::score_symm_complex ]();

	if (!isSymm || !remapSymm) return;

	int nsubunits = symmInfo->subunits();
	int nres_per = symmInfo->num_independent_residues();

	// mapping at the (non-virtual) leaves
	// MUST BE DONE EVERY TIME THE POSE CHANGES
	for (int subunit_i=1 ;  subunit_i<=nsubunits; ++subunit_i) {
		// symm
		int startRes = (subunit_i-1)*nres_per+1;
		int source_subunit = subunit_i;
		numeric::xyzMatrix< core::Real > R_i = numeric::xyzMatrix<core::Real>::rows(1,0,0, 0,1,0, 0,0,1);

		if (!symmInfo->bb_is_independent(startRes)) {
			core::Size sourceRes = symmInfo->bb_follows( startRes );
			source_subunit = symmInfo->subunit_index( sourceRes );
			R_i = numeric::alignVectorSets(
				pose.residue(startRes).atom(1).xyz() - pose.residue(startRes).atom(2).xyz(),
				pose.residue(startRes).atom(3).xyz() - pose.residue(startRes).atom(2).xyz(),
				pose.residue(sourceRes).atom(1).xyz() - pose.residue(sourceRes).atom(2).xyz(),
				pose.residue(sourceRes).atom(3).xyz() - pose.residue(sourceRes).atom(2).xyz());
		}
		// vrt of 0 => from the non-vrt point of view

		utility::vector1<int> mapping_i(nsubunits,0);
		mapping_i[ subunit_i ] = source_subunit;
		symmap[ -subunit_i ] = make_pair( mapping_i, R_i );
	}

	// at the internal VRTs traverse up the tree to get rotations
	// mapping only needs to be calculated once for each fold tree, however,
	//    Rs must still be percolated up the tree.
	// because of that, we'll just recompute the mapping each time;
	//    however, correcting this could save some running time.
	int nres = pose.total_residue();
	int vrtStart = nres_per*nsubunits;
	int nvrts = nres - vrtStart;
	utility::vector1<bool> vrts_mapped( nvrts, false );
	bool fully_mapped = false;
	while ( !fully_mapped ) {
		fully_mapped = true;
		for (int i=1 ; i<=nvrts; ++i) {
			int resid = i+vrtStart;
			if (vrts_mapped[i]) {
				continue;
			}
			utility::vector1< core::kinematics::Edge > edges_i = pose.fold_tree().get_outgoing_edges(resid);
			int nchildren = edges_i.size();

			if (nchildren == 0) {
				TR.Warning << "[ WARNING ] VRT (" << resid << ") at leaf ... ignoring" << std::endl;
				symmap[ resid ] = make_pair( utility::vector1<int>(0), numeric::xyzMatrix<core::Real>::rows(1,0,0, 0,1,0, 0,0,1) );
			} else if ( nchildren == 1) {
				// if 1 child, steal mapping from child
				int downstream = edges_i[1].stop();
				if ( downstream <= vrtStart || vrts_mapped[downstream-vrtStart] ) {
					if ( downstream <= vrtStart ) {   // do we jump to a subunit?
						symmap[ resid ] = symmap[ -symmInfo->subunit_index( downstream ) ];
					} else {
						symmap[ resid ] = symmap[ downstream ];
					}
					vrts_mapped[i] = true;
				} else {
					fully_mapped = false;
				}
			} else {
				// if >= 2 children, merge
				bool allChildrenMapped = true;
				for (int j=1; j<=nchildren; ++j) {
					int downstream = edges_i[j].stop();
					if (downstream <= vrtStart) {
						TR.Error << "[ Error ] VRT (" << resid << ") contains multiple jumps to subunits!  Exiting." << std::endl;
						exit(1);
					}
					allChildrenMapped &= vrts_mapped[ downstream-vrtStart ];
				}

				if ( allChildrenMapped ) {
					int firstChild = edges_i[1].stop();

					// use rot from 1st child
					numeric::xyzMatrix< core::Real > R_i = symmap[ firstChild ].second;

					// the first child keeps its mapping
					utility::vector1<int> mapping_i = symmap[ firstChild ].first;

					// for each subunit of each child
					for (int j=2; j<=nchildren; ++j) {
						utility::vector1<int> const &mapping_j = symmap[ edges_i[j].stop() ].first;
						for (int k=1; k<=nsubunits; ++k) {
							if (mapping_j[k] == 0) continue;   // subunit k is not under child j

							// find the subunit to which R_i maps k
							numeric::xyzMatrix< core::Real > R_ik = (symmap[ -k ].second) * R_i;  // R_i * R_k
							core::Real bestFit = 999999;
							int bestL=-1;
							for (int l=1; l<=nsubunits; ++l) {
								numeric::xyzMatrix< core::Real > R_diff = R_ik - (symmap[ -l ].second);
								core::Real thisErr = R_diff.col_x().length_squared() +  R_diff.col_y().length_squared() +  R_diff.col_z().length_squared();
								if (thisErr < bestFit) {
									bestFit = thisErr;
									bestL = l;
								}
							}
							//std::cerr << "at vrt (" << resid << ") ... mapping " << k << " to " << bestL << " / " << bestFit << std::endl;
							mapping_i[ k ] = bestL;
						}
					}

					symmap[ resid ] = make_pair( mapping_i, R_i );
					vrts_mapped[i] = true;
				} else {
					fully_mapped = false;
				}
			}
		}
	}
}


/// Match a residue to the density map, returning correlation coefficient between
///    map and residue.
/// Caches information about scoring, to be used in derivative computation
core::Real ElectronDensity::matchRes( int resid,
                                      core::conformation::Residue const &rsd,
                                      core::pose::Pose const &pose,
                                      const core::conformation::symmetry::SymmetryInfo *symmInfo /*=NULL*/,
                                      bool cacheCCs /* = false */) {
	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::matchRes called but no map is loaded!\n";
		return 0.0;
	}

	if ( scoring_mask_.find(resid) != scoring_mask_.end() ) return 0.0;
	if (!DensScoreInMinimizer) cacheCCs = false;

	// symmetry
	bool isSymm = (symmInfo != NULL);
	bool remapSymm = core::options::option[ core::options::OptionKeys::edensity::score_symm_complex ]();

	// symm
	if (isSymm && !symmInfo->bb_is_independent(resid) && !remapSymm) return 0.0; // only score monomer


	////
	//// grab atoms to be included in scoring
	////
	// only extend left/right to the next chainbreak
	int nres = (int)pose.total_residue();
	utility::vector1< conformation::Atom > contextAtoms;
	utility::vector1< core::Real > contextAtomAs;

	int WINDOW = core::options::option[ core::options::OptionKeys::edensity::sliding_window ]/2;
	int win_start=(int)resid-WINDOW, win_stop = (int)resid+WINDOW;
	for (int i=WINDOW-1; i>=0; --i) {
		if ( ((int)resid-i)>1 && pose.fold_tree().is_cutpoint( resid-i-1 ) ) win_start = resid-i;
		if ( ((int)resid+i)<=nres && pose.fold_tree().is_cutpoint( resid+i ) ) win_stop = resid+i;
	}

	// score struct in this window
	for (int i=std::max(1,win_start), i_end=std::min(win_stop,nres); i<=i_end; ++i) {
		if (i!=(int)resid) {
			core::conformation::Residue const &rsd_i( pose.residue(i) );
			if (!rsd_i.is_protein()) continue; // only protein residues

			chemical::AtomTypeSet const & atom_type_set( rsd_i.atom_type_set() );

			// only use BB atoms
			for ( core::Size j=1; j<=rsd_i.last_backbone_atom(); ++j ) {
				std::string elt_i = atom_type_set[ rsd_i.atom_type_index( j ) ].element();
				core::Real a = get_a( elt_i );
				contextAtoms.push_back( rsd_i.atom( j ) );
				contextAtomAs.push_back( a );
			}
		}
	}

	int nResAtms = rsd.nheavyatoms(), nContextAtms = contextAtoms.size();
	int nTotalAtms = nResAtms+nContextAtms;


	///////////////////////////
	/// 1 COMPUTE BOUNDING BOX
	numeric::xyzVector< core::Real > idxX_low(MAX_FLT,MAX_FLT,MAX_FLT), idxX_high(-MAX_FLT,-MAX_FLT,-MAX_FLT);
	utility::vector1< numeric::xyzVector< Real > > atmList;

	for ( int j=1; j<=nTotalAtms; ++j ) {
		conformation::Atom const &atom (j<=nResAtms? rsd.atom(j) : contextAtoms[j-nResAtms]);
		numeric::xyzVector< core::Real > cartX, idxX, fracX;

		if ( is_missing_density( atom.xyz() ) ) continue;

		// work in the "real" coordinate system, not the "derived" coords from the CoM
		cartX = atom.xyz() - getTransform();
		fracX = c2f*cartX;
		idxX = numeric::xyzVector< core::Real >( fracX[0]*grid[0] - origin[0] + 1 ,
		                                         fracX[1]*grid[1] - origin[1] + 1  ,
		                                         fracX[2]*grid[2] - origin[2] + 1  );
		atmList.push_back( idxX );

		fracX = c2f*( cartX-(ATOM_MASK+2) );
		idxX = numeric::xyzVector< core::Real >( fracX[0]*grid[0] - origin[0] + 1  ,
		                                         fracX[1]*grid[1] - origin[1] + 1  ,
		                                         fracX[2]*grid[2] - origin[2] + 1  );
		idxX_low[0] = std::min(idxX[0],idxX_low[0]);
		idxX_low[1] = std::min(idxX[1],idxX_low[1]);
		idxX_low[2] = std::min(idxX[2],idxX_low[2]);

		fracX = c2f*( cartX+(ATOM_MASK+2) );
		idxX = numeric::xyzVector< core::Real >( fracX[0]*grid[0] - origin[0] + 1  ,
		                                         fracX[1]*grid[1] - origin[1] + 1  ,
		                                         fracX[2]*grid[2] - origin[2] + 1  );
		idxX_high[0] = std::max(idxX[0],idxX_high[0]);
		idxX_high[1] = std::max(idxX[1],idxX_high[1]);
		idxX_high[2] = std::max(idxX[2],idxX_high[2]);
	}

	numeric::xyzVector< int > bbox_min, bbox_max, bbox_dims;
	bbox_min[0] = (int)floor(idxX_low[0])-1; bbox_max[0] = (int)ceil(idxX_high[0]); bbox_dims[0] = bbox_max[0]-bbox_min[0];
	bbox_min[1] = (int)floor(idxX_low[1])-1; bbox_max[1] = (int)ceil(idxX_high[1]); bbox_dims[1] = bbox_max[1]-bbox_min[1];
	bbox_min[2] = (int)floor(idxX_low[2])-1; bbox_max[2] = (int)ceil(idxX_high[2]); bbox_dims[2] = bbox_max[2]-bbox_min[2];

	ObjexxFCL::FArray3D< double >  rho_obs, rho_calc, inv_rho_mask;
	rho_obs.dimension(bbox_dims[0],bbox_dims[1],bbox_dims[2]);
	rho_calc.dimension(bbox_dims[0],bbox_dims[1],bbox_dims[2]);
	inv_rho_mask.dimension(bbox_dims[0],bbox_dims[1],bbox_dims[2]);

	for (int x=0; x<bbox_dims[0]*bbox_dims[1]*bbox_dims[2]; ++x) {
		rho_obs[x] = 0.0;
		rho_calc[x] = 0.0;
		inv_rho_mask[x] = 1.0;
	}

	utility::vector1< numeric::xyzVector<core::Real> >                     atm_idx(nResAtms);
	utility::vector1< utility::vector1< int > >                            rho_dx_pt(nResAtms);
	utility::vector1< utility::vector1< numeric::xyzVector<core::Real> > > rho_dx_mask(nResAtms), rho_dx_atm(nResAtms);
	                      // store individual atom's + atom mask's gradient

	///////////////////////////
	/// 2 COMPUTE RHO_C, MASK
	core::Real k=square(M_PI/reso);
	core::Real C=pow(k/M_PI,1.5);
	//core::Real a=6.0;  // call everything 'C' for now
	chemical::AtomTypeSet const & atom_type_set( rsd.atom_type_set() );
	core::Real SC_scaling = core::options::option[ core::options::OptionKeys::edensity::sc_scaling ]();

	core::Real clc_x, obs_x;
	int mapX,mapY,mapZ;

	for (int i=1; i<=(int)atmList.size(); ++i) {
		numeric::xyzVector< core::Real > & atm_i = atmList[i];
		atm_i[0] -= bbox_min[0];
		atm_i[1] -= bbox_min[1];
		atm_i[2] -= bbox_min[2];

		numeric::xyzVector< core::Real > atm_j, del_ij;

		core::Real a;
		if (i<=nResAtms) {
			std::string elt_i = atom_type_set[ rsd.atom_type_index( i ) ].element();
			a = get_a( elt_i );
			if (i > rsd.last_backbone_atom())
				a *= SC_scaling;
		} else {
			a = contextAtomAs[i-nResAtms];
		}

		for (int x=1; x<=bbox_dims[0]; ++x) {
			atm_j[0] = x;

			// early exit?
			del_ij[0] = min_mod( (atm_i[0] - atm_j[0]) / grid[0] , 1.0 );
			del_ij[1] = del_ij[2] = 0.0;
			if ((f2c*del_ij).length_squared() > ATOM_MASK*ATOM_MASK) continue;

			mapX = (x+bbox_min[0]) % grid[0];
			if (mapX <= 0) mapX += grid[0];
			if (mapX > density.u1()) continue;

			for (int y=1; y<=bbox_dims[1]; ++y) {
				atm_j[1] = y;

				// early exit?
				del_ij[1] = min_mod( (atm_i[1] - atm_j[1]) / grid[1] , 1.0);
				del_ij[2] = 0.0;
				if ((f2c*del_ij).length_squared() > ATOM_MASK*ATOM_MASK) continue;

				mapY = (y+bbox_min[1]) % grid[1];
				if (mapY <= 0) mapY += grid[1];
				if (mapY > density.u2()) continue;

				for (int z=1; z<=bbox_dims[2]; ++z) {
					atm_j[2] = z;
					del_ij[2] = min_mod( (atm_i[2] - atm_j[2]) / grid[2] , 1.0);

					mapZ = (z+bbox_min[2]) % grid[2];
					if (mapZ <= 0) mapZ += grid[2];
					if (mapZ > density.u3()) continue;

					numeric::xyzVector< core::Real > cart_del_ij = (f2c*del_ij);  // cartesian offset from atom_i to (x,y,z)
					core::Real d2 = (cart_del_ij).length_squared();
					if (d2 <= (ATOM_MASK+1)*(ATOM_MASK+1)) {
						core::Real atm = C*a*exp(-k*d2);
						core::Real inv_msk = 1/(1+exp( d2 - (ATOM_MASK-1)*(ATOM_MASK-1) ));

						rho_obs(x,y,z) = density(mapX,mapY,mapZ);  // we're making a copy of this ... is it wasteful?
						rho_calc(x,y,z) += atm;
						inv_rho_mask(x,y,z) *= (1 - inv_msk);

						if (cacheCCs && i<=nResAtms) {
							int idx = (z-1)*rho_calc.u2()*rho_calc.u1() + (y-1)*rho_calc.u1() + x-1;
							rho_dx_pt[i].push_back  ( idx );
							rho_dx_mask[i].push_back( -inv_msk*cart_del_ij );
							rho_dx_atm[i].push_back ( atm*cart_del_ij );
						}
					}
				}
			}
		}
	}

	//////////////////////////
	/// 2 COMPUTE SUMMARY STATISTICS
	core::Real sumC_i=0, sumO_i=0, sumCO_i=0, vol_i=0, CC_i=0;
 	core::Real sumO2_i=0.0, sumC2_i=0.0, varC_i=0, varO_i=0;

	for (int x=0; x<bbox_dims[0]*bbox_dims[1]*bbox_dims[2]; ++x) {
		// fetch this point
		clc_x = rho_calc[x];
		obs_x = rho_obs[x];

		core::Real wt = 1-inv_rho_mask[x];
		sumCO_i += wt*clc_x*obs_x;
		sumO_i  += wt*obs_x;
		sumO2_i += wt*obs_x*obs_x;
		sumC_i  += wt*clc_x;
		sumC2_i += wt*clc_x*clc_x;
		vol_i   += wt;
	}
	varC_i = (sumC2_i - sumC_i*sumC_i / vol_i );
	varO_i = (sumO2_i - sumO_i*sumO_i / vol_i ) ;
	if (varC_i == 0 || varO_i == 0)
		CC_i = 0;
	else
		CC_i = (sumCO_i - sumC_i*sumO_i/ vol_i) / sqrt( varC_i * varO_i );

	if (cacheCCs) {
		CCs[resid]     = CC_i;
	}

	///////////////////////////
	/// 3  CALCULATE PER-ATOM DERIVATIVES
	if (cacheCCs) {
		dCCdxs_res[resid].resize( nResAtms, numeric::xyzVector< core::Real >(0,0,0) );
		for (int j=1 ; j<=nResAtms; ++j) {
			utility::vector1< int > const &rho_dx_pt_ij   = rho_dx_pt[j];
			utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_mask_ij = rho_dx_mask[j];
			utility::vector1< numeric::xyzVector<core::Real> > const &rho_dx_atm_ij  = rho_dx_atm[j];

			numeric::xyzVector< core::Real > dVdx_ij(0,0,0), dOdx_ij(0,0,0), dO2dx_ij(0,0,0), dCOdx_ij(0,0,0), dC2dx_ij(0,0,0);

			int npoints = rho_dx_pt_ij.size();
			for (int n=1; n<=npoints; ++n) {
				const int x(rho_dx_pt_ij[n]);
				clc_x = rho_calc[x];
				obs_x = rho_obs[x];
				core::Real inv_eps_x = inv_rho_mask[x];

				numeric::xyzVector<double> del_mask = 2.0*inv_eps_x*rho_dx_mask_ij[n];
				dVdx_ij  += del_mask;
				dOdx_ij  += del_mask*obs_x;
				dO2dx_ij += del_mask*obs_x*obs_x;
				numeric::xyzVector<double> del_rhoc =-2.0*k*rho_dx_atm_ij[n];
				dCOdx_ij += del_rhoc*obs_x;
				dC2dx_ij += 2.0*del_rhoc*clc_x;
			}

			// finally compute dCC/dx_ij
			core::Real f = ( sumCO_i - sumC_i*sumO_i / vol_i );
			core::Real g = sqrt ( varO_i * varC_i );

			numeric::xyzVector<core::Real> fprime = dCOdx_ij - 1/(vol_i*vol_i) * ( dOdx_ij*sumC_i*vol_i - sumO_i*sumC_i*dVdx_ij);
			numeric::xyzVector<core::Real> gprime = 0.5 * (
					sqrt(varO_i)/sqrt(varC_i) * ( dC2dx_ij + ( sumC_i*sumC_i*dVdx_ij/(vol_i*vol_i) ) )  +
					sqrt(varC_i)/sqrt(varO_i) * ( dO2dx_ij - ( 1/(vol_i*vol_i) * ( 2*vol_i*sumO_i*dOdx_ij - sumO_i*sumO_i*dVdx_ij ) ) ) );

			dCCdxs_res[resid][j] = (g*fprime - f*gprime) / (g*g);
		}
	}

	return( CC_i );
}


//
//  Compute the gradient
//
void ElectronDensity::dCCdx_res(
				int atmid, int resid,
				numeric::xyzVector<core::Real> const &X,
        core::conformation::Residue const &rsd,
        core::pose::Pose const &pose,
				numeric::xyzVector<core::Real> &dCCdX )
{
	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::dCCdx_res called but no map is loaded!\n";
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		exit(1);
	}

	static bool warned=false;
	if (!DensScoreInMinimizer) {
		if (!warned)
			TR << "[ WARNING ] dCCdx_res called but DensityScoreInMinimizer = false ... returning 0" << std::endl;
		//warned = true;
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		return;
	}

	// if we didn't score this residue
	//      (because it was missing density or masked or a symm copy)
	// then don't compute its derivative
	if ( (int)dCCdxs_res[resid].size() < atmid ) {
		//std::cerr << "no " << resid << "." << atmid << std::endl;
 		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		return;
	}

	if (! ExactDerivatives ) {
		//std::cerr << "dCCdxs_res[" << resid << "][" << atmid << "] = " << dCCdxs_res[resid][atmid] << std::endl;
		dCCdX = dCCdxs_res[resid][atmid];
	} else {
		////////////////////////////////
		//////  debug : compare d_true to d_exact
		//
		core::conformation::Residue rsd_copy = rsd;

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0]+NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_px = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0]-NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_mx = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0],X[1]+NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_py = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0],X[1]-NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_my = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0],X[1],X[2]+NUM_DERIV_H_CEN ) );
		core::Real CC_pz = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		rsd_copy.atom( atmid ).xyz( numeric::xyzVector<core::Real>( X[0],X[1],X[2]-NUM_DERIV_H_CEN ) );
		core::Real CC_mz = getDensityMap().matchRes( resid, rsd_copy, pose, NULL, true );

		// rescore with orig pose
		getDensityMap().matchRes( resid, rsd, pose, NULL, true );

		dCCdX[0] = (CC_px-CC_mx)/(2*NUM_DERIV_H_CEN);
		dCCdX[1] = (CC_py-CC_my)/(2*NUM_DERIV_H_CEN);
		dCCdX[2] = (CC_pz-CC_mz)/(2*NUM_DERIV_H_CEN);

		//////
		//////  debug : compare d_true to d_exact
		numeric::xyzVector<core::Real> dCCdX1 = dCCdxs_res[resid][atmid];
		TR << "   " <<  dCCdX<< "  ;  " <<  dCCdX1 << std::endl;
	}
}


//
//  Compute the gradient
//
void ElectronDensity::dCCdx_cen( int resid,
                                 numeric::xyzVector<core::Real> const &X,
                                 core::pose::Pose const &pose,
                                 numeric::xyzVector<core::Real> &dCCdX ) {
	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::dCCdx_cen called but no map is loaded!\n";
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		exit(1);
	}
	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::dCCdx_cen called but no map is loaded!\n";
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		exit(1);
	}

	static bool warned=false;
	if (!DensScoreInMinimizer) {
		if (!warned)
			TR << "[ WARNING ] dCCdx_cen called but DensityScoreInMinimizer = false ... returning 0" << std::endl;
		//warned = true;
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		return;
	}

	if (! ExactDerivatives ) {
		dCCdX = dCCdxs_cen[resid];
	} else {
		//
		core::pose::Pose pose_copy = pose;
		id::AtomID id(2,resid);

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0]+NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_px = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0]-NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_mx = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1]+NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_py = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1]-NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_my = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1],X[2]+NUM_DERIV_H_CEN ) );
		core::Real CC_pz = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1],X[2]-NUM_DERIV_H_CEN ) );
		core::Real CC_mz = getDensityMap().matchPose( pose_copy , NULL, true );

		// rescore with orig pose
		getDensityMap().matchCentroidPose( pose , NULL, true );

		dCCdX[0] = (CC_px-CC_mx)/(2*NUM_DERIV_H_CEN);
		dCCdX[1] = (CC_py-CC_my)/(2*NUM_DERIV_H_CEN);
		dCCdX[2] = (CC_pz-CC_mz)/(2*NUM_DERIV_H_CEN);

		//////  debug : compare d_true to d_exact
		numeric::xyzVector<core::Real> dCCdX1 = dCCdxs_cen[resid];
		TR << "   " <<  dCCdX<< "  ;  " <<  dCCdX1 << std::endl;
	}
}


//
//  Compute the gradient
//
void ElectronDensity::dCCdx_aacen( int atmid, int resid,
                             numeric::xyzVector<core::Real> const &X,
                             core::pose::Pose const &pose,
                             numeric::xyzVector<core::Real> &dCCdX ) {
	// make sure map is loaded
	if (!isLoaded) {
		TR << "[ ERROR ]  ElectronDensity::dCCdx_aacen called but no map is loaded!\n";
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		exit(1);
	}

	static bool warned=false;
	if (!DensScoreInMinimizer) {
		if (!warned)
			TR << "[ WARNING ] dCCdx_aacen called but DensityScoreInMinimizer = false ... returning 0" << std::endl;
		//warned = true;
		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		return;
	}

	// if we didn't score this residue
	//      (because it was missing density or masked or a symm copy)
	// then don't compute its derivative
	if ( (int)dCCdxs_aacen[resid].size() < atmid ) {
 		dCCdX = numeric::xyzVector<core::Real>(0.0,0.0,0.0);
		return;
	}

	if (! ExactDerivatives ) {
		dCCdX = dCCdxs_aacen[resid][atmid];
	} else {
		//
		core::pose::Pose pose_copy = pose;
		id::AtomID id(atmid,resid);

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0]+NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_px = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0]-NUM_DERIV_H_CEN,X[1],X[2] ) );
		core::Real CC_mx = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1]+NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_py = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1]-NUM_DERIV_H_CEN,X[2] ) );
		core::Real CC_my = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1],X[2]+NUM_DERIV_H_CEN ) );
		core::Real CC_pz = getDensityMap().matchPose( pose_copy , NULL, true );

		pose_copy.set_xyz(id, numeric::xyzVector<core::Real>( X[0],X[1],X[2]-NUM_DERIV_H_CEN ) );
		core::Real CC_mz = getDensityMap().matchPose( pose_copy , NULL, true );

		// rescore with orig pose
		getDensityMap().matchPose( pose , NULL, true );

		dCCdX[0] = (CC_px-CC_mx)/(2*NUM_DERIV_H_CEN);
		dCCdX[1] = (CC_py-CC_my)/(2*NUM_DERIV_H_CEN);
		dCCdX[2] = (CC_pz-CC_mz)/(2*NUM_DERIV_H_CEN);

		//////
		//////  debug : compare d_true to d_exact
		numeric::xyzVector<core::Real> dCCdX1 = dCCdxs_aacen[resid][atmid];
		TR << "   " <<  dCCdX<< "  ;  " <<  dCCdX1 << std::endl;
	}
}


//
// ElectronDensity::readMRC(std::string mapfile)
//      read an MRC/CCP4 density map
bool ElectronDensity::readMRCandResize( std::string mapfile, core::Real reso , core::Real gridSpacing ) {
	std::fstream mapin(mapfile.c_str() , std::ios::binary | std::ios::in );

	// set map resolution
	this->reso = reso;
	if ( core::options::option[ core::options::OptionKeys::edensity::atom_mask ]() <= 0 )
		ATOM_MASK = 1.25*reso; //  ??? 0.45*r ~ sqrt(2)*stdev = 1/(sqrt(2)*pi)
	else
		ATOM_MASK = core::options::option[ core::options::OptionKeys::edensity::atom_mask ]();

	if ( core::options::option[ core::options::OptionKeys::edensity::ca_mask ]() <= 0 )
		CA_MASK = 1.25*reso;
	else
		CA_MASK = core::options::option[ core::options::OptionKeys::edensity::ca_mask ]();

	char mapString[4], symData[81];

	int  crs2xyz[3], extent[3], mode, symBytes, grid[3], origin[3];
	int  xyz2crs[3], vol_xsize, vol_ysize, vol_zsize;
	int xIndex, yIndex, zIndex, vol_xySize, coord[3];
	long dataOffset, filesize;
	float *rowdata;

	bool swap=false;

	if (!mapin) {
		TR << "[ ERROR ]  Error opening MRC map " << mapfile << ".  Not loading map." << std::endl;
		return false;
	}

	if (    !mapin.read(reinterpret_cast <char*> (extent), 3*sizeof(int))
	     || !mapin.read(reinterpret_cast <char*> (&mode), 1*sizeof(int))
	     || !mapin.read(reinterpret_cast <char*> (&origin[0]), 3*sizeof(int))
	     || !mapin.read(reinterpret_cast <char*> (&grid[0]), 3*sizeof(int))
	     || !mapin.read(reinterpret_cast <char*> (&cellDimensions[0]), 3*sizeof(float))
	     || !mapin.read(reinterpret_cast <char*> (&cellAngles[0]), 3*sizeof(float))
	     || !mapin.read(reinterpret_cast <char*> (crs2xyz), 3*sizeof(int)) )  {
		TR << "[ ERROR ]   Improperly formatted line in MRC map.  Not loading map." << std::endl;
		return false;
	}

	// Check the number of bytes used for storing symmetry operators
	mapin.seekg(92, std::ios::beg);
	if ( !mapin.read(reinterpret_cast <char*> (&symBytes), 1*sizeof(int)) ) {
		TR << "[ ERROR ]   Failed reading symmetry bytes record.  Not loading map." << "\n";
		return false;
	}

	// Check for the string "MAP" at byte 208, indicating a CCP4 file.
	mapin.seekg(208, std::ios::beg);
	mapString[3] = '\0';
	if ( !mapin.read(mapString, 3) || (std::string(mapString) != "MAP")) {
		TR << "[ ERROR ]  'MAP' string missing, not a valid MRC map.  Not loading map." << std::endl;
		return false;
	}
	// Check the file endianness
	if (mode != 2) {
		swap4_aligned(&mode, 1);
		if (mode != 2) {
			TR << "[ ERROR ]   Non-real (32-bit float) data types are unsupported.  Not loading map." << std::endl;
			return false;
		} else {
			swap = true; // enable byte swapping
		}
	}

	// Swap all the information obtained from the header
	if (swap) {
		swap4_aligned(extent, 3);
		swap4_aligned(&origin[0], 3);
		swap4_aligned(&grid[0], 3);
		swap4_aligned(&cellDimensions[0], 3);
		swap4_aligned(&cellAngles[0], 3);
		swap4_aligned(crs2xyz, 3);
		swap4_aligned(&symBytes, 1);
	}

	TR << " Setting resolution to " << reso << "A" << std::endl;
	TR << "          atom mask to " << ATOM_MASK << "A" << std::endl;
	TR << "            CA mask to " << CA_MASK << "A" << std::endl;
	TR << " Read density map'" << mapfile << "'" << std::endl;
	TR << "     extent: " << extent[0] << " x " << extent[1] << " x " << extent[2] << std::endl;
	TR << "     origin: " << origin[0] << " x " << origin[1] << " x " << origin[2] << std::endl;
	TR << "       grid: " << grid[0] << " x " << grid[1] << " x " << grid[2] << std::endl;
	TR << "    celldim: " << cellDimensions[0] << " x " << cellDimensions[1] << " x " << cellDimensions[2] << std::endl;
	TR << " cellangles: " << cellAngles[0] << " x " << cellAngles[1] << " x " << cellAngles[2] << std::endl;
	TR << "    crs2xyz: " << crs2xyz[0] << " x " << crs2xyz[1] << " x " << crs2xyz[2] << std::endl;
	TR << "   symBytes: " << symBytes << "\n";

	// Check the dataOffset: this fixes the problem caused by files claiming
	// to have symmetry records when they do not.
	mapin.seekg(0, std::ios::end);
	filesize = mapin.tellg();
	dataOffset = filesize - 4*(extent[0]*extent[1]*extent[2]);
	if (dataOffset != (CCP4HDSIZE + symBytes)) {
		if (dataOffset == CCP4HDSIZE) {
			// Bogus symmetry record information
			TR << "[ WARNING ] File contains bogus symmetry record.  Continuing." << std::endl;
			symBytes = 0;
		} else if (dataOffset < CCP4HDSIZE) {
			TR << "[ ERROR ] File appears truncated and doesn't match header.  Not loading map." << std::endl;
			return false;
		} else if ((dataOffset > CCP4HDSIZE) && (dataOffset < (1024*1024))) {
				// Fix for loading SPIDER files which are larger than usual
		 		// In this specific case, we must absolutely trust the symBytes record
				dataOffset = CCP4HDSIZE + symBytes;
				TR << "[ WARNING ]  File is larger than expected and doesn't match header.  Reading anyway." << std::endl;
		} else {
			TR << "[ ERROR ] File is MUCH larger than expected and doesn't match header.  Not loading map." << std::endl;
			return false;
		}
	}

	// Read symmetry records -- organized as 80-byte lines of text.
	utility::vector1< std::string > symList;
	symData[80]='\0';
	if (symBytes != 0) {
		TR << "Symmetry records found:" << std::endl;
		mapin.seekg(CCP4HDSIZE, std::ios::beg);
		for (int i = 0; i < symBytes/80; i++) {
			mapin.read(symData, 80);
			symList.push_back(symData);
			TR << symData << std::endl;
		}
	} else {
		// no symm info; assume P 1
		symList.push_back("X,  Y,  Z");
	}
	initializeSymmOps( symList );

	// check extent and grid interval counts
	if (grid[0] == 0 && extent[0] > 0) {
		grid[0] = extent[0] - 1;
		TR << "[ WARNING ] Fixed X interval count.  Continuing." << std::endl;
  	}
	if (grid[1] == 0 && extent[1] > 0) {
		grid[1] = extent[1] - 1;
		TR << "[ WARNING ]  Fixed Y interval count.  Continuing." << std::endl;
  	}
	if (grid[2] == 0 && extent[2] > 0) {
		grid[2] = extent[2] - 1;
		TR << "[ WARNING ]  Fixed Z interval count.  Continuing." << std::endl;
	}

	// Mapping between CCP4 column, row, section and Cartesian x, y, z.
	if (crs2xyz[0] == 0 && crs2xyz[1] == 0 && crs2xyz[2] == 0) {
		TR << "[ WARNING ]  All crs2xyz records are zero." << std::endl;
		TR << "[ WARNING ]  Setting crs2xyz to 1, 2, 3 and continuing." << std::endl;
		crs2xyz[0] = 1; crs2xyz[1] = 2; crs2xyz[2] = 3;
	}

	xyz2crs[crs2xyz[0]-1] = 0; xyz2crs[crs2xyz[1]-1] = 1; xyz2crs[crs2xyz[2]-1] = 2;
	xIndex = xyz2crs[0]; yIndex = xyz2crs[1]; zIndex = xyz2crs[2];

	vol_xsize = extent[xIndex];
	vol_ysize = extent[yIndex];
	vol_zsize = extent[zIndex];
	vol_xySize = vol_xsize * vol_ysize;

	// coord = <col, row, sec>
	// extent = <colSize, rowSize, secSize>
	rowdata = new float[extent[0]];
	mapin.seekg(dataOffset, std::ios::beg);

   	// 'alloc' changes ordering of "extent"
	density.dimension( vol_xsize,vol_ysize,vol_zsize );

	for (coord[2] = 1; coord[2] <= extent[2]; coord[2]++) {
		for (coord[1] = 1; coord[1] <= extent[1]; coord[1]++) {
			// Read an entire row of data from the file, then write it into the
			// datablock with the correct slice ordering.
			if ( mapin.eof() ) {
				TR << "[ ERROR ] Unexpected end-of-file. Not loading map." << std::endl;
				return false;
			}
			if ( mapin.fail() ) {
				TR << "[ ERROR ] Problem reading the file. Not loading map." << std::endl;
				return false;
			}
			if ( !mapin.read( reinterpret_cast< char* >(rowdata), sizeof(float)*extent[0]) ) {
				TR << "[ ERROR ] Error reading data row. Not loading map." << std::endl;
				return false;
			}

			for (coord[0] = 1; coord[0] <= extent[0]; coord[0]++) {
				density( coord[xyz2crs[0]], coord[xyz2crs[1]], coord[xyz2crs[2]]) = rowdata[coord[0]-1];
			}
		}
	}

	if (swap == 1)
		swap4_aligned( &density[0], vol_xySize * vol_zsize);
	delete [] rowdata;

 	mapin.close();

	this->origin[0] = origin[xyz2crs[0]];
	this->origin[1] = origin[xyz2crs[1]];
	this->origin[2] = origin[xyz2crs[2]];
	this->efforigin = this->origin;

	// grid doesnt seemed to get remapped in ccp4 maps
	this->grid[0] = grid[0];
	this->grid[1] = grid[1];
	this->grid[2] = grid[2];

	// we're done!
	isLoaded = true;

	///////////////////////////////////
	/// POST PROCESSING
	// expand to unit cell
	this->computeCrystParams();
	this->expandToUnitCell();

	// resample the map
	if (gridSpacing > 0) this->resize( gridSpacing );

	// post-processing
	this->computeStats();
	this->computeGradients();

	return true;
}


// read a list of symm ops (as in ccp4 map header)
void ElectronDensity::initializeSymmOps( utility::vector1< std::string > const & symList ) {
		using core::kinematics::RT;

		symmOps.clear();

		if ( symList.size() == 0 ) { // no symminfo in header, assume P 1
			symmOps.push_back( RT( numeric::xyzMatrix< core::Real >::identity(),
			                       numeric::xyzVector< core::Real >(0.0,0.0,0.0) ) );
		}

		for (int i=1; i<=(int)symList.size(); ++i) {
			std::string line = symList[i];
			std::vector< std::string > rows = utility::string_split(line, ',');
			if ( rows.size() != 3 ) {
				TR.Error << "[ ERROR ] invalid symmop in map file" << std::endl;
				TR.Error << line << std::endl;
				TR.Error << "Setting symmetry to P1 and continuing!" << line << std::endl;

				// should we throw an exception here????  nah, just set symm to P1 and continue
				symmOps.clear();
				symmOps.push_back( RT( numeric::xyzMatrix< core::Real >::identity(),
															 numeric::xyzVector< core::Real >(0.0,0.0,0.0) ) );

				return;
			}

			// _REALLY_ simple parser
			numeric::xyzMatrix< core::Real > rot(0);
			numeric::xyzVector< core::Real > trans(0,0,0);
			int k;
			for (int j=0; j<3; ++j) {
				k = rows[j].find('/');
				if (k != (int)std::string::npos) {
					// numerator
					int startNum = rows[j].find_last_not_of("0123456789",k-1) + 1;
					int startDenom = k+1;
					trans[j] = std::atof( &rows[j][startNum]) / std::atof( &rows[j][startDenom]);
				} else {
					trans[j] = 0;
				}

				if (rows[j].find("-X") != std::string::npos)
					rot(j+1,1) = -1;
				else if (rows[j].find("X") != std::string::npos)
					rot(j+1,1) = 1;

				if (rows[j].find("-Y") != std::string::npos)
					rot(j+1,2) = -1;
				else if (rows[j].find("Y") != std::string::npos)
					rot(j+1,2) = 1;

				if (rows[j].find("-Z") != std::string::npos)
					rot(j+1,3) = -1;
				else if (rows[j].find("Z") != std::string::npos)
					rot(j+1,3) = 1;
			}

			symmOps.push_back( RT( rot, trans ) );
		}
		return;
}

// expand density to cover unit cell
// maintain origin
void ElectronDensity::expandToUnitCell() {
	numeric::xyzVector< int > extent( density.u1(), density.u2(), density.u3() );

	// if it already covers unit cell do nothing
	if ( grid[0] == extent[0] && grid[1] == extent[1] && grid[2] == extent[2] )
		return;

	ObjexxFCL::FArray3D< float > newDensity( grid[0],grid[1],grid[2], 0.0 );

	// copy the block
	int limX=std::min(extent[0],grid[0]),
	    limY=std::min(extent[1],grid[1]),
	    limZ=std::min(extent[2],grid[2]);
	for (int x=1; x<=limX; ++x)
	for (int y=1; y<=limY; ++y)
	for (int z=1; z<=limZ; ++z) {
		newDensity( x,y,z ) = density( x,y,z );
	}

	// apply symmetry
	// why backwards? it is a mystery
	for (int x=grid[0]; x>=1; --x)
	for (int y=grid[1]; y>=1; --y)
	for (int z=grid[2]; z>=1; --z) {
		if (x <= limX && y <= limY && z <= limZ)
			continue;

		numeric::xyzVector<core::Real> fracX(
			( (core::Real)x + origin[0] - 1 ) / grid[0],
			( (core::Real)y + origin[1] - 1 ) / grid[1],
			( (core::Real)z + origin[2] - 1 ) / grid[2] );

		for (int symm_idx=1; symm_idx<=(int)symmOps.size(); symm_idx++) {
			numeric::xyzVector<core::Real> SfracX =
				symmOps[symm_idx].get_rotation() * fracX +  symmOps[symm_idx].get_translation();

			// indices of symm copy
			int Sx = pos_mod((int)floor(SfracX[0]*grid[0]+0.5 - origin[0]) , grid[0]) + 1;
			int Sy = pos_mod((int)floor(SfracX[1]*grid[1]+0.5 - origin[1]) , grid[1]) + 1 ;
			int Sz = pos_mod((int)floor(SfracX[2]*grid[2]+0.5 - origin[2]) , grid[2]) + 1 ;

			if (Sx <= limX && Sy <= limY && Sz <= limZ) {
				newDensity( x,y,z ) = density(Sx,Sy,Sz);
			}
		}
	}

	//ElectronDensity(density).writeMRC( "before.mrc" );
	//ElectronDensity(newDensity).writeMRC( "after.mrc" );
	//exit(1);

	// new map!
	density = newDensity;
}


//
//  DEBUGGING: write _this_ map in MATLAB v5 format
bool ElectronDensity::writeMAT(std::string filestem) {
	using namespace std;

	string outfile = filestem + ".mat";
	numeric::xyzVector<int> dims( density.u1(), density.u2(), density.u3() );

	ofstream out;
	out.open(outfile.c_str(), ios::out);

	// write file header
	short buff_s;
	long buff_l;
	out.write(
		"MATLAB 5.0 MAT-file, Platform: GLNX86   "
		"                                        "
		"                                        "
		"    ", 124);
	buff_s=0x0100;            // endian-ness test
	out.write((char*)&buff_s, 2);
	buff_s=0x4D49;            // code(?)
	out.write((char*)&buff_s, 2);

	// write data block header
	buff_l = 0x0000000E;      // data type (array)
	out.write((char*)&buff_l, 4);
	buff_l = 4*dims[0]*dims[1]*dims[2] + 64;      // number of bytes
	out.write((char*)&buff_l, 4);

	// write array header
	buff_l = 0x00000006;      // array data type (single)
	out.write((char*)&buff_l, 4);
	buff_l = 0x00000008;      // # bytes in array data block
	out.write((char*)&buff_l, 4);
	buff_l = 0x00000007;      // global/logical flags
	out.write((char*)&buff_l, 4);
	buff_l = 0x00000000;      // padding
	out.write((char*)&buff_l, 4);

	// write dims array
	buff_l = 0x00000005;      // dims data type (int32)
	out.write((char*)&buff_l, 4);
	buff_l = 0x0000000C;      // # bytes in dims data block
	out.write((char*)&buff_l, 4);
	out.write((char*)&dims[0], 12);
	buff_l = 0x00000000;      // padding
	out.write((char*)&buff_l, 4);

	// write name array
	buff_l=0x0001;            // name data type (int8)
	out.write((char*)&buff_l, 4);
	buff_l=0x0005;            // # bytes in name
	out.write((char*)&buff_l, 4);
	out.write("match   ", 8); // pad with spaces so next write is word-aligned

	// write array
	buff_l = 0x00000007;      // array data type (single)
	out.write((char*)&buff_l, 4);
	buff_l = 4*dims[0]*dims[1]*dims[2];
	out.write((char*)&buff_l, 4);

	for (int k=1; k<=dims[2]; k++) {
		for (int j=1; j<=dims[1]; j++) {
			for (int i=1; i<=dims[0]; i++) {
				float buff_f = (float) density(i,j,k);
				out.write((char*)&buff_f, 4);
			}
		}
	}

	out.close();

	return true;
}


void ElectronDensity::showCachedScores( utility::vector1< int > const &reses ) {
	for (int i=1; i<=(int)reses.size(); ++i ) {
		TR << "   " << reses[i] << ": " << CCs[reses[i]] << std::endl;
	}
}

//
//  DEBUGGING: write _this_ map in MRC format
bool ElectronDensity::writeMRC(std::string mapfilename) {
	std::fstream outx( (mapfilename).c_str() , std::ios::binary | std::ios::out );

	float buff_f;
	int buff_i;
	float buff_vf[3];
	int buff_vi[3];
	int symBytes = 0;

	if (!outx ) {
		TR.Error << "[ ERROR ]  Error opening MRC map for writing." << std::endl;
		return false;
	}

	// extent
	buff_vi[0] = density.u1(); buff_vi[1] = density.u2(); buff_vi[2] = density.u3();
	outx.write(reinterpret_cast <char*>(buff_vi), sizeof(int)*3);

	// mode
	buff_i = 2;
	outx.write(reinterpret_cast <char*>(&buff_i), sizeof(int));

	// origin
 	outx.write(reinterpret_cast <char*>(&origin[0]), sizeof(int)*3);

 	// grid
 	outx.write(reinterpret_cast <char*>(&grid[0]), sizeof(int)*3);

	// cell params
	outx.write(reinterpret_cast <char*>(&cellDimensions), sizeof(float)*3);
	outx.write(reinterpret_cast <char*>(&cellAngles), sizeof(float)*3);

	// crs2xyz
	buff_vi[0] = 1; buff_vi[1] = 2; buff_vi[2] = 3;
	outx.write(reinterpret_cast <char*>(buff_vi), sizeof(int)*3);

	// min, max, mean dens
	buff_vf[0] = -100.0; buff_vf[1] = 100.0; buff_vf[2] = 0.0;
	outx.write(reinterpret_cast <char*>(buff_vf), sizeof(float)*3);

	// 4 bytes junk
	buff_i = 0;
	outx.write(reinterpret_cast <char*>(&buff_i), sizeof(int));

	// symmops (to do!)
	outx.write(reinterpret_cast <char*>(&symBytes), sizeof(int));

	// 112 bytes junk
	buff_i = 0;
	for (int i=0; i<28; ++i) {
		outx.write(reinterpret_cast <char*>(&buff_i), sizeof(int));
	}

	// Write "MAP" at byte 208, indicating a CCP4 file.
	char buff_s[80]; strcpy(buff_s, "MAP DD");
	outx.write(reinterpret_cast <char*>(buff_s), 8);

	// fill remainder of head with junk
	int nJunkWords = (CCP4HDSIZE - 216) /4;
	buff_i = 0;
	for (int i=0; i<nJunkWords; ++i) {
		outx.write(reinterpret_cast <char*>(&buff_i), sizeof(int));
	}

	// data
	int coord[3];
	for (coord[2] = 1; coord[2] <= density.u3(); coord[2]++) {
		for (coord[1] = 1; coord[1] <= density.u2(); coord[1]++) {
			for (coord[0] = 1; coord[0] <= density.u1(); coord[0]++) {
				buff_f = (float) density(coord[0],coord[1],coord[2]);
				outx.write(reinterpret_cast <char*>(&buff_f), sizeof(float));
			}
		}
	}

	return true;
}


//
// resize a map (using FFT-interpolation)
void ElectronDensity::resize( core::Real approxGridSpacing ) {
	// potentially expand map to cover entire unit cell
	if ( grid[0] != density.u1() || grid[1] != density.u2() || grid[2] != density.u3() ){
		TR << "[ ERROR ] resize() not supported for maps not covering the entire unit cell."<< std::endl;
		TR << "   " << grid[0] << " != " << density.u1()
		   << " || " << grid[1] << " != " << density.u2()
		   << " || " << grid[2] << " != " << density.u3() << std::endl;
		exit(1);
	}

	// compute new dimensions & origin
	numeric::xyzVector<int> newDims,  newGrid;
	numeric::xyzVector<double> newOri;
	newDims[0] = (int)floor( cellDimensions[0] / approxGridSpacing + 0.5);
	newDims[1] = (int)floor( cellDimensions[1] / approxGridSpacing + 0.5);
	newDims[2] = (int)floor( cellDimensions[2] / approxGridSpacing + 0.5);
	newOri[0] = newDims[0]*origin[0] / ((core::Real)grid[0]);
	newOri[1] = newDims[1]*origin[1] / ((core::Real)grid[1]);
	newOri[2] = newDims[2]*origin[2] / ((core::Real)grid[2]);
	newGrid = newDims;

	ObjexxFCL::FArray3D< std::complex<double> > newDensity;
	newDensity.dimension( newDims[0], newDims[1], newDims[2] );
	TR << "Resizing " << density.u1() << "x" << density.u2() << "x" << density.u3() << " to "
	                  << newDensity.u1() << "x" << newDensity.u2() << "x" << newDensity.u3() << std::endl;

	// convert map to complex<double>
	ObjexxFCL::FArray3D< std::complex<double> > Foldmap, Fnewmap;
	Fnewmap.dimension( newDims[0], newDims[1], newDims[2] );

	// fft
	ObjexxFCL::FArray3D< std::complex<double> > cplx_density;
	cplx_density.dimension( density.u1() , density.u2() , density.u3() );
	for (int i=0; i<density.u1()*density.u2()*density.u3(); ++i) cplx_density[i] = (double)density[i];
	numeric::fourier::fft3(cplx_density, Foldmap);

	// reshape (handles both shrinking and growing in each dimension)
	for (int i=0; i<Fnewmap.u1()*Fnewmap.u2()*Fnewmap.u3(); ++i) Fnewmap[i] = std::complex<double>(0,0);

	numeric::xyzVector<int> nyq( std::min(Foldmap.u1(), Fnewmap.u1())/2,
	                             std::min(Foldmap.u2(), Fnewmap.u2())/2,
	                             std::min(Foldmap.u3(), Fnewmap.u3())/2 );
	numeric::xyzVector<int> nyqplus1_old( std::max(Foldmap.u1() - (std::min(Foldmap.u1(),Fnewmap.u1())-nyq[0]) + 1 , nyq[0]+1) ,
	                                      std::max(Foldmap.u2() - (std::min(Foldmap.u2(),Fnewmap.u2())-nyq[1]) + 1 , nyq[1]+1) ,
	                                      std::max(Foldmap.u3() - (std::min(Foldmap.u3(),Fnewmap.u3())-nyq[2]) + 1 , nyq[2]+1) );
	numeric::xyzVector<int> nyqplus1_new( std::max(Fnewmap.u1() - (std::min(Foldmap.u1(),Fnewmap.u1())-nyq[0]) + 1 , nyq[0]+1) ,
	                                      std::max(Fnewmap.u2() - (std::min(Foldmap.u2(),Fnewmap.u2())-nyq[1]) + 1 , nyq[1]+1) ,
	                                      std::max(Fnewmap.u3() - (std::min(Foldmap.u3(),Fnewmap.u3())-nyq[2]) + 1 , nyq[2]+1) );

	for (int i=1; i<=Fnewmap.u1(); i++)
	for (int j=1; j<=Fnewmap.u2(); j++)
	for (int k=1; k<=Fnewmap.u3(); k++) {
		if (i-1<=nyq[0]) {
			if (j-1<=nyq[1]) {
				if (k-1<=nyq[2])
					Fnewmap(i,j,k) = Foldmap(i, j, k);
				else if (k-1>=nyqplus1_new[2])
					Fnewmap(i,j,k) = Foldmap(i, j, k-nyqplus1_new[2]+nyqplus1_old[2]);

			} else if (j-1>=nyqplus1_new[1]) {
				if (k-1<=nyq[2])
					Fnewmap(i,j,k) = Foldmap(i, j-nyqplus1_new[1]+nyqplus1_old[1], k);
				else if (k-1>=nyqplus1_new[2])
					Fnewmap(i,j,k) = Foldmap(i, j-nyqplus1_new[1]+nyqplus1_old[1], k-nyqplus1_new[2]+nyqplus1_old[2]);
			}
		} else if (i-1>=nyqplus1_new[0]) {
			if (j-1<=nyq[1]) {
				if (k-1<=nyq[2])
					Fnewmap(i,j,k) = Foldmap(i-nyqplus1_new[0]+nyqplus1_old[0],j,k);
				else if (k-1>=nyqplus1_new[2])
					Fnewmap(i,j,k) = Foldmap(i-nyqplus1_new[0]+nyqplus1_old[0],j, k-nyqplus1_new[2]+nyqplus1_old[2]);

			} else if (j-1>=nyqplus1_new[1]){
				if (k-1<=nyq[2])
					Fnewmap(i,j,k) = Foldmap(i-nyqplus1_new[0]+nyqplus1_old[0],j-nyqplus1_new[1]+nyqplus1_old[1],k);
				else if (k-1>=nyqplus1_new[2])
					Fnewmap(i,j,k) = Foldmap(i-nyqplus1_new[0]+nyqplus1_old[0],
					                         j-nyqplus1_new[1]+nyqplus1_old[1],
					                         k-nyqplus1_new[2]+nyqplus1_old[2]);
			}
		}
	}

	// ifft
	numeric::fourier::ifft3(Fnewmap, newDensity);

	// update density
	density.dimension( newDims[0], newDims[1], newDims[2] );
	for (int i=0; i< newDims[0]*newDims[1]*newDims[2] ; ++i)
		density[i] = (float)newDensity[i].real();
	grid = newGrid;
	efforigin = origin = newOri;

	TR << " new extent: " << density.u1() << " x " << density.u2() << " x " << density.u3() << std::endl;
	TR << " new origin: " << origin[0] << " x " << origin[1] << " x " << origin[2] << std::endl;
	TR << "   new grid: " << grid[0] << " x " << grid[1] << " x " << grid[2] << std::endl;

	// recompute stats ??
}


//
// void ElectronDensity::computeGradients()
//     this is only used for visualization so is probably a huge waste of memory and time.  meh.
void ElectronDensity::computeGradients() {
	numeric::xyzVector<int> dims( density.u1(), density.u2(), density.u3() );

	// Allocate arrays
	ObjexxFCL::FArray3D< float > grad_x, grad_y, grad_z;
	grad_x.dimension( dims[0] , dims[1] , dims[2] ); grad_x = 0;
	grad_y.dimension( dims[0] , dims[1] , dims[2] ); grad_y = 0;
	grad_z.dimension( dims[0] , dims[1] , dims[2] ); grad_z = 0;

	for (int x=2; x<dims[0]; ++x) {
		for (int y=2; y<dims[1]; ++y) {
			for (int z=2; z<dims[2]; ++z) {
				grad_x(x,y,z) = 0.5*(density(x+1,y,z) - density(x-1,y,z));
				grad_x(x,y,z) = 0.125*(density(x+1,y+1,z) - density(x-1,y+1,z));
				grad_x(x,y,z) = 0.125*(density(x+1,y-1,z) - density(x-1,y-1,z));
				grad_x(x,y,z) = 0.125*(density(x+1,y,z+1) - density(x-1,y,z+1));
				grad_x(x,y,z) = 0.125*(density(x+1,y,z-1) - density(x-1,y,z-1));

				grad_y(x,y,z) = 0.5*(density(x,y+1,z) - density(x,y-1,z));
				grad_y(x,y,z) = 0.125*(density(x+1,y+1,z) - density(x+1,y-1,z));
				grad_y(x,y,z) = 0.125*(density(x-1,y+1,z) - density(x-1,y-1,z));
				grad_y(x,y,z) = 0.125*(density(x,y+1,z+1) - density(x,y-1,z+1));
				grad_y(x,y,z) = 0.125*(density(x,y+1,z-1) - density(x,y-1,z-1));

				grad_z(x,y,z) = 0.5*(density(x,y,z+1) - density(x,y,z-1));
				grad_z(x,y,z) = 0.125*(density(x+1,y,z+1) - density(x+1,y,z-1));
				grad_z(x,y,z) = 0.125*(density(x-1,y,z+1) - density(x-1,y,z-1));
				grad_z(x,y,z) = 0.125*(density(x,y+1,z+1) - density(x,y+1,z-1));
				grad_z(x,y,z) = 0.125*(density(x,y-1,z+1) - density(x,y-1,z-1));
			}
		}
	}
	// spline coeff of dCOdx's
	spline_coeffs( grad_x , coeff_grad_x );
	spline_coeffs( grad_y , coeff_grad_y );
	spline_coeffs( grad_z , coeff_grad_z );

	// >> debugging <<
	//ElectronDensity(atm_dX).writeMRC( "atm_dX.mrc" );
	//ElectronDensity(dCOdx).writeMRC( "dCOdz.mrc" );
	//exit(1);

	TR << "Finished computing gradient maps" << std::endl;
}


//
// void ElectronDensity::computeStats()
//
void ElectronDensity::computeStats() {
	core::Real sum=0, sum2=0, sumCoM=0;
	dens_max = -FLT_MAX;
	dens_min = FLT_MAX;
	centerOfMass = numeric::xyzVector<core::Real>(0,0,0);

	int N = density.u1()*density.u2()*density.u3();
	core::Real N2 = square((core::Real)N);

	for (int i=1; i<density.u1(); i++)
	for (int j=1; j<density.u2(); j++)
	for (int k=1; k<density.u3(); k++) {
		sum += density(i,j,k);
		sum2 += density(i,j,k)*density(i,j,k);
		dens_max = std::max(dens_max, (core::Real)density(i,j,k));
		dens_min = std::min(dens_min, (core::Real)density(i,j,k));
	}
	dens_mean = sum/N;
	dens_stdev = sqrt( ( (N*sum2-sum*sum) /  (N2) ) );

	for (int i=1; i<density.u1(); i++)
	for (int j=1; j<density.u2(); j++)
	for (int k=1; k<density.u3(); k++) {
		// do we only care about nonnegative density when compuing com?  think about this
		if (density(i,j,k) > 0) {
			centerOfMass[0] += i*density(i,j,k);
			centerOfMass[1] += j*density(i,j,k);
			centerOfMass[2] += k*density(i,j,k);
			sumCoM += density(i,j,k);
		}
	}
	centerOfMass /= sumCoM;

	// now that we have CoM, update effective CoM
	//std::string align = core::options::option[ core::options::OptionKeys::edensity::realign ]();
	//if ( align != "no" && align != "min") {
	//	efforigin = -centerOfMass+1.0;
	//}

	TR << "Density map stats:" << std::endl;
	TR << "      min = " << dens_min << std::endl;
	TR << "      max = " << dens_max << std::endl;
	TR << "     mean = " << dens_mean << std::endl;
	TR << "    stdev = " << dens_stdev << std::endl;
	TR << "      CoM = " << centerOfMass[0] << " , " << centerOfMass[1] << " , " << centerOfMass[2] << std::endl;
	numeric::xyzVector< core::Real > trans = getTransform();
	TR << "    xform = " << trans[0] << " , " << trans[1] << " , " << trans[2] << std::endl;
	TR << "  efforig = " << efforigin[0] << " , " << efforigin[1] << " , " << efforigin[2] << std::endl;
}

//
// void ElectronDensity::computeCrystParams()
//
void ElectronDensity::computeCrystParams() {
	// recompute reciprocal cell
	// f2c, c2f
	core::Real ca = cos(d2r(cellAngles[0])), cb = cos(d2r(cellAngles[1])), cg = cos(d2r(cellAngles[2]));
	core::Real sa = sin(d2r(cellAngles[0])), sb = sin(d2r(cellAngles[1])), sg = sin(d2r(cellAngles[2]));

	// conversion from fractional cell coords to cartesian coords
	this->f2c = numeric::xyzMatrix<core::Real>::rows(
		cellDimensions[0]  , cellDimensions[1] * cg, cellDimensions[2] * cb,
		0.0, cellDimensions[1] * sg, cellDimensions[2] * (ca - cb*cg) / sg,
		0.0, 0.0   , cellDimensions[2] * sb * sqrt(1.0 - square((cb*cg - ca)/(sb*sg))) );
	core::Real D = this->f2c.det();
	if (D == 0) {
		TR << "[ WARNING ] Invalid crystal cell dimensions." << std::endl;
		return;
	}
	// c2f is inverse of f2c
	this->c2f = numeric::xyzMatrix<core::Real>::rows(
		(f2c(2,2)*f2c(3,3)-f2c(2,3)*f2c(3,2))/D,
		-(f2c(1,2)*f2c(3,3)-f2c(1,3)*f2c(3,2))/D,
		(f2c(1,2)*f2c(2,3)-f2c(1,3)*f2c(2,2))/D,
		-(f2c(2,1)*f2c(3,3)-f2c(3,1)*f2c(2,3))/D,
		(f2c(1,1)*f2c(3,3)-f2c(1,3)*f2c(3,1))/D,
		-(f2c(1,1)*f2c(2,3)-f2c(1,3)*f2c(2,1))/D,
		(f2c(2,1)*f2c(3,2)-f2c(3,1)*f2c(2,2))/D,
		-(f2c(1,1)*f2c(3,2)-f2c(1,2)*f2c(3,1))/D,
		(f2c(1,1)*f2c(2,2)-f2c(1,2)*f2c(2,1))/D );
	this->V = cellDimensions[0]*cellDimensions[1]*cellDimensions[2]* sqrt(1-square(ca)-square(cb)-square(cg)+2*ca*cb*cg);

	// reciprocal space cell dimensions
	this->RcellDimensions[0] = cellDimensions[1]*cellDimensions[2]*sa/V;
	this->RcellDimensions[1] = cellDimensions[0]*cellDimensions[2]*sb/V;
	this->RcellDimensions[2] = cellDimensions[0]*cellDimensions[1]*sg/V;
	this->cosRcellAngles[0] = cos(  asin( std::min( std::max( V/(cellDimensions[0]*cellDimensions[1]*cellDimensions[2]*sb*sg) , -1.0) , 1.0) )  );
	this->cosRcellAngles[1] = cos(  asin( std::min( std::max( V/(cellDimensions[0]*cellDimensions[1]*cellDimensions[2]*sa*sg) , -1.0) , 1.0) )  );
	this->cosRcellAngles[2] = cos(  asin( std::min( std::max( V/(cellDimensions[0]*cellDimensions[1]*cellDimensions[2]*sa*sb) , -1.0) , 1.0) )  );
	this->RV = 1.0/V;
}

}
}
}
