// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/electron_density/ElectronDensity.hh
/// @brief  Scoring a structure against an electron density map
/// @author Frank DiMaio

#ifndef INCLUDED_core_scoring_electron_density_ElectronDensity_HH
#define INCLUDED_core_scoring_electron_density_ElectronDensity_HH

// Project headers
#include <core/util/Tracer.fwd.hh>
#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/conformation/Atom.fwd.hh>
#include <core/scoring/electron_density/ElectronDensity.fwd.hh>
#include <core/kinematics/RT.hh>
#include <core/conformation/symmetry/SymmetryInfo.fwd.hh>

// Utility headers
#include <utility/vector1.hh>
#include <numeric/xyzMatrix.fwd.hh>
#include <numeric/xyzVector.io.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyz.functions.hh>
#include <utility/exit.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray3D.hh>

// C++ headers
#include <string>
#include <map>

namespace core {
namespace scoring {
namespace electron_density {

const core::Real MAX_FLT = 1e37;

class ElectronDensity {
public:
	/// @brief constructor
	ElectronDensity();

	/// @brief constructor
	ElectronDensity(std::string mapfile, core::Real reso=5.0);

	/// @brief constructor (for debugging only)
	template<class T>
	ElectronDensity( ObjexxFCL::FArray3D< T > const &map,
	                 core::Real apix = 1.0,
	                 numeric::xyzVector< int > new_origin=numeric::xyzVector< int >(0,0,0)) {

		// set defaults!!!
		isLoaded = true;

		efforigin = origin = new_origin;  // ignore elec_dens::align_model
		grid = numeric::xyzVector< int >(map.u1(),map.u2(),map.u3());
		cellDimensions = numeric::xyzVector< float >(apix*map.u1(),apix*map.u2(),apix*map.u3());
		cellAngles = numeric::xyzVector< float >(90,90,90);
		density.dimension( map.u1(),map.u2(),map.u3() );

		for (int i=0; i<map.u1()*map.u2()*map.u3(); ++i) {
			density[i] = (float)map[i];
		}
	}

	/// @brief Load an MRC (="new-CCP4") density map
	bool readMRCandResize(std::string mapfile, core::Real reso=5.0, core::Real gridSpacing=0.0);

	/// @brief (debugging) Write MRC mapfile
	bool writeMRC(std::string mapfilestem);

	/// @brief (debugging) Write MATLAB v5 mapfile
	bool writeMAT(std::string mapfilestem);

	/// @brief Quickly align a pose to the density map by aligning centers of mass.
	///    Only works if monomer is masked in the density data
	///core::Real fastTransAlign( pose::Pose const & pose );

	/// @brief Align a pose to loaded density map; return optimal translation & rotation of structure
	//core::Real fastRotAlign( pose::Pose const & pose ,
	//                  numeric::xyzMatrix<core::Real> &rot,
	//                  numeric::xyzVector<core::Real> &trans );

	/// @brief Quickly matches a centroid pose into a low-resolution density map
	///   by placing a single Gaussian at each CA
	core::Real matchCentroidPose( core::pose::Pose const &pose,
	                              const core::conformation::symmetry::SymmetryInfo *symmInfo=NULL,
	                              bool cacheCCs=false );

	/// @brief Quickly matches a centroid pose into a low-resolution density map
	///   by placing a single Gaussian at each atom
	core::Real matchPose( core::pose::Pose const &pose,
	                      const core::conformation::symmetry::SymmetryInfo *symmInfo=NULL,
	                      bool cacheCCs=false );

	/// @brief Match a residue's conformation to the density map.
	///   Backbone atoms from adjacent residues are also used for scoring.
	///   Returns the correlation coefficient between map and pose
	///   Internally stores per-res CCs, per-atom dCC/dxs
	core::Real matchRes( int resid,
	                     core::conformation::Residue const &rsd,
	                     core::pose::Pose const &pose,
	                     const core::conformation::symmetry::SymmetryInfo *symmInfo=NULL,
	                     bool cacheCCs=false );

	/// @brief Computes the symmatric rotation matrices
	void compute_symm_rotations( core::pose::Pose const &pose,
	                             const core::conformation::symmetry::SymmetryInfo *symmInfo=NULL );

	/// @brief Return the gradient of CC w.r.t. atom X's movement
	/// Uses information stored from the previous call to matchRes with this resid
	void  dCCdx_res( int atmid, int resid,
	                 numeric::xyzVector<core::Real> const &X,
	                 core::conformation::Residue const &rsd,
	                 core::pose::Pose const &pose,
	                 numeric::xyzVector<core::Real> &gradX);

	/// @brief Return the gradient of CC w.r.t. res X's CA's movement
	/// Centroid-mode analogue of dCCdx
	void  dCCdx_cen( int resid,
	            		 numeric::xyzVector<core::Real> const &X,
              		 core::pose::Pose const &pose,
	                 numeric::xyzVector<core::Real> &gradX);

	/// @brief Return the gradient of whole-structure-CC w.r.t. atom X's movement
	/// non-sliding-window analogue of dCCdx
	void  dCCdx_aacen( int atmid, int resid,
	                   numeric::xyzVector<core::Real> const &X,
                     core::pose::Pose const &pose,
	                   numeric::xyzVector<core::Real> &gradX);

	/// @brief Get the (previously set) resolution of this map
	core::Real getResolution( ) { return this->reso; }

	/// @brief Is a map loaded?
	bool isMapLoaded() { return this->isLoaded; };

	/// @brief Only match a subset of residues
	void maskResidues( int scoring_mask ) {
		scoring_mask_[ scoring_mask ] = 1;
	}

	/// @brief Only match a subset of residues
	void maskResidues( utility::vector1< int > const & scoring_mask ) {
		for (core::Size i=1; i<= scoring_mask.size(); ++i)
			scoring_mask_[ scoring_mask[i] ] = 1;
	}

	/// @brief Reset scoring mask to default (score all non-VRT residues)
	void clearMask( ) {
		scoring_mask_.clear();
	}

	/// @brief Get the transformation from indices to Cartesian coords using 'real' origin
	numeric::xyzVector<core::Real> getTransform() {
		numeric::xyzVector<core::Real> idxX(  grid[0]-origin[0]+1 , grid[1]-origin[1]+1 , grid[2]-origin[2]+1 ) , cartX;
		idx2cart( idxX , cartX );
		return cartX;
	}

	/// @brief Get the transformation from 'fake' origin to 'real' origin
	/// [DEPRECIATED] effOrigin is getting phased out and should always be = to origin; this should always return 0
	numeric::xyzVector<core::Real> getGlobalTransform() {
		numeric::xyzVector<core::Real> idxX(  origin[0]-efforigin[0] , origin[1]-efforigin[1] , origin[2]-efforigin[2] ) , cartX;
		idxoffset2cart( idxX , cartX );
		return cartX;
	}

	/// @brief set # of residues
	void set_nres(int nres) {
		if ( (int) CCs.size() != nres ) {
			CCs.resize(nres, 0.0);
			dCCdxs_res.resize(nres);
			dCCdxs_cen.resize(nres);
			dCCdxs_aacen.resize(nres);
			symmap.clear();   // reset if #residues changes
		}
	}

	void showCachedScores( utility::vector1< int > const &reses );

	//////////////////////////////////
	//////////////////////////////////
	// getters and setters
	inline core::Real getCCs( int resid ) { return CCs[resid]; }

	inline void setUseDensityInMinimizer( bool newVal ) { DensScoreInMinimizer = newVal; }
	inline bool getUseDensityInMinimizer() const { return DensScoreInMinimizer; }

	inline void setUseExactDerivatives( bool newVal ) { ExactDerivatives = newVal; }
	inline bool getUseExactDerivatives() const { return ExactDerivatives; }

	inline core::Real getNumDerivH() const { return NUM_DERIV_H; }

	inline core::Real getMean() const { return dens_mean; }
	inline core::Real getMin()  const { return dens_min;  }
	inline core::Real getMax()  const { return dens_max;  }
	inline core::Real getStdev() const { return dens_stdev; }

	inline numeric::xyzVector<core::Real> getCoM() const { return centerOfMass; }
	inline numeric::xyzVector<core::Real> getOrigin() const { return origin; }
	inline numeric::xyzVector<core::Real> getEffOrigin() const { return efforigin; }

	// raw data pointer
	inline ObjexxFCL::FArray3D< float > const & data() const { return density; };


	//////////////////////////////////
	//////////////////////////////////
	// helper functions to convert between indices and cartesian coords
	inline void cart2idx( numeric::xyzVector<core::Real> const & cartX , numeric::xyzVector<core::Real> &idxX ) const {
		numeric::xyzVector<core::Real> fracX = c2f*cartX;
		idxX = numeric::xyzVector<core::Real>( fracX[0]*grid[0] - efforigin[0] + 1,
		                                       fracX[1]*grid[1] - efforigin[1] + 1,
		                                       fracX[2]*grid[2] - efforigin[2] + 1);
	}

	template<class Q>
	void idx2cart( numeric::xyzVector<Q> const & idxX , numeric::xyzVector<core::Real> &cartX ) const {
		numeric::xyzVector<core::Real> fracX( (idxX[0]  + efforigin[0] -1 ) / grid[0],
		                                      (idxX[1]  + efforigin[1] -1 ) / grid[1],
		                                      (idxX[2]  + efforigin[2] -1 ) / grid[2] );
		cartX = f2c*fracX;
	}

	template<class Q>
	void idxoffset2cart( numeric::xyzVector<Q> const & idxX , numeric::xyzVector<core::Real> &cartX ) const {
		numeric::xyzVector<core::Real> fracX( ( (core::Real) idxX[0] ) / grid[0],
		                                      ( (core::Real) idxX[1] ) / grid[1],
		                                      ( (core::Real) idxX[2] ) / grid[2] );
		cartX = f2c*fracX;
	}

	numeric::xyzVector<core::Real> dens_grad ( numeric::xyzVector<core::Real> const & idxX ) const;

	// trilinear interpolation of electron density
	template <class S>
	core::Real interp_linear(
                       ObjexxFCL::FArray3D< S > const & data ,
                       numeric::xyzVector< core::Real > const & idxX) const ;

	// spline interpolation of electron density
	core::Real interp_spline( ObjexxFCL::FArray3D< double > & coeffs ,
	                          numeric::xyzVector<core::Real> const & idxX ) const;

	/// resize the map via FFT
	void resize( core::Real approxGridSpacing );


	////
	////
	//// access cached data from last scored pose
	////	std::map< int , std::pair< utility::vector1<int> , numeric::xyzMatrix<core::Real> > > symmap_aacen
	void get_symmMap(int vrtid, utility::vector1<int> &X_map, numeric::xyzMatrix<core::Real> &R) {
		runtime_assert( symmap.find( vrtid ) != symmap.end() );
		X_map = symmap[ vrtid ].first;
		R = symmap[ vrtid ].second;
	}

	void get_R(int subunit, numeric::xyzMatrix<core::Real> &R) {
		runtime_assert( symmap.find( -subunit ) != symmap.end() );
		R = symmap[ -subunit ].second;
	}


///////////
// PRIVATE MEMBER FUNCTIONS
///////////
private:
	// wrapper functs to compute spline coeffs
	void spline_coeffs( ObjexxFCL::FArray3D< double > & data ,
	                    ObjexxFCL::FArray3D< double > & coeffs);
	void spline_coeffs( ObjexxFCL::FArray3D< float > & data ,
	                    ObjexxFCL::FArray3D< double > & coeffs);

	// trilinear inverse interpolation of electron density
	void inc_linear( ObjexxFCL::FArray3D< double > & data ,
	                 numeric::xyzVector<core::Real> const & idxX,
	                 core::Real value);

	// NN inverse interpolation of electron density
	template<class T>
	void set_nearest( ObjexxFCL::FArray3D< T > & data ,
	                  numeric::xyzVector<core::Real> const & idxX,
	                  T value);

	// helper functions for map statistics
	void computeGradients();
	void computeStats();
	int suggestRadius();

	// helper functions for symmetry
	void computeCrystParams();
	void initializeSymmOps( utility::vector1< std::string > const & symList );
	void expandToUnitCell();

///////////
// DATA
///////////
private:
	// the density data array
	ObjexxFCL::FArray3D< float > density;

	// parameters for derivative computation
	bool DensScoreInMinimizer, ExactDerivatives;
	core::Real NUM_DERIV_H, NUM_DERIV_H_CEN;

	// statistics in the last-scored pose
	utility::vector1<core::Real>  CCs;
	core::Real CC_cen;
	core::Real CC_aacen;

	// more caching for derivative computation
	utility::vector1< utility::vector1< numeric::xyzVector< core::Real > > > dCCdxs_res;

	// ... for centroid mode
	utility::vector1< numeric::xyzVector< core::Real > > dCCdxs_cen;

	// ... for aa_cen mode
	utility::vector1< utility::vector1< numeric::xyzVector< core::Real > > > dCCdxs_aacen;

	// SYMMETRY ONLY
	// map vrtid -> subunit mapping, rotation
	// if (vrtid < 0) then it refers to the mapping from a non-vrt in subunit# -vrtid
	std::map< int , std::pair< utility::vector1<int> , numeric::xyzMatrix<core::Real> > > symmap;

	// map info
	numeric::xyzVector< int > grid;
	numeric::xyzVector< core::Real > origin, efforigin;

	// gradients, used for displaying isocontoured surface .. mutable for the viewer to access
	mutable ObjexxFCL::FArray3D< double > coeff_grad_x, coeff_grad_y, coeff_grad_z;

	// do we have a map loaded?
	bool isLoaded;

	// (if set) don't score these residues
	std::map< core::Size, bool > scoring_mask_;

	// map resolution, mask widths
	core::Real reso, ATOM_MASK, CA_MASK;

	// unit cell, reciprocal unit cell parameters
	numeric::xyzVector<float> cellDimensions, cellAngles;
	numeric::xyzVector<float> RcellDimensions, cosRcellAngles;

	// converting fractional to cartesian coords
	numeric::xyzMatrix<core::Real> f2c, c2f;

	// unit cell/reciprocal cell volume
	core::Real V, RV;

	// symmetric transforms (in frac. coords) -- UNUSED (all symminfo comes from SymmInfo object)
	utility::vector1< core::kinematics::RT > symmOps;

	// map statistics
	numeric::xyzVector<core::Real> centerOfMass;
	core::Real dens_mean, dens_min, dens_max, dens_stdev;
};

/// @brief The EDM instance
ElectronDensity& getDensityMap();

}
}
}


#endif

