// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/etable/EtableEnergy.cc
/// @brief  Atom pair energy functions
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)
/// @author Andrew Leaver-Fay

#ifndef INCLUDED_core_scoring_etable_BaseEtableEnergy_tmpl_HH
#define INCLUDED_core_scoring_etable_BaseEtableEnergy_tmpl_HH


// Unit headers
#include <core/scoring/etable/EtableEnergy.hh>

// Package headers
#include <core/scoring/EnergiesCacheableDataType.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/NeighborList.hh>
#include <core/scoring/NeighborList.tmpl.hh>
#include <core/scoring/methods/Methods.hh>
#include <core/scoring/methods/EnergyMethodCreator.hh>

#include <core/scoring/etable/count_pair/CountPairFunction.hh>
#include <core/scoring/etable/count_pair/CountPairFactory.hh>

//#include <core/scoring/etable/count_pair/CountPair1BC3.hh> // remove this
//#include <core/scoring/etable/count_pair/CountPair1BC4.hh> // remove this
//#include <core/scoring/etable/count_pair/CountPairIntraResC3.hh>
//#include <core/scoring/etable/count_pair/CountPairIntraResC4.hh>
#include <core/scoring/etable/count_pair/CountPairAll.hh> // remove this
#include <core/scoring/etable/count_pair/CountPairNone.hh> // maybe this stays

#include <core/scoring/trie/CPDataCorrespondence.hh>
#include <core/scoring/trie/RotamerDescriptor.hh>
#include <core/scoring/trie/RotamerTrie.hh>
#include <core/scoring/trie/TrieCollection.hh>
#include <core/scoring/trie/TrieCountPairBase.hh>
#include <core/scoring/trie/trie.functions.hh>

#include <core/scoring/etable/etrie/CountPairData_1_1.hh>
#include <core/scoring/etable/etrie/CountPairData_1_2.hh>
#include <core/scoring/etable/etrie/CountPairData_1_3.hh>
#include <core/scoring/etable/etrie/CountPairDataGeneric.hh>
#include <core/scoring/etable/etrie/EtableAtom.hh>
#include <core/scoring/etable/etrie/TrieCountPair1BC3.hh>
#include <core/scoring/etable/etrie/TrieCountPair1BC4.hh>
#include <core/scoring/etable/etrie/TrieCountPairAll.hh>
#include <core/scoring/etable/etrie/TrieCountPairNone.hh>
#include <core/scoring/etable/etrie/TrieCountPairGeneric.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/optimization/MinimizerMap.hh>
#include <core/pack/rotamer_set/RotamerSet.hh>
#include <core/pack/rotamer_set/RotamerSetFactory.hh>
#include <core/chemical/VariantType.hh>
#include <core/chemical/AtomType.hh>
#include <core/options/option.hh>

// option key includes

#include <core/options/keys/score.OptionKeys.gen.hh>
#include <core/options/keys/run.OptionKeys.gen.hh>



// Numeric headers

namespace core {
namespace scoring {
namespace etable {

using namespace etrie;
using namespace trie;
using namespace core::options;

  /// construction with an etable
template < class Derived >
BaseEtableEnergy< Derived >::BaseEtableEnergy(
	methods::EnergyMethodCreatorOP creator,
	Etable const & etable_in,
	methods::EnergyMethodOptions const & options,
	ScoreType st_atr,
	ScoreType st_rep,
	ScoreType st_sol
):
	parent( creator ),
	etable_( etable_in ),
	//atom_set_ptr_( &(etable_in.atom_set() ) ),
	ljatr_( etable_in.ljatr() ),
	ljrep_( etable_in.ljrep() ),
	solv1_( etable_in.solv1() ),
	solv2_( etable_in.solv2() ),
	dljatr_( etable_in.dljatr() ),
	dljrep_( etable_in.dljrep() ),
	dsolv_( etable_in.dsolv() ),
	safe_max_dis2( etable_in.get_safe_max_dis2() ),
	etable_bins_per_A2( etable_in.get_bins_per_A2() ),
	hydrogen_interaction_cutoff2_( option[ OptionKeys::score::fa_Hatr ] ?
		std::pow( std::sqrt( etable_in.hydrogen_interaction_cutoff2()) + std::sqrt( safe_max_dis2 ), 2)
		: etable_in.hydrogen_interaction_cutoff2() ),
	st_rep_(st_rep),
	st_atr_(st_atr),
	st_sol_(st_sol),
	exclude_DNA_DNA( options.exclude_DNA_DNA() )
{}


///////////////////////////////////////////////////////////////////////////////
template < class Derived >
void
BaseEtableEnergy< Derived >::finalize_total_energy(
	pose::Pose & pose,
	ScoreFunction const &,
	EnergyMap & totals
) const
{
	if ( pose.energies().use_nblist() ) {
		TwoBodyEnergyMap tbenergy_map;
		// add in contributions from the nblist atom-pairs
		NeighborList const & nblist
			( pose.energies().nblist( EnergiesCacheableDataType::ETABLE_NBLIST ) );

		nblist.check_domain_map( pose.energies().domain_map() );

		/// Trick to avoid calls to Conformation::residue()
		utility::vector1< conformation::Residue const * > resvect;
		resvect.reserve( pose.total_residue() );
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			resvect.push_back( & pose.residue( ii ) );
		}
		Real dsq(0.0);
		for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
			conformation::Residue const & ires( *resvect[i] );
			for ( Size ii=1, ii_end=ires.natoms(); ii<= ii_end; ++ii ) {
				/// 1. Iterate across intra-residue atom neighbors if there are any;
				/// 2. Iterate across inter-residue atom neighbors.
				prepare_for_residue_pair( 1,1, pose ); // set intra-res
				AtomNeighbors const & intranbrs( nblist.intrares_upper_atom_neighbors(i,ii) );
				conformation::Atom const & iatom( ires.atom(ii) );
				for ( AtomNeighbors::const_iterator nbr=intranbrs.begin(),
								nbr_end=intranbrs.end(); nbr!= nbr_end; ++nbr ) {
					Size const jj( nbr->atomno() );
					Real const cp_weight( nbr->weight_func()*nbr->weight() );  //fpd

					conformation::Atom const & jatom( ires.atom(jj) );
					atom_pair_energy( iatom, jatom, cp_weight, tbenergy_map, dsq );
				}

				prepare_for_residue_pair( 1,2, pose ); // set inter-res
				AtomNeighbors const & nbrs( nblist.upper_atom_neighbors(i,ii) );
				for ( AtomNeighbors::const_iterator nbr=nbrs.begin(),
								nbr_end=nbrs.end(); nbr!= nbr_end; ++nbr ) {
					Size const  j( nbr->rsd() );
					Size const jj( nbr->atomno() );

					Real const cp_weight( nbr->weight_func()*nbr->weight() );
					conformation::Atom const & jatom( resvect[j]->atom(jj) );
					atom_pair_energy( iatom, jatom, cp_weight, tbenergy_map, dsq );
				}
			}
		}
		//std::cout << "totals[ fa_atr ] : " << totals[ fa_atr ] << std::endl;
		//std::cout << "totals[ fa_rep ] : " << totals[ fa_rep ] << std::endl;
		//std::cout << "totals[ fa_sol ] : " << totals[ fa_sol ] << std::endl;
		//std::cout << "totals[ fa_intra_atr ] : " << totals[ fa_intra_atr ] << std::endl;
		//std::cout << "totals[ fa_intra_rep ] : " << totals[ fa_intra_rep ] << std::endl;
		//std::cout << "totals[ fa_intra_sol ] : " << totals[ fa_intra_sol ] << std::endl;
		totals += tbenergy_map;
	}
}

///////////////////////////////////////////////////////////////////////////////
template < class Derived >
void
BaseEtableEnergy< Derived >::setup_for_minimizing(
	pose::Pose & pose,
	ScoreFunction const & sfxn,
	optimization::MinimizerMap const & min_map
) const
{
	using namespace core::options;
	using namespace core::options::OptionKeys;

	if ( pose.energies().use_nblist() ) {
		// stash our nblist inside the pose's energies object
		Energies & energies( pose.energies() );

		// setup the atom-atom nblist
		NeighborListOP nblist;

		if ( pose.energies().use_nblist_auto_update() ) {
			Real const tolerated_narrow_nblist_motion = option[ run::nblist_autoupdate_narrow ];
			Real const XX = etable_.max_dis() + 2 * tolerated_narrow_nblist_motion;
			Real const XH = etable_.max_non_hydrogen_lj_radius() + etable_.max_hydrogen_lj_radius()
				+ 2 * tolerated_narrow_nblist_motion;
			Real const HH = etable_.max_hydrogen_lj_radius() + etable_.max_hydrogen_lj_radius()
				+ 2 * tolerated_narrow_nblist_motion;

			nblist = new NeighborList(
				min_map.domain_map(),
				XX*XX,
				XH*XH,
				HH*HH);
			nblist->set_auto_update( tolerated_narrow_nblist_motion );
		} else {

			/// Use the default parameters
			nblist = new NeighborList(
				min_map.domain_map(),
				etable_.nblist_dis2_cutoff_XX(),
				etable_.nblist_dis2_cutoff_XH(),
				etable_.nblist_dis2_cutoff_HH());
		}
		// this partially becomes the EtableEnergy classes's responsibility
		nblist->setup( pose, sfxn, static_cast<Derived const&> (*this));

		energies.set_nblist( EnergiesCacheableDataType::ETABLE_NBLIST, nblist );
	}
}

// check compatibility with atomtypeset
template < class Derived >
void
BaseEtableEnergy< Derived >::setup_for_scoring(
	pose::Pose &pose,
	ScoreFunction const &scfxn
) const
{
	assert( dynamic_cast< Derived const* > (this) );
	Derived const * ptr = static_cast< Derived const* > (this);
	ptr->setup_for_scoring_(pose,scfxn);
	if ( pose.energies().use_nblist() ) {
		NeighborList const & nblist( pose.energies().nblist( EnergiesCacheableDataType::ETABLE_NBLIST ) );
		nblist.prepare_for_scoring( pose, scfxn, *ptr );
	}
}

///@details Make sure that the neighborlist is up-to-date bfore evaluating derivatives
template < class Derived >
void
BaseEtableEnergy< Derived >::setup_for_derivatives(
	pose::Pose &pose,
	ScoreFunction const &scfxn
) const
{
	//std::cout << "BaseEtable Setup for derivatives" << std::endl;
	assert( dynamic_cast< Derived const* > (this) );
	Derived const * ptr = static_cast< Derived const* > (this);
	ptr->setup_for_scoring_(pose,scfxn);
	if ( pose.energies().use_nblist() ) {
		NeighborList const & nblist( pose.energies().nblist( EnergiesCacheableDataType::ETABLE_NBLIST ) );
		nblist.prepare_for_scoring( pose, scfxn, *ptr );
	}
}



// The EtableEnergy method stores a vector of rotamer trie objects in the Energies
// object for use in rapid rotamer/background energy calculations.  Overrides default
// do-nothing behavior.
template < class Derived >
void
BaseEtableEnergy< Derived >::setup_for_packing(
	pose::Pose & pose,
	pack::task::PackerTask const &
) const
{
	using namespace pack::rotamer_set;

	TrieCollectionOP tries = new TrieCollection;
	tries->total_residue( pose.total_residue() );
	for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
		// Do not compute energy for virtual residues.
		if (pose.residue(ii).aa() == core::chemical::aa_vrt) continue;
		//iwd  Temporary hack: also skip ligand residues
		//if ( !pose.residue(ii).is_polymer() )
		//	continue;

		// something of a hack: create a residue set, put inside it a single residue,
		// and then use the existing trie creation code -- single residue tries are
		// constructed with the same logic as are multiple-residue tries.
		RotamerSetFactory rsf;
		RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( ii ) );
		one_rotamer_set->set_resid( ii ); // change the interface so that the rotset initializes its seqpos at creation
		one_rotamer_set->add_rotamer( pose.residue( ii ) );
		EtableRotamerTrieOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );
		tries->trie( ii, one_rotamer_trie );
	}
	pose.energies().data().set( EnergiesCacheableDataType::ETABLE_TRIE_COLLECTION, tries );

}

// @brief Creates a rotamer trie for the input set of rotamers and stores the trie
// in the rotamer set.
template < class Derived >
void
BaseEtableEnergy< Derived >::prepare_rotamers_for_packing(
	pose::Pose const & pose,
	pack::rotamer_set::RotamerSet & set
) const
{

	EtableRotamerTrieOP rottrie = create_rotamer_trie( set, pose );
	set.store_trie( methods::etable_method, rottrie );
}


// @brief Updates the cached rotamer trie for a residue if it has changed during the course of
// a repacking
template < class Derived >
void
BaseEtableEnergy< Derived >::update_residue_for_packing(
	pose::Pose & pose,
	Size resid
) const
{
	using namespace pack::rotamer_set;

	RotamerSetFactory rsf; // change RSF to have a static method create_rotamer_set
	RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( resid ) );
	one_rotamer_set->set_resid( resid ); // change the interface so that the rotset initializes its seqpos at creation
	one_rotamer_set->add_rotamer( pose.residue( resid ) );
	EtableRotamerTrieOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );

	// grab non-const & of the cached tries and replace resid's trie with a new one.
	TrieCollection & trie_collection
		( static_cast< TrieCollection & > (pose.energies().data().get( EnergiesCacheableDataType::ETABLE_TRIE_COLLECTION )));
	trie_collection.trie( resid, one_rotamer_trie );

}


// @brief convenience function
template < class Derived >
count_pair::CountPairFunctionCOP
BaseEtableEnergy< Derived >::get_count_pair_function(
	Size res1,
	Size res2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn
) const
{
	return get_count_pair_function( pose.residue( res1 ), pose.residue( res2 ), pose, sfxn );
}

// @brief all of the residue level count pair logic goes here
/// and gets duplicated in a handful of other places within this class... this NEEDS to be reworked.
template < class Derived >
count_pair::CountPairFunctionCOP
BaseEtableEnergy< Derived >::get_count_pair_function(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	pose::Pose const & pose, // does the pose hold the connectivity information or do the residues?
	ScoreFunction const & sfxn
) const
{
	using namespace count_pair;

	if ( exclude_DNA_DNA && res1.is_DNA() && res2.is_DNA() ) {
		return new CountPairNone;
	}

	CPCrossoverBehavior crossover = determine_crossover_behavior( res1, res2, pose, sfxn );
	return CountPairFactory::create_count_pair_function( res1, res2, crossover );
}

// @brief get a count pair object for intraresidue pair energies
template < class Derived >
count_pair::CountPairFunctionOP
BaseEtableEnergy< Derived >::get_intrares_countpair(
	conformation::Residue const & res,
	pose::Pose const & pose,
	ScoreFunction const & sfxn
) const
{
	using namespace count_pair;

	if ( exclude_DNA_DNA && res.is_DNA() ) {
		return new CountPairNone;
	}

	CPCrossoverBehavior crossover = determine_crossover_behavior( res, res, pose, sfxn );
	return CountPairFactory::create_intrares_count_pair_function( res, crossover );
}


/// @brief figure out the trie count pair function to use
/// Need to refactor this so that the decision "what kind of count pair behavior should I use" can be decoupled
/// from class instantiation, and therefore shared between the creation of the trie count pair classes and the regular
/// count pair classes
template < class Derived >
TrieCountPairBaseOP
BaseEtableEnergy< Derived >::get_count_pair_function_trie(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn
) const
{
	using namespace methods;
	conformation::Residue const & res1( pose.residue( set1.resid() ) );
	conformation::Residue const & res2( pose.residue( set2.resid() ) );
	trie::RotamerTrieBaseCOP trie1 = set1.get_trie( etable_method );
	trie::RotamerTrieBaseCOP trie2 = set2.get_trie( etable_method );

	return get_count_pair_function_trie( res1, res2, trie1, trie2, pose, sfxn );
}

template < class Derived >
TrieCountPairBaseOP
BaseEtableEnergy< Derived >::get_count_pair_function_trie(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	trie::RotamerTrieBaseCOP trie1,
	trie::RotamerTrieBaseCOP trie2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn
) const
{
	using namespace count_pair;

	TrieCountPairBaseOP tcpfxn;
	if ( exclude_DNA_DNA && res1.is_DNA() && res2.is_DNA() ) {
		tcpfxn = new TrieCountPairNone();
		return tcpfxn;
	}

	CPResidueConnectionType connection = count_pair::CountPairFactory::determine_residue_connection( res1, res2 );
	if ( connection == CP_ONE_BOND ) {
		CPCrossoverBehavior crossover = determine_crossover_behavior( res1, res2, pose, sfxn );
		Size conn1 = trie1->get_count_pair_data_for_residue( res2.seqpos() );
		Size conn2 = trie2->get_count_pair_data_for_residue( res1.seqpos() );
		switch ( crossover ) {
			case CP_CROSSOVER_3 :
				tcpfxn = new TrieCountPair1BC3( conn1, conn2 );
			break;
			case CP_CROSSOVER_4 :
				tcpfxn = new TrieCountPair1BC4( conn1, conn2 );
			break;
			default:
				utility_exit();
			break;
		}
	} else if ( connection == CP_MULTIPLE_BONDS_OR_PSEUDOBONDS ) {
		CPCrossoverBehavior crossover = determine_crossover_behavior( res1, res2, pose, sfxn );

		Size conn1 = trie1->get_count_pair_data_for_residue( res2.seqpos() );
		Size conn2 = trie2->get_count_pair_data_for_residue( res1.seqpos() );

		TrieCountPairGenericOP cpgen = new TrieCountPairGeneric( res1, res2, conn1, conn2 );
		if ( crossover == CP_CROSSOVER_3 ) {
			cpgen->crossover( 3 );
		} else if ( crossover == CP_CROSSOVER_4 ) {
			cpgen->crossover( 4 );
		} else {
			utility_exit();
		}
		cpgen->hard_crossover( false );
		tcpfxn = cpgen;
	} else {
		tcpfxn = new TrieCountPairAll;
	}
	return tcpfxn;

}


/*
template < class Derived >
count_pair::CPResidueConnectionType
BaseEtableEnergy< Derived >::determine_residue_connection(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	pose::Pose const &
) const
{
	using namespace count_pair;
	/// this code is incompatible with designing both disulfides and non-disulfies
	/// at the same residue...
	if ( res1.is_pseudobonded( res2.seqpos ) {
		return CP_MULTIPLE_BONDS_OR_PSEUDOBONDS;
	} else if ( res1.is_bonded(res2) ) {
		if ( res1.connections_to_residue( res2 ).size() == 1 ) {
			return CP_ONE_BOND;
		} else {
			return CP_MULTIPLE_BONDS_OR_PSEUDOBONDS;
		}
	} else {
		return CP_NO_BONDS;
	}
}
*/

template < class Derived >
count_pair::CPCrossoverBehavior
BaseEtableEnergy< Derived >::determine_crossover_behavior(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	pose::Pose const &,
	ScoreFunction const & sfxn
) const
{
	using namespace count_pair;
	// maybe should ask "are these two residues polymers and do they share a polymeric bond"
	if ( res1.polymeric_sequence_distance(res2) == 1 ) {
		if ( ( !sfxn.has_zero_weight( mm_twist ) && sfxn.has_zero_weight( rama )) ||
				 ( ( !res1.is_protein() || !res2.is_protein()) &&
					 ( !res1.is_RNA() || !res2.is_RNA())
					 ) ) {
			return CP_CROSSOVER_3;
		} else {
			return CP_CROSSOVER_4; // peptide bond w/ or w/o rama, but definately w/o mm_twist
		}
	} else if ( res1.seqpos() == res2.seqpos() ) {
		// logic for controlling intra-residue count pair behavior goes here; for now, default to crossover 3
		return CP_CROSSOVER_3;
	}else {
		return CP_CROSSOVER_3; // e.g. disulfides where seqsep != 1
	}
}

///
template < class Derived >
void
BaseEtableEnergy< Derived >::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	TwoBodyEnergyMap & emap
) const
{
	assert( rsd1.seqpos() != rsd2.seqpos() );
	//std::cerr << __FILE__<< ' ' << __LINE__ << std::endl;

	if ( ! pose.energies().use_nblist() ) {
		prepare_for_residue_pair(rsd1.seqpos(),rsd2.seqpos(),pose);
		count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd1, rsd2, pose, sfxn );
		cpfxn->residue_atom_pair_energy( rsd1, rsd2, static_cast<Derived const&> (*this), emap );
	}
}

/// @details do not use this during minimization
template < class Derived >
void
BaseEtableEnergy< Derived >::backbone_backbone_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	TwoBodyEnergyMap & emap
) const
{
	assert( ! pose.energies().use_nblist() );
	prepare_for_residue_pair(rsd1.seqpos(),rsd2.seqpos(),pose);
	count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd1, rsd2, pose, sfxn );
	cpfxn->residue_atom_pair_energy_backbone_backbone( rsd1, rsd2, static_cast< Derived const&> (*this), emap );
}


/// @details do not use this during minimization
template < class Derived >
void
BaseEtableEnergy< Derived >::backbone_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	TwoBodyEnergyMap & emap
) const
{
	assert( ! pose.energies().use_nblist() );
	prepare_for_residue_pair(rsd2.seqpos(),rsd1.seqpos(),pose);
	count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd2, rsd1, pose, sfxn );
	cpfxn->residue_atom_pair_energy_sidechain_backbone( rsd2, rsd1, static_cast<Derived const&> (*this), emap );
}

//@details do not use this during minimization
template < class Derived >
void
BaseEtableEnergy< Derived >::sidechain_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	TwoBodyEnergyMap & emap
) const
{
	assert( ! pose.energies().use_nblist() );
	prepare_for_residue_pair(rsd1.seqpos(),rsd2.seqpos(),pose);
	count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd1, rsd2, pose, sfxn );
	cpfxn->residue_atom_pair_energy_sidechain_sidechain( rsd1, rsd2, static_cast<Derived const&> (*this), emap );
}



template < class Derived >
void
BaseEtableEnergy< Derived >::evaluate_rotamer_pair_energies(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
) const
{
	assert( set1.resid() != set2.resid() );

	using namespace methods;
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table1( energy_table );
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table2( energy_table );

	temp_table1 = 0; temp_table2 = 0;

	// save weight information so that its available during tvt execution
	weights_ = weights;

	/// this will later retrieve a stored rotamer trie from inside the set;
	//EtableRotamerTrieOP trie1 = create_rotamer_trie( set1, pose );
	//EtableRotamerTrieOP trie2 = create_rotamer_trie( set2, pose );

	EtableRotamerTrieCOP trie1 = set1.get_trie( etable_method );
	EtableRotamerTrieCOP trie2 = set2.get_trie( etable_method );

	prepare_for_residue_pair( set1.resid(), set2.resid(), pose );
	assert( rep_scoretype() == fa_rep || rep_scoretype() == coarse_fa_rep );

	// figure out which trie countPairFunction needs to be used for this set
	TrieCountPairBaseOP cp = get_count_pair_function_trie( set1, set2, pose, sfxn );

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge (and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_trie( *trie2, *cp, static_cast<Derived const&>(*this), temp_table1, temp_table2 );

	/// add in the energies calculated by the tvt alg.
	energy_table += temp_table1;
	//std::cout << "FINISHED evaluate_rotamer_pair_energies" << std::endl;

	/* // There should be a way to turn this on without recompiling...
	// debug
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table3( energy_table );
	temp_table3 = 0;
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set1.num_rotamers(); ii <= ii_end; ++ii ) {
		for ( Size jj = 1, jj_end = set2.num_rotamers(); jj <= jj_end; ++jj ) {
			emap.zero();
			residue_pair_energy( *set1.rotamer( ii ), *set2.rotamer( jj ), pose, sfxn, emap );
			temp_table3( jj, ii ) += weights.dot( emap );
			if ( std::abs( temp_table1( jj, ii ) - temp_table3( jj, ii )) > 0.001 ) {
				std::cout << "Residues " << set1.resid() << " & " << set2.resid() << " rotamers: " << ii << " & " << jj;
				std::cout << " tvt/reg discrepancy: tvt= " <<  temp_table1( jj, ii ) << " reg= " << temp_table3( jj, ii );
				std::cout << " delta: " << temp_table1( jj, ii ) - temp_table3( jj, ii ) << std::endl;
			}
		}
	}
	//std::cout << "Finished RPE calcs for residues " << set1.resid() << " & " << set2.resid() << std::endl;
	*/
}

template < class Derived >
void
BaseEtableEnergy< Derived >::evaluate_rotamer_background_energies(
	pack::rotamer_set::RotamerSet const & set,
	conformation::Residue const & residue,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	utility::vector1< pack::PackerEnergy > & energy_vector
) const
{
	//iwd  Temporary hack:  for ligands, call base class implementation
	//if ( !residue.is_polymer() ) {
	//	ShortRangeTwoBodyEnergy::evaluate_rotamer_background_energies(set, residue, pose, sfxn, weights, energy_vector);
	//	return;
	//}

	using namespace methods;
	// allocate space for the trie-vs-trie algorithm
	utility::vector1< pack::PackerEnergy > temp_vector1( set.num_rotamers(), 0.0 );
	utility::vector1< pack::PackerEnergy > temp_vector2( set.num_rotamers(), 0.0 );

	// save weight information so that its available during tvt execution
	weights_ = weights;

	EtableRotamerTrieCOP trie1 = set.get_trie( etable_method );
	EtableRotamerTrieCOP trie2 = ( static_cast< TrieCollection const & >
		( pose.energies().data().get( EnergiesCacheableDataType::ETABLE_TRIE_COLLECTION )) ).trie( residue.seqpos() );

	prepare_for_residue_pair( set.resid(), residue.seqpos(), pose );
	assert( rep_scoretype() == fa_rep || rep_scoretype() == coarse_fa_rep );
	// figure out which trie countPairFunction needs to be used for this set
	TrieCountPairBaseOP cp = get_count_pair_function_trie( pose.residue( set.resid() ), residue, trie1, trie2, pose, sfxn );

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge (and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_path( *trie2, *cp, static_cast<Derived const&> (*this), temp_vector1, temp_vector2 );

	/// add in the energies calculated by the tvt alg.
	for ( Size ii = 1; ii <= set.num_rotamers(); ++ii ) {
		energy_vector[ ii ] += temp_vector1[ ii ];
	}
	//std::cout << "FINISHED evaluate_rotamer_background_energies" << std::endl;

	/*
	//debug
	utility::vector1< Energy > temp_vector3( energy_vector.size(), 0.0f );
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set.num_rotamers(); ii <= ii_end; ++ii ) {
		emap.zero();
		residue_pair_energy( *set.rotamer( ii ), residue, pose, sfxn, emap );
		temp_vector3[ ii ] += weights.dot( emap );
		if ( std::abs( temp_vector1[ ii ] - temp_vector3[ ii ]) > 0.001 ) {
			std::cout << "Residues " << set.resid() << " & " << residue.seqpos() << " rotamers: " << ii << " & bg";
			std::cout << " tvt/reg discrepancy: tvt= " <<  temp_vector1[ ii ] << " reg= " << temp_vector3[ ii ];
			std::cout << " delta: " << temp_vector1[ ii ] - temp_vector3[ ii ] << std::endl;
		}
	}
	std::cout << "Finished Rotamer BG calcs for residues " << set.resid() << " & " << residue.seqpos() << std::endl;
	*/
}

/// @brief create a rotamer trie for a particular set, deciding upon the kind of count pair data that
/// needs to be contained by the trie.
///
template < class Derived >
EtableRotamerTrieOP
BaseEtableEnergy< Derived >::create_rotamer_trie(
	pack::rotamer_set::RotamerSet const & rotset,
	pose::Pose const & // will be need to create tries for disulfides
) const
{
	using namespace etrie;
	using namespace trie;

	CPDataCorrespondence cpdata_map( create_cpdata_correspondence_for_rotamerset( rotset ) );
	if ( cpdata_map.has_pseudobonds() ||
			cpdata_map.max_connpoints_for_residue() > 1 ||
			cpdata_map.n_entries() > 3 ) {
		EtableAtom at; CountPairDataGeneric cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, atomic_interaction_cutoff() );
	} else if ( cpdata_map.n_entries() == 1 || cpdata_map.n_entries() == 0 /* HACK! */ ) {
		EtableAtom at; CountPairData_1_1 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, atomic_interaction_cutoff() );
	} else if ( cpdata_map.n_entries() == 2 ) {
		EtableAtom at; CountPairData_1_2 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, atomic_interaction_cutoff() );
	} else if ( cpdata_map.n_entries() == 3 ) {
		EtableAtom at; CountPairData_1_3 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, atomic_interaction_cutoff() );
	} else {
		/// As of 10/21, all count pair data combinations should be covered. This code should not execute.
		std::cerr << "Unsupported number of residue connections in trie construction." << std::endl;
		utility_exit();
		return 0;
	}
}

template < class Derived >
void
BaseEtableEnergy< Derived >::indicate_required_context_graphs( utility::vector1< bool > & /*context_graphs_required*/ ) const
{
	//context_graphs_required[ ten_A_neighbor_graph ] = true; // when did this get here?
}

/*
template < class Derived >
EtableRotamerTrieOP
BaseEtableEnergy< Derived >::create_rotamer_trie_2(
	pack::rotamer_set::RotamerSet const & rotset
) const
{
	utility::vector1< RotamerDescriptor< EtableAtom, CountPairData_1_2 > > rotamer_descriptors( rotset.num_rotamers() );

	for ( Size ii = 1; ii <= rotset.num_rotamers(); ++ii ) {
		conformation::ResidueCOP ii_rotamer( rotset.rotamer( ii ) );
		rotamer_descriptors[ ii ].natoms( ii_rotamer->natoms() );
		Size count_added_atoms = 0;
		for ( Size jj = 1; jj <= ii_rotamer->nheavyatoms(); ++jj ) {

			EtableAtom newatom;
			newatom.xyz( ii_rotamer->atom(jj).xyz() );
			newatom.atom_type( ii_rotamer->atom(jj).type() );
			newatom.is_hydrogen( false );

			CountPairData_1_2 cpdata;
			cpdata.set_dist_to_connect_point(
				1, ii_rotamer->path_distance( jj )[ ii_rotamer->lower_connect_atom() ] );
			cpdata.set_dist_to_connect_point(
				2, ii_rotamer->path_distance( jj )[ ii_rotamer->upper_connect_atom() ] );

			RotamerDescriptorAtom< EtableAtom, CountPairData_1_2 > rdatom( newatom, cpdata );
			rotamer_descriptors[ ii ].atom( ++count_added_atoms, rdatom );

			for ( Size kk = ii_rotamer->attached_H_begin( jj ),
					kk_end = ii_rotamer->attached_H_end( jj );
					kk <= kk_end; ++kk ) {

				EtableAtom newhatom;
				newhatom.xyz( ii_rotamer->atom(kk).xyz() );
				newhatom.atom_type( ii_rotamer->atom(kk).type() );
				newhatom.is_hydrogen( true );

				CountPairData_1_2 cpdata;
				cpdata.set_dist_to_connect_point(
					1, ii_rotamer->path_distance( kk )[ ii_rotamer->lower_connect_atom() ] );
				cpdata.set_dist_to_connect_point(
					2, ii_rotamer->path_distance( kk )[ ii_rotamer->upper_connect_atom() ] );

				RotamerDescriptorAtom< EtableAtom, CountPairData_1_2 > rdatom( newhatom, cpdata );
				rotamer_descriptors[ ii ].atom( ++count_added_atoms, rdatom );

			}
			rotamer_descriptors[ ii ].rotamer_id( ii );
		}
	}

	sort( rotamer_descriptors.begin(), rotamer_descriptors.end() );

	return new RotamerTrie< EtableAtom, CountPairData_1_2 >( rotamer_descriptors, atomic_interaction_cutoff() );

}

// Andrew, refactor this!
template < class Derived >
EtableRotamerTrieOP
BaseEtableEnergy< Derived >::create_rotamer_trie_1(
	pack::rotamer_set::RotamerSet const & rotset,
	Size connection_type // HACK: 1 for lower connect, 2 for upper connect.  Replace this with an enum(?)
) const
{
	assert( connection_type == 1 || connection_type == 2 );
	utility::vector1< RotamerDescriptor< EtableAtom, CountPairData_1_1 > > rotamer_descriptors( rotset.num_rotamers() );

	for ( Size ii = 1; ii <= rotset.num_rotamers(); ++ii ) {
		conformation::ResidueCOP ii_rotamer( rotset.rotamer( ii ) );

		rotamer_descriptors[ ii ].natoms( ii_rotamer->natoms() );
		Size count_added_atoms = 0;
		for ( Size jj = 1; jj <= ii_rotamer->nheavyatoms(); ++jj ) {

			EtableAtom newatom;
			newatom.xyz( ii_rotamer->atom(jj).xyz() );
			newatom.atom_type( ii_rotamer->atom(jj).type() );
			newatom.is_hydrogen( false );

			CountPairData_1_1 cpdata;
			// pb mod, not sure...
			Size connect_atom( connection_type == 2 ? ii_rotamer->lower_connect_atom() : ii_rotamer->upper_connect_atom() );
			cpdata.set_dist_to_connect_point( 1, ii_rotamer->path_distance( jj )[ connect_atom ] );

			RotamerDescriptorAtom< EtableAtom, CountPairData_1_1 > rdatom( newatom, cpdata );
			rotamer_descriptors[ ii ].atom( ++count_added_atoms, rdatom );

			for ( Size kk = ii_rotamer->attached_H_begin( jj ),
				kk_end = ii_rotamer->attached_H_end( jj );
				kk <= kk_end; ++kk ) {

				EtableAtom newhatom;
				newhatom.xyz( ii_rotamer->atom(kk).xyz() );
				newhatom.atom_type( ii_rotamer->atom(kk).type() );
				newhatom.is_hydrogen( true );

				CountPairData_1_1 cpdata;

				cpdata.set_dist_to_connect_point( 1, ii_rotamer->path_distance( kk )[ connect_atom ] );

				RotamerDescriptorAtom< EtableAtom, CountPairData_1_1 > rdatom( newhatom, cpdata );
				rotamer_descriptors[ ii ].atom( ++count_added_atoms, rdatom );

			}
			rotamer_descriptors[ ii ].rotamer_id( ii );
		}
	}

	sort( rotamer_descriptors.begin(), rotamer_descriptors.end() );

	return new RotamerTrie< EtableAtom, CountPairData_1_1 >( rotamer_descriptors, atomic_interaction_cutoff());

}
*/

template < class Derived >
void
BaseEtableEnergy< Derived >::bump_energy_full(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap & emap
) const
{
	TwoBodyEnergyMap tbemap;
	prepare_for_residue_pair(rsd1.seqpos(),rsd2.seqpos(),pose);
	count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd1, rsd2, pose, sfxn );
	cpfxn->residue_atom_pair_energy_sidechain_whole( rsd1, rsd2, static_cast<Derived const&> (*this), tbemap );
	emap[ st_atr_ ] += tbemap[ st_atr_ ];
	emap[ st_rep_ ] += tbemap[ st_rep_ ];

}

template < class Derived >
void
BaseEtableEnergy< Derived >::bump_energy_backbone(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap & emap
) const
{
	TwoBodyEnergyMap tbemap;
	prepare_for_residue_pair(rsd1.seqpos(),rsd2.seqpos(),pose);
	count_pair::CountPairFunctionCOP cpfxn = get_count_pair_function( rsd1, rsd2, pose, sfxn );
	cpfxn->residue_atom_pair_energy_sidechain_backbone( rsd1, rsd2, static_cast<Derived const&> (*this), tbemap );
	emap[ st_atr_ ] += tbemap[ st_atr_ ];
	emap[ st_rep_ ] += tbemap[ st_rep_ ];

}


///////////////////////////////////////////////////////////////////////////////

template < class Derived >
void
BaseEtableEnergy< Derived >::eval_atom_derivative(
	id::AtomID const & id,
	pose::Pose const & pose,
	kinematics::DomainMap const &, // domain_map,
	ScoreFunction const & /*sfxn*/, // needed for non-nblist minimization
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	Size const idresid = id.rsd();
	conformation::Atom const & atom1
		( pose.residue( idresid ).atom( id.atomno() ) );

	if ( pose.energies().use_nblist() ) {
		scoring::AtomNeighbors const & nbrs
			( pose.energies().nblist( EnergiesCacheableDataType::ETABLE_NBLIST ).atom_neighbors( id ) );

		Vector f1,f2;
		for ( scoring::AtomNeighbors::const_iterator it2=nbrs.begin(),
						it2e=nbrs.end(); it2 != it2e; ++it2 ) {
			scoring::AtomNeighbor const & nbr( *it2 );

			// Intra residue weights separate from interresidue weights; set which weights to use before scoring.
			// Compairison between idresid and nb.rsd performed by EtableEnergy but not by
			// CoarseEtableEnergy -> both are passed as arguments
			// static_cast<Derived const&> (*this).decide_scoretypes( idresid, nbr.rsd() );
			prepare_for_residue_pair( idresid, nbr.rsd(), pose );

			Real const cp_weight( nbr.weight() );  // do not use nbr->weight_func() here
			conformation::Atom const & atom2
				( pose.residue( nbr.rsd() ).atom( nbr.atomno() ) );
			Real const dE_dR_over_r
				( eval_dE_dR_over_r( atom1, atom2, weights, f1, f2 ) );
			if ( dE_dR_over_r != 0.0 ) {
				F1 += dE_dR_over_r * cp_weight * f1;
				F2 += dE_dR_over_r * cp_weight * f2;
			}
		}
	} else {
		utility_exit_with_message("non-nblist minimize!");
	}
}

template < class Derived >
void
BaseEtableEnergy< Derived >::set_scoretypes(
	ScoreType atr_type,
	ScoreType rep_type,
	ScoreType sol_type
) const
{
	st_atr_ = atr_type;
	st_rep_ = rep_type;
	st_sol_ = sol_type;
}

template < class Derived >
ScoreType
BaseEtableEnergy< Derived >::rep_scoretype() const
{
	return st_rep_;
}

/// @brief return the Etables atomic distance cutoff
template < class Derived >
Distance
BaseEtableEnergy< Derived >::atomic_interaction_cutoff() const
{
	using namespace options;
	using namespace options::OptionKeys;
	return etable_.max_dis() +
		( option[ score::fa_Hatr ] ? chemical::MAX_CHEMICAL_BOND_TO_HYDROGEN_LENGTH * 2 : 0.0 ); /// HACK -- add in hydrogen/heavy max dist * 2
}



} // namespace etable
} // namespace scoring
} // namespace core


#endif
