// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet;
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @brief
/// @author jk

// Project Headers
#include <core/scoring/geometric_solvation/ExactOccludedHbondSolEnergy.hh>
#include <core/scoring/geometric_solvation/ExactOccludedHbondSolEnergyCreator.hh>

#include <core/init.hh>
#include <core/types.hh>
#include <core/io/pdb/pose_io.hh>
#include <core/id/AtomID.hh>
#include <core/id/AtomID_Map.hh>
#include <core/id/AtomID_Map.Pose.hh>
#include <core/pose/Pose.hh>
#include <core/chemical/util.hh>
#include <core/chemical/ChemicalManager.hh>
#include <core/chemical/ResidueTypeSet.hh>
#include <core/conformation/ResidueFactory.hh>
#include <core/conformation/Atom.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/etable/Etable.hh>
#include <core/scoring/etable/EtableOptions.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoringManager.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/TenANeighborGraph.hh>
#include <core/options/util.hh>
#include <core/util/Tracer.hh>

#include <core/scoring/hbonds/hbonds_geom.hh>
#include <core/scoring/hbonds/types.hh>
#include <core/scoring/hbonds/constants.hh>

#include <numeric/constants.hh>
#include <numeric/xyzVector.hh>
#include <numeric/xyzMatrix.hh>

//#include <core/scoring/ScoreFunction.hh>
//#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/options/option_macros.hh>

// Utility Headers
#include <utility/vector1.hh>
#include <utility/io/ozstream.hh>

// C++ Headers
#include <cmath>
#include <iostream>
#include <iomanip>
#include <map>
#include <vector>



//Vector dummy_res_energy_vector_;

static core::util::Tracer TR( "core.scoring.geometric_solvation.ExactOccludedHbondSolEnergy" );

namespace core {
namespace scoring {
namespace geometric_solvation {


/// @details This must return a fresh instance of the ExactOccludedHbondSolEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
ExactOccludedHbondSolEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new ExactOccludedHbondSolEnergy(
		options::option[ options::OptionKeys::score::exact_occ_pairwise ],
		options::option[ options::OptionKeys::score::exact_occ_split_between_res ] );

}

ScoreTypes
ExactOccludedHbondSolEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( occ_sol_exact );
	return sts;
}


using namespace core;
using namespace core::scoring;
using namespace core::scoring::hbonds;

core::Real const geosol_kT = { 0.593 };

// apply this weight to everything, so that scale will match LK
core::Real const LK_MATCHING_WEIGHT_EXACT = { 0.23968 };

GridInfo* GridInfo::instance_( 0 );
WaterWeightGridSet* WaterWeightGridSet::instance_( 0 );

//singleton class
GridInfo * GridInfo::get_instance()
{
	if ( instance_ == 0 )
	{
		 instance_ = new GridInfo();
	}
	return instance_;
}


// private constructor
GridInfo::GridInfo() {

	// Setup dimensions for water grids - use the same dimensions for all grids
	core::Real const water_grid_width = 10.;
	core::Real const water_grid_depth = 8.;
	// For speed use only 52 thousand points per grid - gives identical results...
	xnum_points_ = 41;
	ynum_points_ = 41;
	znum_points_ = 31;
	// Note: below gives 51 million points per grid
		//	xnum_points_ = 401;
		//	ynum_points_ = 401;
		//	znum_points_ = 321;

	xstep_ = water_grid_width / ( xnum_points_ - 1 );
	ystep_ = water_grid_width / ( ynum_points_ - 1 );
	zstep_ = water_grid_depth / ( znum_points_ - 1 );
	// Note: the point at the origin will NOT be considered in calculations - the grid starts AFTER the origin!!
	xorigin_ = -xstep_ * ( xnum_points_ + 1) / 2;
	yorigin_ = -ystep_ * ( ynum_points_ + 1) / 2;
	zorigin_ = 0;

}


//singleton class
WaterWeightGridSet * WaterWeightGridSet::get_instance()
{
	if ( instance_ == 0 )
	{
		 instance_ = new WaterWeightGridSet();
	}
	return instance_;
}

// private constructor
WaterWeightGridSet::WaterWeightGridSet() {

	// Build a template pose, to use for measuring bond lengths and angles...
	pose::Pose template_pose;
	chemical::make_pose_from_sequence( template_pose, "ASHNCDEGKMPQRTWY", *( chemical::ChemicalManager::get_instance()->residue_type_set( chemical::FA_STANDARD )));
	//	core::Size const ala_resnum = 1; // note: can't use this template for NH, since it's at the terminus...
	core::Size const ser_resnum = 2;
	core::Size const his_resnum = 3;
	core::Size const asn_resnum = 4;

	// We need four water grids (donor, acceptor O-sp2, acceptor O-sp3, acceptor N-his)
	// We could read them in, but let's just compute them from scratch instead
	// We'll store them as a map keyed on hbonds::HBEvalType, with each value a 3D vector of the weights
	core::Vector outer_atom_position(0,0,0);  // note: this is the proton for a donor, or the oxygen for an acceptor
	core::Vector base_atom_position(0,0,0);

	{ // scope setup backbone donor map
		TR << "computing and storing backbone donor water weight grid" << std::endl;
		bool const water_is_donor = false;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("H") , ser_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("N") , ser_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_SP3, hbdon_BBN );
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

	{ // scope setup non-backbone donor map
		TR << "computing and storing non-backbone donor water weight grid" << std::endl;
		bool const water_is_donor = false;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("HG") , ser_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("OG") , ser_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_SP3, hbdon_SC );
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

	{ // scope setup backbone acceptor map
		TR << "computing and storing backbone acceptor water weight grid" << std::endl;
		bool const water_is_donor = true;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("O") , ser_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("C") , ser_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_BB, hbdon_SC);
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

	{ // scope setup acceptor O-sp2 map
		TR << "computing and storing O-sp2 water weight grid" << std::endl;
		bool const water_is_donor = true;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( asn_resnum ).atom_index("OD1") , asn_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( asn_resnum ).atom_index("CG") , asn_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_SP2, hbdon_SC);
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

	{ // scope setup acceptor O-sp3 map
		TR << "computing and storing O-sp3 water weight grid" << std::endl;
		bool const water_is_donor = true;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("OG") , ser_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( ser_resnum ).atom_index("CB") , ser_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_SP3, hbdon_SC);
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

	{ // scope setup acceptor N-his map
		TR << "computing and storing N-his water weight grid" << std::endl;
		bool const water_is_donor = true;
		id::AtomID const outer_atom_id = id::AtomID( template_pose.residue( his_resnum ).atom_index("ND1") , his_resnum );
		id::AtomID const base_atom_id = id::AtomID( template_pose.residue( his_resnum ).atom_index("CG") , his_resnum );
		base_atom_position.z() = - template_pose.conformation().bond_length( base_atom_id, outer_atom_id );
		hbonds::HBEvalType const hbond_eval_type = hbonds::HBeval_lookup( hbacc_RING, hbdon_SC);
		sum_all_water_weights_[ hbond_eval_type ] = fill_water_grid( all_water_weights_[ hbond_eval_type ], hbond_eval_type,
			*GridInfo::get_instance(), water_is_donor, outer_atom_position, base_atom_position );
	}

}



// Fill in the water grid
core::Real WaterWeightGridSet::fill_water_grid(
	std::vector < std::vector < std::vector <core::Real> > > & water_weights,
	hbonds::HBEvalType const & hbond_eval_type,
	GridInfo const & grid_info, bool const water_is_donor,
	core::Vector const & outer_atom_position, core::Vector const & base_atom_position )
{

	core::Real const entropy_scaling = 1.0;
	core::Real const water_O_H_distance = 0.958;

	// Setup grid, initialize to zero
	water_weights.clear();
	water_weights.resize(grid_info.xnum_points());
	for (core::Size tx=0;tx<grid_info.xnum_points();tx++){
		water_weights[tx].resize(grid_info.ynum_points());
		for (core::Size ty=0;ty<grid_info.ynum_points();ty++){
			water_weights[tx][ty].resize(grid_info.znum_points(), 0.);
		}
	}

	// Fill in the water weight grid
	core::Real sum_grid_water_weight = 0.;
	core::Vector water_position(grid_info.xorigin(),grid_info.yorigin(),grid_info.zorigin());

	for (core::Size tx=0;tx<grid_info.xnum_points();tx++){
		water_position.x() += grid_info.xstep();
		water_position.y() = grid_info.yorigin();
		for (core::Size ty=0;ty<grid_info.ynum_points();ty++){
			water_position.y() += grid_info.ystep();
			water_position.z() = grid_info.zorigin();
			for (core::Size tz=0;tz<grid_info.znum_points();tz++){
				water_position.z() += grid_info.zstep();

				// Compute the current geometry
				core::Real AHdis, xD, xH;
				if ( water_is_donor ) {

					// water is the donor, give it perfect geometry
					xD = 0.9999;

					// compute the distance to the accepting water proton
					// subtract the water's OH distance to get the AHdis,
					// since the distance computed was from the acceptor to the water oxygen
					// note: water proton lies on the line between the acceptor and the water oxygen
					AHdis = distance ( outer_atom_position, water_position );
					AHdis -= water_O_H_distance; // water O-H distance

					// find cosine of the base-acceptor-water_proton angle (xH)
					// note: this is the same as the base-acceptor-water_oxygen angle
					xH = dot( (outer_atom_position - base_atom_position).normalize(),  (water_position - outer_atom_position).normalize() );

				} else {

					// water is the acceptor, give it perfect geometry
					xH = 1./3.;  // perfect geometry is cos( 180 - 109.5 degrees), which is 1/3

					// compute the distance to the accepting water
					AHdis = distance ( outer_atom_position, water_position );

					// find the cosine of the base-proton-water angle (xD)
					xD = dot( (outer_atom_position - base_atom_position).normalize(),  (water_position - outer_atom_position).normalize() );

				}

				if ( xH < MIN_xH ) continue;
				if ( xH > MAX_xH ) continue;
				if ( xD < MIN_xD ) continue;
				if ( xD > MAX_xD ) continue;
				if ( AHdis < MIN_R ) continue;
				if ( AHdis > MAX_R ) continue;

				// Get the Hbond energy
				core::Real curr_water_hbond;
				core::scoring::hbonds::hbond_compute_energy(hbond_eval_type, AHdis, xD, xH, curr_water_hbond);

				// Save the Hbond energy
				curr_water_hbond *= entropy_scaling;
				if ( curr_water_hbond < 0 ) {
					core::Real curr_water_weight = exp( - curr_water_hbond / geosol_kT );
					water_weights[tx][ty][tz] = curr_water_weight;
					sum_grid_water_weight += curr_water_weight;
				}
			}
		}
	}

	return sum_grid_water_weight;
}


std::vector < std::vector < std::vector <core::Real> > > const & WaterWeightGridSet::get_water_weight_grid( hbonds::HBEvalType const & hbond_eval_type ) const {
	// Check that we have weights for this Hbond type
	all_water_weights_iterator curr_water_weights_iter = all_water_weights_.find( hbond_eval_type );
	if ( curr_water_weights_iter == all_water_weights_.end( ) ) {
		TR << "Could not look map element" << std::endl;
		exit(1);
	}
	return curr_water_weights_iter->second;
}

core::Real WaterWeightGridSet::get_sum_water_weight_grid( hbonds::HBEvalType const & hbond_eval_type ) const {
	// Check that we have weights for this Hbond type
	sum_water_weights_iterator curr_sum_water_weights_iter = sum_all_water_weights_.find( hbond_eval_type );
	if ( curr_sum_water_weights_iter == sum_all_water_weights_.end( ) ) {
		TR << "Could not look map element" << std::endl;
		exit(1);
	}
	return curr_sum_water_weights_iter->second;
}


ExactOccludedHbondSolEnergy::ExactOccludedHbondSolEnergy(
	bool const exact_occ_pairwise,
	bool const exact_occ_split_between_res,
	bool const verbose
) :
	parent( new ExactOccludedHbondSolEnergyCreator ),
	exact_occ_pairwise_( exact_occ_pairwise ),
	exact_occ_split_between_res_( exact_occ_split_between_res ),
	verbose_( verbose )
{
	if ( verbose_ ) TR <<"ExactOccludedHbondSolEnergy constructor" << std::endl;
	if ( exact_occ_split_between_res_ && ! exact_occ_pairwise_ ) {
		TR << "Error - cannot split occ energy between residues unless pairwise calculations are used!" << std::endl;
		exit(1);
	}
	if ( verbose_ && exact_occ_pairwise_ ) TR <<"exact_occ_pairwise is ON" << std::endl;
	if ( verbose_ && exact_occ_split_between_res_ ) TR <<"exact_occ_split_between_res is ON" << std::endl;

	// Keep a copy of the Etable (to lookup atomic radii)
	etable_ptr_ = new core::scoring::etable::Etable( chemical::ChemicalManager::get_instance()->atom_type_set( chemical::FA_STANDARD ), core::scoring::etable::EtableOptions() );

	// Allocate memory for grid of occluded sites
	occluded_sites_.clear();
	occluded_sites_.resize(GridInfo::get_instance()->xnum_points());
	for (core::Size tx=0;tx<GridInfo::get_instance()->xnum_points();tx++){
		occluded_sites_[tx].resize(GridInfo::get_instance()->ynum_points());
		for (core::Size ty=0;ty<GridInfo::get_instance()->ynum_points();ty++){
			occluded_sites_[tx][ty].resize(GridInfo::get_instance()->znum_points());
		}
	}

}

ExactOccludedHbondSolEnergy::ExactOccludedHbondSolEnergy( ExactOccludedHbondSolEnergy const & src ):
	parent( src ),
	exact_occ_pairwise_( src.exact_occ_pairwise_ ),
	exact_occ_split_between_res_( src.exact_occ_split_between_res_ ),
	verbose_( src.verbose_ ),
	etable_ptr_( src.etable_ptr_ )
{
	if ( verbose_ ) TR <<"ExactOccludedHbondSolEnergy constructor" << std::endl;
	if ( exact_occ_split_between_res_ && ! exact_occ_pairwise_ ) {
		TR << "Error - cannot split occ energy between residues unless pairwise calculations are used!" << std::endl;
		exit(1);
	}
	if ( verbose_ && exact_occ_pairwise_ ) TR <<"exact_occ_pairwise is ON" << std::endl;
	if ( verbose_ && exact_occ_split_between_res_ ) TR <<"exact_occ_split_between_res is ON" << std::endl;

	// Allocate memory for grid of occluded sites
	occluded_sites_.clear();
	occluded_sites_.resize(GridInfo::get_instance()->xnum_points());
	for (core::Size tx=0;tx<GridInfo::get_instance()->xnum_points();tx++){
		occluded_sites_[tx].resize(GridInfo::get_instance()->ynum_points());
		for (core::Size ty=0;ty<GridInfo::get_instance()->ynum_points();ty++){
			occluded_sites_[tx][ty].resize(GridInfo::get_instance()->znum_points());
		}
	}

}

methods::EnergyMethodOP
ExactOccludedHbondSolEnergy::clone() const
{
	return new ExactOccludedHbondSolEnergy( *this );
}

void
ExactOccludedHbondSolEnergy::setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const
{
	pose.update_residue_neighbors();
}

void
ExactOccludedHbondSolEnergy::setup_for_packing( pose::Pose & pose, ScoreFunction const & ) const
{
	pose.update_residue_neighbors();
}

void
ExactOccludedHbondSolEnergy::setup_for_derivatives( pose::Pose & , ScoreFunction const & ) const
{
	TR << "Error - cannot compute derivatives for ExactOccludedHbondSolEnergy (occ_sol_exact)" << std::endl;
	exit(1);
}

void
ExactOccludedHbondSolEnergy::setup_for_minimizing( pose::Pose & , ScoreFunction const & , optimization::MinimizerMap const & ) const
{
	TR << "Error - cannot compute derivatives for ExactOccludedHbondSolEnergy (occ_sol_exact)" << std::endl;
	exit(1);
}

Distance ExactOccludedHbondSolEnergy::atomic_interaction_cutoff() const { return 7.5; }

void ExactOccludedHbondSolEnergy::residue_energy(
	conformation::Residue const & polar_rsd,
	pose::Pose const & pose,
	EnergyMap & emap
) const {

	// note: the "restrict_to_single_occluding_residue / atom" options make the code run twice as slow as it otherwise would,
	// but that's okay because they're just for parameterization / analysis

	core::Size polar_resnum = polar_rsd.seqpos();
	core::Real residue_geosol(0.);

	// loop over donors in polar_rsd
	for ( chemical::AtomIndices::const_iterator
					hnum  = polar_rsd.Hpos_polar().begin(),
					hnume = polar_rsd.Hpos_polar().end(); hnum != hnume; ++hnum ) {
		Size const polar_atom( *hnum );
		Size const base_atom( polar_rsd.atom_base( polar_atom ) );
		hbonds::HBEvalType const curr_hbond_eval_type = hbonds::HBeval_lookup( hbacc_SP3, get_hb_don_chem_type( polar_atom, polar_rsd ));

		// Figure out max LK energy
		std::string const base_atom_name = polar_rsd.atom_name( base_atom );
		core::Real max_possible_LK = etable_ptr_->lk_dgfree( polar_rsd.atom_type_index( base_atom ) );
		if ( ( base_atom_name == " N  " ) && polar_rsd.is_lower_terminus() ) max_possible_LK /= 3; // charged N-terminus
		if ( base_atom_name == " NZ " ) max_possible_LK /= 3; // Lys
		if ( base_atom_name == " ND2" ) max_possible_LK /= 2; // Asn
		if ( base_atom_name == " NE2" ) max_possible_LK /= 2; // Gln
		if ( base_atom_name == " NH1" ) max_possible_LK /= 2; // Arg
		if ( base_atom_name == " NH2" ) max_possible_LK /= 2; // Arg
		// Note: inner nitrogen of Arg (NE) is extra strong, since it's the same atom type as the other two but doesn't get
		// cut in half because there's only one proton...
		//			TR << "jk max LK for donor with base " << base_atom_name << " is  " << max_possible_LK << std::endl;

		// Compute Ebulk (using the LK energy)
		core::Real const Emax_weight = exp( max_possible_LK / geosol_kT );
		core::Real const sum_water_weights = WaterWeightGridSet::get_instance()->get_sum_water_weight_grid( curr_hbond_eval_type );
		core::Real const Ebulk_weight = ( sum_water_weights * Emax_weight ) / ( 1. - Emax_weight);
		// This grid constant is the denominator in computing solvation energies,
		// it depends on the grid dimensions, and sets the max possible solvation energy (in this case to match LK)
		core::Real const grid_constant = sum_water_weights + Ebulk_weight;

		if ( ! exact_occ_pairwise_ ) {
			residue_geosol += compute_polar_group_sol_energy(pose, polar_resnum, polar_atom, *GridInfo::get_instance(),
				grid_constant, WaterWeightGridSet::get_instance()->get_water_weight_grid( curr_hbond_eval_type ) );

		} else {
			// loop over all atoms of neighboring residues, INCLUDING SELF
			core::scoring::TenANeighborGraph const & graph = pose.energies().tenA_neighbor_graph();
			utility::vector1 <core::Size> neighborlist;
			neighborlist.push_back( polar_resnum);
			for ( core::graph::Graph::EdgeListConstIter
							neighbor_iter = graph.get_node( polar_resnum )->const_edge_list_begin(),
							neighbor_iter_end = graph.get_node( polar_resnum )->const_edge_list_end();
						neighbor_iter != neighbor_iter_end; ++neighbor_iter ) {
				neighborlist.push_back( (*neighbor_iter)->get_other_ind( polar_resnum ) );
			}
			for ( Size occ_inx = 1; occ_inx <= neighborlist.size(); ++occ_inx ) {
				core::Size const occ_resnum( neighborlist[occ_inx] );
				conformation::Residue const occ_rsd = pose.residue(occ_resnum);
				for ( Size occ_atomno = 1; occ_atomno <= occ_rsd.natoms(); ++occ_atomno ) {
					residue_geosol += compute_polar_group_sol_energy(pose, polar_resnum, polar_atom, *GridInfo::get_instance(),
						grid_constant, WaterWeightGridSet::get_instance()->get_water_weight_grid( curr_hbond_eval_type ),
						true, occ_resnum, true, occ_atomno );
				}
			}
		}

	}

	// loop over acceptors in polar_rsd
	for ( chemical::AtomIndices::const_iterator
					anum  = polar_rsd.accpt_pos().begin(),
					anume = polar_rsd.accpt_pos().end(); anum != anume; ++anum ) {
		Size const polar_atom( *anum );
		Size const base_atom ( polar_rsd.atom_base( polar_atom ) );
		hbonds::HBEvalType const curr_hbond_eval_type = hbonds::HBeval_lookup( get_hb_acc_chem_type( polar_atom, polar_rsd ), hbdon_SC);

		// Figure out max LK energy
		std::string const base_atom_name = polar_rsd.atom_name( base_atom );
		core::Real max_possible_LK = etable_ptr_->lk_dgfree( polar_rsd.atom_type_index( polar_atom ) );
		//			TR << "jk max LK for acceptor " << polar_rsd.atom_name(polar_atom) << " is  " << max_possible_LK << std::endl;
		// Compute Ebulk (using the LK energy)
		core::Real const Emax_weight = exp( max_possible_LK / geosol_kT );
		core::Real const sum_water_weights = WaterWeightGridSet::get_instance()->get_sum_water_weight_grid( curr_hbond_eval_type );
		core::Real const Ebulk_weight = ( sum_water_weights * Emax_weight ) / ( 1. - Emax_weight);
		// This grid constant is the denominator in computing solvation energies,
		// it depends on the grid dimensions, and sets the max possible solvation energy (in this case to match LK)
		core::Real const grid_constant = sum_water_weights + Ebulk_weight;

		if ( ! exact_occ_pairwise_ ) {
			residue_geosol += compute_polar_group_sol_energy(pose, polar_resnum, polar_atom, *GridInfo::get_instance(),
				grid_constant, WaterWeightGridSet::get_instance()->get_water_weight_grid( curr_hbond_eval_type ) );

		} else {
			// loop over all atoms of neighboring residues, INCLUDING SELF
			core::scoring::TenANeighborGraph const & graph = pose.energies().tenA_neighbor_graph();
			utility::vector1 <core::Size> neighborlist;
			neighborlist.push_back( polar_resnum);
			for ( core::graph::Graph::EdgeListConstIter
							neighbor_iter = graph.get_node( polar_resnum )->const_edge_list_begin(),
							neighbor_iter_end = graph.get_node( polar_resnum )->const_edge_list_end();
						neighbor_iter != neighbor_iter_end; ++neighbor_iter ) {
				neighborlist.push_back( (*neighbor_iter)->get_other_ind( polar_resnum ) );
			}
			for ( Size occ_inx = 1; occ_inx <= neighborlist.size(); ++occ_inx ) {
				core::Size const occ_resnum( neighborlist[occ_inx] );
				conformation::Residue const occ_rsd = pose.residue(occ_resnum);
				for ( Size occ_atomno = 1; occ_atomno <= occ_rsd.natoms(); ++occ_atomno ) {
					residue_geosol += compute_polar_group_sol_energy(pose, polar_resnum, polar_atom, *GridInfo::get_instance(),
						grid_constant, WaterWeightGridSet::get_instance()->get_water_weight_grid( curr_hbond_eval_type ),
						true, occ_resnum, true, occ_atomno );
				}
			}
		}

	}

	if ( exact_occ_split_between_res_ ) {
		TR << "PAIRWISE OUTPUT FORMAT IS NOT YET SUPPORTED" << std::endl;
		// Here we need to add contributions from this residue occluding all neighboring polar groups then divide everything by two
		exit(1);
	}

	emap[ occ_sol_exact ] = LK_MATCHING_WEIGHT_EXACT * residue_geosol;

}



core::Real ExactOccludedHbondSolEnergy::compute_polar_group_sol_energy(
	pose::Pose const & pose,
	core::Size const polar_resnum,
	core::Size const polar_atomno,
	GridInfo const & grid_info,
	core::Real const & grid_constant,
	std::vector < std::vector < std::vector <core::Real> > > const & water_weights,
	bool const restrict_to_single_occluding_residue, // = false
	core::Size const single_occluding_resinx, // = 0
	bool const restrict_to_single_occluding_atom, // = false
	core::Size const single_occluding_atominx // = 0
) const {

	// note: the "restrict_to_single_occluding_residue / atom" options are so that pairwise additivity can be enforced (for parameterization / analysis)

	bool const hydrogens_can_occlude = false;
	core::Real const water_radius = 1.4;

	// Reset grid of occluded sites, by setting everything to false (for "not occluded")
	for (core::Size tx=0;tx<grid_info.xnum_points();tx++){
		for (core::Size ty=0;ty<grid_info.ynum_points();ty++){
			for (core::Size tz=0;tz<grid_info.znum_points();tz++){
				occluded_sites_[tx][ty][tz] = false;
			}
		}
	}

	// Find the transformation which puts the donor/acceptor of interest onto the existing grid
	// The plan is to apply this transformation to bring each occluding atom onto the existing grid (translation then matrix multiplication)
	core::Size const base_atomno( pose.residue(polar_resnum).atom_base( polar_atomno ) );
	core::Vector const & orig_polar_atom_xyz( pose.residue(polar_resnum).atom( polar_atomno ).xyz() );
	core::Vector const & orig_base_atom_xyz( pose.residue(polar_resnum).atom( base_atomno ).xyz() );
	core::Vector translation_vector = -1.0 * orig_polar_atom_xyz;
	core::Vector translated_base_atom = orig_base_atom_xyz + translation_vector;

	// We want to translate positions of occluding atoms _from_ a cartesian basis set into one using the reference frame of the polar group
	core::Vector cartesian_x(1,0,0);
	core::Vector cartesian_y(0,1,0);
	core::Vector cartesian_z(0,0,1);

	// There's not a unique solution for this, so we'll arbitrarily pick a second basis vector, requiring that the dot product with desired_z is zero
	core::Vector desired_z = -1. * translated_base_atom.normalized();

	// Treat the case where polar_atom.z == base_atom.z ; this leads to desired_z.z == 0
	core::Real arbitrary_x, arbitrary_y, arbitrary_z;
	if ( std::abs( desired_z.z() ) > 0.01 ) {
		arbitrary_x = 1;
		arbitrary_y = 1;
		arbitrary_z = -1. * ((arbitrary_x * desired_z.x()) + (arbitrary_y * desired_z.y())) / desired_z.z();
	} else {
		arbitrary_x = 1;
		arbitrary_z = 1;
		arbitrary_y = -1. * ((arbitrary_x * desired_z.x()) + (arbitrary_z * desired_z.z())) / desired_z.y();
	}
	core::Vector desired_x(arbitrary_x,arbitrary_y,arbitrary_z);
	desired_x.normalize();
	core::Vector desired_y = cross_product( desired_x, desired_z );

	// The transformation matrix to do this is i.i'  i.j', etc. where i,j,k are the unit vectors of the starting system
	// and i',j',k' are the unit vectors of the target system
	// for reference see http://kwon3d.com/theory/transform/transform.html
	numeric::xyzMatrix< Length > transformation_matrix;
	transformation_matrix.xx( desired_x.dot(cartesian_x) );
	transformation_matrix.xy( desired_x.dot(cartesian_y) );
	transformation_matrix.xz( desired_x.dot(cartesian_z) );
	transformation_matrix.yx( desired_y.dot(cartesian_x) );
	transformation_matrix.yy( desired_y.dot(cartesian_y) );
	transformation_matrix.yz( desired_y.dot(cartesian_z) );
	transformation_matrix.zx( desired_z.dot(cartesian_x) );
	transformation_matrix.zy( desired_z.dot(cartesian_y) );
	transformation_matrix.zz( desired_z.dot(cartesian_z) );

	// Double-check transformation matrix
	core::Vector new_base_atom_location = transformation_matrix * translated_base_atom;
	assert( std::abs(new_base_atom_location.normalized().x()) < 0.001 );
	assert( std::abs(new_base_atom_location.normalized().y()) < 0.001 );
	assert( std::abs(new_base_atom_location.normalized().z() + 1.) < 0.001 );

	utility::vector1 <core::Size> neighborlist;
	if ( restrict_to_single_occluding_residue ) {
		// consider only a single occluding residue (for pairwise additivity)
		assert ( single_occluding_resinx > 0 );
		neighborlist.push_back( single_occluding_resinx );
	} else {
		// loop over all atoms of neighboring residues, INCLUDING SELF
		core::scoring::TenANeighborGraph const & graph = pose.energies().tenA_neighbor_graph();
		neighborlist.push_back( polar_resnum);
		for ( core::graph::Graph::EdgeListConstIter
						neighbor_iter = graph.get_node( polar_resnum )->const_edge_list_begin(),
						neighbor_iter_end = graph.get_node( polar_resnum )->const_edge_list_end();
					neighbor_iter != neighbor_iter_end; ++neighbor_iter ) {
			neighborlist.push_back( (*neighbor_iter)->get_other_ind( polar_resnum ) );
		}
	}

	for ( Size occ_inx = 1; occ_inx <= neighborlist.size(); ++occ_inx ) {
		core::Size const occ_resnum( neighborlist[occ_inx] );
		conformation::Residue const occ_rsd = pose.residue(occ_resnum);

		// loop over all occluding atoms in this residue
		Size atom_startinx = 1;
		Size atom_lastinx = occ_rsd.natoms();
		if ( restrict_to_single_occluding_atom ) {
			// consider only a single occluding atom (for pairwise additivity)
			assert ( single_occluding_atominx > 0 );
			atom_startinx = single_occluding_atominx;
			atom_lastinx = single_occluding_atominx;
		}

		//		TR << "computing occlusion of polar residue " << polar_resnum << " by residue " << occ_resnum << std::endl;

		for ( Size occ_atomno = atom_startinx; occ_atomno <= atom_lastinx; ++occ_atomno ) {

			bool const occ_atom_is_hydrogen = occ_rsd.atom_is_hydrogen( occ_atomno );
			if ( occ_atom_is_hydrogen && ! hydrogens_can_occlude ) continue;

			// can be occluded by atoms directly bonded to this group, but not by self
			if ( polar_resnum == occ_resnum ) {
				if ( polar_atomno == occ_atomno ) continue;
				if ( base_atomno == occ_atomno ) continue;
			}

			core::Real const occ_radius = etable_ptr_->lj_radius( occ_rsd.atom_type_index( occ_atomno ) );
			// catch proline NV here (and other virtual atoms, etc.)
			if ( occ_radius < 0.1 ) continue;
			core::Real const sq_dist_cut = ( occ_radius + water_radius ) * ( occ_radius + water_radius );

			// Apply the transformation to put this atom onto the current grid
			core::Vector const & orig_occ_atom_xyz( occ_rsd.atom( occ_atomno ).xyz() );
			core::Vector const translated_occ_atom_xyz = orig_occ_atom_xyz + translation_vector;
			core::Vector const transformed_occ_atom_xyz = transformation_matrix * ( orig_occ_atom_xyz + translation_vector );

			// Double-check transformations
			assert( std::abs(distance( orig_polar_atom_xyz, orig_occ_atom_xyz ) - transformed_occ_atom_xyz.magnitude()) < 0.001 );
			assert( std::abs(distance( orig_base_atom_xyz, orig_occ_atom_xyz ) - distance( new_base_atom_location, transformed_occ_atom_xyz )) < 0.001 );

			// Loop over all water positions, mark those which are occluded by this atom
			core::Vector water_position(grid_info.xorigin(),grid_info.yorigin(),grid_info.zorigin());
			for (core::Size wx=0;wx<grid_info.xnum_points();wx++){
				water_position.x() += grid_info.xstep();
				core::Real sq_xdist = ( water_position.x() - transformed_occ_atom_xyz.x() ) * ( water_position.x() - transformed_occ_atom_xyz.x() );
				if ( sq_xdist > sq_dist_cut ) continue;
				water_position.y() = grid_info.yorigin();
				for (core::Size wy=0;wy<grid_info.ynum_points();wy++){
					water_position.y() += grid_info.ystep();
					core::Real sq_ydist = ( water_position.y() - transformed_occ_atom_xyz.y() ) * ( water_position.y() - transformed_occ_atom_xyz.y() );
					if ( sq_ydist > sq_dist_cut ) continue;
					water_position.z() = grid_info.zorigin();
					for (core::Size wz=0;wz<grid_info.znum_points();wz++){
						water_position.z() += grid_info.zstep();
						core::Real sq_zdist = ( water_position.z() - transformed_occ_atom_xyz.z() ) * ( water_position.z() - transformed_occ_atom_xyz.z() );
						if ( sq_zdist > sq_dist_cut ) continue;
						core::Real sq_curr_dist = sq_xdist + sq_ydist + sq_zdist;
						if ( sq_curr_dist < sq_dist_cut ) {
							// this atom occludes this water site
							occluded_sites_[wx][wy][wz] = true;
						}
					}
				}
			}

		}
	}

	// Compute and store the solvation energy for the grid occluded by all nearby atoms
	// Compute the numerator (the sum of occluded weights)
	core::Real sum_occluded_weights(0.);
	for (core::Size tx=0;tx<grid_info.xnum_points();tx++){
		for (core::Size ty=0;ty<grid_info.ynum_points();ty++){
			for (core::Size tz=0;tz<grid_info.znum_points();tz++){
				if ( occluded_sites_[tx][ty][tz] ) {
					core::Real const curr_water_weight = water_weights[tx][ty][tz];
					sum_occluded_weights += curr_water_weight;
				}
			}
		}
	}
	// Return the energetic cost of occluding this polar group
	core::Real const geometric_solvation_energy = - geosol_kT * log( 1 - ( sum_occluded_weights / grid_constant ) );
	return geometric_solvation_energy;

}


} // geometric_solvation
} // scoring
} // core

