// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/HBondEnergy.fwd.hh
/// @brief  Hydrogen bond energy method forward declaration
/// @author Phil Bradley
/// @author Andrew Leaver-Fay



// Unit Headers
#include <core/scoring/hbonds/HBondEnergy.hh>
#include <core/scoring/hbonds/HBondEnergyCreator.hh>

// Package headers
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergiesCacheableDataType.hh>
#include <core/scoring/hbonds/HBondSet.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/hbonds/hbonds_geom.hh>

#include <core/scoring/hbonds/hbtrie/HBAtom.hh>
#include <core/scoring/hbonds/hbtrie/HBCPData.hh>
#include <core/scoring/hbonds/hbtrie/HBondTrie.fwd.hh>
#include <core/scoring/hbonds/hbtrie/HBCountPairFunction.hh>

#include <core/scoring/trie/TrieCollection.hh>
#include <core/scoring/trie/RotamerTrieBase.hh>
#include <core/scoring/trie/RotamerTrie.hh>
#include <core/scoring/trie/RotamerDescriptor.hh>

#include <core/scoring/methods/EnergyMethodOptions.hh>

//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/TenANeighborGraph.hh>

// Project headers
#include <core/pack/rotamer_set/RotamerSetFactory.hh>
#include <core/pack/rotamer_set/RotamerSet.hh>

#include <core/pose/Pose.hh>


namespace core {
namespace scoring {
namespace hbonds {


/// @details This must return a fresh instance of the HBondEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
HBondEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const & options
) const {
	return new HBondEnergy( options );
}

ScoreTypes
HBondEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( hbond_lr_bb );
	sts.push_back( hbond_sr_bb );
	sts.push_back( hbond_bb_sc );
	sts.push_back( hbond_sr_bb_sc );
	sts.push_back( hbond_lr_bb_sc );
	sts.push_back( hbond_sc );
	return sts;
}


/// ctor
HBondEnergy::HBondEnergy( methods::EnergyMethodOptions const & options ):
	parent( new HBondEnergyCreator ),
	exclude_DNA_DNA_   ( options.exclude_DNA_DNA() ),
	use_hb_env_dep_    ( options.use_hb_env_dep() ),
	use_hb_env_dep_DNA_( options.use_hb_env_dep_DNA() ),
	smooth_hb_env_dep_ ( options.smooth_hb_env_dep() ),
	decompose_bb_hb_into_pair_energies_( options.decompose_bb_hb_into_pair_energies() )
{}


/// copy ctor
HBondEnergy::HBondEnergy( HBondEnergy const & src ):
	parent( src ),
	exclude_DNA_DNA_   ( src.exclude_DNA_DNA_ ),
	use_hb_env_dep_    ( src.use_hb_env_dep_ ),
	use_hb_env_dep_DNA_( src.use_hb_env_dep_DNA_ ),
	smooth_hb_env_dep_ ( src.smooth_hb_env_dep_ ),
	decompose_bb_hb_into_pair_energies_( src.decompose_bb_hb_into_pair_energies_ )
{}


/// clone
methods::EnergyMethodOP
HBondEnergy::clone() const
{
	return new HBondEnergy( *this );
}

///
void
HBondEnergy::setup_for_packing( pose::Pose & pose, pack::task::PackerTask const & ) const
{
	using EnergiesCacheableDataType::HBOND_SET;
	using EnergiesCacheableDataType::HBOND_TRIE_COLLECTION;

	pose.update_residue_neighbors();
	hbonds::HBondSetOP hbond_set( new hbonds::HBondSet() );
	hbond_set->exclude_DNA_DNA   ( exclude_DNA_DNA_ );
	hbond_set->use_hb_env_dep    ( use_hb_env_dep_ );
	hbond_set->use_hb_env_dep_DNA( use_hb_env_dep_DNA_ );
	hbond_set->smooth_hb_env_dep ( smooth_hb_env_dep_ );
	hbond_set->setup_for_residue_pair_energies( pose );
	pose.energies().data().set( HBOND_SET, hbond_set );

	using namespace pack::rotamer_set;
	using namespace trie;
	using namespace hbtrie;

	TrieCollectionOP tries = new TrieCollection;
	tries->total_residue( pose.total_residue() );
	for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
		// Do not compute energy for virtual residues.
		if ( pose.residue(ii).aa() == core::chemical::aa_vrt) continue;

		// something of a hack: create a residue set, put inside it a single residue,
		// and then use the existing trie creation code -- single residue tries are
		// constructed with the same logic as are multiple-residue tries.
		RotamerSetFactory rsf;
		RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( ii ) );
		one_rotamer_set->set_resid( ii ); // change the interface so that the rotset initializes its seqpos at creation
		one_rotamer_set->add_rotamer( pose.residue( ii ) );
		HBondRotamerTrieOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );
		tries->trie( ii, one_rotamer_trie );
	}
	pose.energies().data().set( HBOND_TRIE_COLLECTION, tries );
}

void
HBondEnergy::prepare_rotamers_for_packing(
	pose::Pose const & pose,
	pack::rotamer_set::RotamerSet & set
) const
{
	using namespace hbtrie;

	HBondRotamerTrieOP rottrie = create_rotamer_trie( set, pose );
	//std::cout << "--------------------------------------------------" << std::endl << " HBROTAMER TRIE: " << set.resid()  << std::endl;
	//rottrie->print();
	set.store_trie( methods::hbond_method, rottrie );
}

// Updates the cached rotamer trie for a residue if it has changed during the course of
// a repacking
void
HBondEnergy::update_residue_for_packing( pose::Pose & pose, Size resid ) const
{
	using namespace pack::rotamer_set;
	using namespace trie;
	using namespace hbtrie;
	using EnergiesCacheableDataType::HBOND_TRIE_COLLECTION;

	RotamerSetFactory rsf;
	RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( resid ) );
	one_rotamer_set->set_resid( resid ); // change the interface so that the rotset initializes its seqpos at creation
	one_rotamer_set->add_rotamer( pose.residue( resid ) );
	HBondRotamerTrieOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );

	// grab non-const & of the cached tries and replace resid's trie with a new one.
	TrieCollection & trie_collection( static_cast< TrieCollection & > (pose.energies().data().get( HBOND_TRIE_COLLECTION )));
	trie_collection.trie( resid, one_rotamer_trie );
}



///
void
HBondEnergy::setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	pose.update_residue_neighbors();
	HBondSetOP hbond_set( new hbonds::HBondSet( pose.total_residue() ) );
	hbond_set->exclude_DNA_DNA   ( exclude_DNA_DNA_ );
	hbond_set->use_hb_env_dep    ( use_hb_env_dep_ );
	hbond_set->use_hb_env_dep_DNA( use_hb_env_dep_DNA_ );
	hbond_set->smooth_hb_env_dep ( smooth_hb_env_dep_ );
	hbond_set->setup_for_residue_pair_energies( pose );

	/// During minimization, keep the set of bb/bb hbonds "fixed" by using the old boolean values.
	if ( pose.energies().use_nblist() && pose.energies().data().has( HBOND_SET ) ) {
		HBondSet const & existing_set = static_cast< HBondSet const & > (pose.energies().data().get( HBOND_SET ));
		hbond_set->copy_bb_donor_acceptor_arrays( existing_set );
	}
	pose.energies().data().set( HBOND_SET, hbond_set );
}

///
void
HBondEnergy::setup_for_derivatives( pose::Pose & pose, ScoreFunction const & ) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	pose.update_residue_neighbors();
	HBondSetOP hbond_set( new hbonds::HBondSet( pose.total_residue() ) );
	hbond_set->exclude_DNA_DNA   ( exclude_DNA_DNA_ );
	hbond_set->use_hb_env_dep    ( use_hb_env_dep_ );
	hbond_set->use_hb_env_dep_DNA( use_hb_env_dep_DNA_ );
	hbond_set->smooth_hb_env_dep ( smooth_hb_env_dep_ );
	//hbonds::fill_hbond_set( pose, true /*calc derivs*/, *hbond_set );
	//hbond_set->resize_bb_donor_acceptor_arrays( pose.total_residue() );
	hbond_set->setup_for_residue_pair_energies( pose, true /*calc derivs?*/, false /*bb only?*/ );
	if ( pose.energies().use_nblist() && pose.energies().data().has( HBOND_SET ) ) {
		HBondSet const & existing_set = static_cast< HBondSet const & > (pose.energies().data().get( HBOND_SET ));
		hbond_set->copy_bb_donor_acceptor_arrays( existing_set );
	}
	pose.energies().data().set( HBOND_SET, hbond_set );
}



/////////////////////////////////////////////////////////////////////////////
// scoring
/////////////////////////////////////////////////////////////////////////////

/// note that this only evaluates sc-sc and sc-bb energies
void
HBondEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	if ( rsd1.seqpos() == rsd2.seqpos() ) return;
	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
			( pose.energies().data().get( HBOND_SET )));

	// this only works because we have already called
	// hbond_set->setup_for_residue_pair_energies( pose )
	Real scE( 0.0 ), bb_scE( 0.0 );
	if (!decompose_bb_hb_into_pair_energies_) {
		hbond_set.get_residue_residue_energy( rsd1, rsd2, scE, bb_scE );
	} else {
		Real sr_bbE( 0.0 ), lr_bbE( 0.0 );
		hbond_set.get_residue_residue_energy( rsd1, rsd2, sr_bbE, lr_bbE, scE, bb_scE );

		// store the energies
		emap[ hbond_sr_bb ] += sr_bbE;
		emap[ hbond_lr_bb ] += lr_bbE;
	}
	emap[ hbond_bb_sc ] += bb_scE;
	emap[ hbond_sc ] += scE;

	if ( std::abs( int( rsd1.seqpos() ) - int( rsd2.seqpos() ) ) <= hbonds::GENERIC_SHORTRANGE_SEQ_CUTOFF ) {
		emap[ hbond_sr_bb_sc ] += bb_scE;
	} else {
		emap[ hbond_lr_bb_sc ] += bb_scE;
	}

}


void
HBondEnergy::backbone_backbone_energy(
	conformation::Residue const &,// rsd1,
	conformation::Residue const &,// rsd2,
	pose::Pose const &,// pose,
	ScoreFunction const &,// sfxn,
	TwoBodyEnergyMap &// emap
) const
{
/*
	if ( rsd1.seqpos() == rsd2.seqpos() ) return;
	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
			( pose.energies().data().get( HBOND_SET )));

	// this only works because we have already called
	// hbond_set->setup_for_residue_pair_energies( pose )
	Real scE( 0.0 ), bb_scE( 0.0 );
	hbond_set.get_residue_residue_energy( rsd1, rsd2, scE, bb1_sc2E, bb2_sc1E );
*/
}


void
HBondEnergy::backbone_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const &,// sfxn,
	TwoBodyEnergyMap & emap
) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	if ( rsd1.seqpos() == rsd2.seqpos() ) return;
	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
			( pose.energies().data().get( HBOND_SET )));

	// this only works because we have already called
	// hbond_set->setup_for_residue_pair_energies( pose )
	Real scE( 0.0 ), bb1_sc2E( 0.0 ), bb2_sc1E( 0.0 );
	hbond_set.get_residue_residue_energy( rsd1, rsd2, scE, bb1_sc2E, bb2_sc1E );

	emap[ hbond_bb_sc ] += bb1_sc2E;

	if ( std::abs( int( rsd1.seqpos() ) - int( rsd2.seqpos() ) ) <= hbonds::GENERIC_SHORTRANGE_SEQ_CUTOFF ) {
		emap[ hbond_sr_bb_sc ] += bb1_sc2E;
	} else {
		emap[ hbond_lr_bb_sc ] += bb1_sc2E;
	}

}

void
HBondEnergy::sidechain_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const &,// sfxn,
	TwoBodyEnergyMap & emap
) const
{
	emap[ hbond_sc ] += get_sc_sc_hbond_energy(
		rsd1, rsd2,
		pose.energies().tenA_neighbor_graph().get_node( rsd1.seqpos() )->num_neighbors_counting_self_static(),
		pose.energies().tenA_neighbor_graph().get_node( rsd2.seqpos() )->num_neighbors_counting_self_static(),
		use_hb_env_dep_, use_hb_env_dep_DNA_, smooth_hb_env_dep_ );
}


void
HBondEnergy::evaluate_rotamer_pair_energies(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & ,
	EnergyMap const & weights,
	ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
) const
{
	assert( set1.resid() != set2.resid() );

	if ( exclude_DNA_DNA_ && pose.residue( set1.resid() ).is_DNA() && pose.residue( set2.resid() ).is_DNA() ) return;

	using namespace methods;
	using namespace hbtrie;
	using namespace trie;
	using EnergiesCacheableDataType::HBOND_SET;

	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table1( energy_table );
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table2( energy_table );

	temp_table1 = 0; temp_table2 = 0;

	// save weight information so that its available during tvt execution
	// and also the neighbor counts for the two residues.
	weights_ = weights;
	res1_ = set1.resid();
	res2_ = set2.resid();

	if ( true ) { // super_hacky
		hbonds::HBondSet const & hbond_set
			( static_cast< hbonds::HBondSet const & >
				( pose.energies().data().get( HBOND_SET )));
		res1_nb_ = hbond_set.nbrs( set1.resid() );
		res2_nb_ = hbond_set.nbrs( set2.resid() );
	} else {
		res1_nb_ = pose.energies().tenA_neighbor_graph().get_node( set1.resid() )->num_neighbors_counting_self();
		res2_nb_ = pose.energies().tenA_neighbor_graph().get_node( set2.resid() )->num_neighbors_counting_self();
	}
	HBondRotamerTrieCOP trie1 = set1.get_trie( hbond_method );
	HBondRotamerTrieCOP trie2 = set2.get_trie( hbond_method );

	//prepare_for_residue_pair( set1.resid(), set2.resid(), pose );
	//assert( rep_scoretype() == fa_rep || rep_scoretype() == coarse_fa_rep );

	// figure out which trie countPairFunction needs to be used for this set
	TrieCountPairBaseOP cp = new HBCountPairFunction;

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_trie( *trie2, *cp, *this, temp_table1, temp_table2 );

	/// add in the energies calculated by the tvt alg.
	energy_table += temp_table1;
	//std::cout << "FINISHED evaluate_rotamer_pair_energies" << std::endl;

	/*
	// debug
	using namespace pack;

	ObjexxFCL::FArray2D< PackerEnergy > temp_table3( energy_table );
	temp_table3 = 0;
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set1.num_rotamers(); ii <= ii_end; ++ii ) {
		for ( Size jj = 1, jj_end = set2.num_rotamers(); jj <= jj_end; ++jj ) {
			emap.zero();
			residue_pair_energy( *set1.rotamer( ii ), *set2.rotamer( jj ), pose, sfxn, emap );
			temp_table3( jj, ii ) += weights.dot( emap );
			if ( std::abs( temp_table1( jj, ii ) - temp_table3( jj, ii )) > 0.001 ) {
				std::cout << "Residues " << set1.resid() << " & " << set2.resid() << " rotamers: " << ii << " & " << jj;
				std::cout << " tvt/reg discrepancy: tvt= " <<  temp_table1( jj, ii ) << " reg= " << temp_table3( jj, ii );
				std::cout << " delta: " << temp_table1( jj, ii ) - temp_table3( jj, ii ) << std::endl;
			}
		}
	}
	std::cout << "Finished RPE calcs for residues " << set1.resid() << " & " << set2.resid() << std::endl;
	*/
}



//@brief overrides default rotamer/background energy calculation and uses
// the trie-vs-trie algorithm instead
void
HBondEnergy::evaluate_rotamer_background_energies(
	pack::rotamer_set::RotamerSet const & set,
	conformation::Residue const & residue,
	pose::Pose const & pose,
	ScoreFunction const & ,
	EnergyMap const & weights,
	utility::vector1< pack::PackerEnergy > & energy_vector
) const
{
	using namespace methods;
	using namespace hbtrie;
	using namespace trie;
	using EnergiesCacheableDataType::HBOND_SET;
	using EnergiesCacheableDataType::HBOND_TRIE_COLLECTION;

	if ( exclude_DNA_DNA_ && residue.is_DNA() && pose.residue( set.resid() ).is_DNA() ) return;

	// allocate space for the trie-vs-trie algorithm
	utility::vector1< pack::PackerEnergy > temp_vector1( set.num_rotamers(), 0.0 );
	utility::vector1< pack::PackerEnergy > temp_vector2( set.num_rotamers(), 0.0 );

	// save weight information so that its available during tvt execution
	weights_ = weights;
	res1_ = set.resid();
	res2_ = residue.seqpos();
	if ( true ) { // super_hacky
		hbonds::HBondSet const & hbond_set
			( static_cast< hbonds::HBondSet const & >
				( pose.energies().data().get( HBOND_SET )));
		res1_nb_ = hbond_set.nbrs( set.resid() );
		res2_nb_ = hbond_set.nbrs( residue.seqpos() );
	} else {
		res1_nb_ = pose.energies().tenA_neighbor_graph().get_node( set.resid() )->num_neighbors_counting_self();
		res2_nb_ = pose.energies().tenA_neighbor_graph().get_node( residue.seqpos() )->num_neighbors_counting_self();
	}

	HBondRotamerTrieCOP trie1 = set.get_trie( hbond_method );
	HBondRotamerTrieCOP trie2 = ( static_cast< TrieCollection const & >
																( pose.energies().data().get( HBOND_TRIE_COLLECTION )) ).trie( residue.seqpos() );

	TrieCountPairBaseOP cp = new HBCountPairFunction;

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge (and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_path( *trie2, *cp, *this, temp_vector1, temp_vector2 );

	/// add in the energies calculated by the tvt alg.
	for ( Size ii = 1; ii <= set.num_rotamers(); ++ii ) {
		energy_vector[ ii ] += temp_vector1[ ii ];
	}
	//std::cout << "FINISHED evaluate_rotamer_background_energies" << std::endl;

	/*
	//debug
	utility::vector1< Energy > temp_vector3( energy_vector.size(), 0.0f );
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set.num_rotamers(); ii <= ii_end; ++ii ) {
		emap.zero();
		residue_pair_energy( *set.rotamer( ii ), residue, pose, sfxn, emap );
		temp_vector3[ ii ] += weights.dot( emap );
		if ( std::abs( temp_vector1[ ii ] - temp_vector3[ ii ]) > 0.001 ) {
			std::cout << "Residues " << set.resid() << " & " << residue.seqpos() << " rotamers: " << ii << " & bg";
			std::cout << " tvt/reg discrepancy: tvt= " <<  temp_vector1[ ii ] << " reg= " << temp_vector3[ ii ];
			std::cout << " delta: " << temp_vector1[ ii ] - temp_vector3[ ii ] << std::endl;
		}
	}
	std::cout << "Finished Rotamer BG calcs for residues " << set.resid() << " & " << residue.seqpos() << std::endl;
	*/

}


///
void
HBondEnergy::finalize_total_energy(
	pose::Pose & pose,
	ScoreFunction const &,
	EnergyMap & totals
) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	if (decompose_bb_hb_into_pair_energies_) return;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
			( pose.energies().data().get( HBOND_SET )));

	Real lr_bbE( 0.0 ), sr_bbE( 0.0 ), bb_scE( 0.0 ), scE( 0.0 );
	//Real lr_bb_scE( 0.0 ), sr_bb_scE( 0.0 );

	get_hbond_energies( hbond_set, sr_bbE, lr_bbE, bb_scE, scE );

	// the current logic is that we fill the hbond set with backbone
	// hbonds only at the beginning of scoring. this is done to setup
	// the bb-bb hbond exclusion logic. so the hbondset should only
	// include bb-bb hbonds.
	// but see get_hb_don_chem_type in hbonds_geom.cc -- that only
	// classifies protein backbone donors as backbone, and the energy
	// accumulation by type is influenced by that via HBeval_lookup
	//
	// the important thing is that there's no double counting, which
	// is I think true since both fill_hbond_set and get_rsd-rsd-energy
	// use atom_is_backbone to check...
	//assert( std::abs( bb_scE ) < 1e-3 && std::abs( scE ) < 1e-3 );
	totals[ hbond_sr_bb ] += sr_bbE;
	totals[ hbond_lr_bb ] += lr_bbE;
}

/// f1 and f2 are zeroed
void
HBondEnergy::eval_atom_derivative(
	id::AtomID const & atom_id,
	pose::Pose const & pose,
	kinematics::DomainMap const &,
	ScoreFunction const &,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	using EnergiesCacheableDataType::HBOND_SET;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
		( pose.energies().data().get( HBOND_SET ) ) );
	Vector f1,f2;
	hbonds::get_atom_hbond_derivative( atom_id, hbond_set, weights, f1, f2 );
	F1 += f1;
	F2 += f2;
}

///@brief HACK!  MAX_R defines the maximum donorH to acceptor distance.
// The atomic_interaction_cutoff method is meant to return the maximum distance
// between two *heavy atoms* for them to have a zero interaction energy.
// I am currently assuming a 1.35 A maximum distance between a hydrogen and the
// heavy atom it is bound to, stealing this number from the CYS.params file since
// the HG in CYS is much further from it's SG than aliphatic hydrogens are from their carbons.
// This is a bad idea.  Someone come up with a way to fix this!
//
// At 4.35 A interaction cutoff, the hbond energy function is incredibly short ranged!
Distance
HBondEnergy::atomic_interaction_cutoff() const
{
	return MAX_R + 1.35; // MAGIC NUMBER
}

/// @brief the atomic interaction cutoff and the hydrogen interaction cutoff are the same.
Real
HBondEnergy::hydrogen_interaction_cutoff2() const
{
	return (MAX_R + 1.35) * ( MAX_R + 1.35 );
}


///@brief HBondEnergy is context sensitive
void
HBondEnergy::indicate_required_context_graphs(
	utility::vector1< bool > & context_graphs_required
) const
{
	context_graphs_required[ ten_A_neighbor_graph ] = true;
}

bool
HBondEnergy::defines_intrares_energy( EnergyMap const & /*weights*/ ) const
{
	return false;
}

void
HBondEnergy::eval_intrares_energy(
	conformation::Residue const & ,
	pose::Pose const & ,
	ScoreFunction const & ,
	EnergyMap &
) const {}


hbtrie::HBondRotamerTrieOP
HBondEnergy::create_rotamer_trie(
	pack::rotamer_set::RotamerSet const & rotset,
	pose::Pose const & pose
) const
{
	using namespace trie;
	using namespace hbtrie;
	using EnergiesCacheableDataType::HBOND_SET;

	hbonds::HBondSet const & hbond_set
		( static_cast< hbonds::HBondSet const & >
		( pose.energies().data().get( HBOND_SET ) ) );

	Size const resid( rotset.resid() );

	utility::vector1< RotamerDescriptor< HBAtom, HBCPData > > rotamer_descriptors( rotset.num_rotamers() );

	for ( Size ii = 1; ii <= rotset.num_rotamers(); ++ii ) {
		conformation::ResidueCOP ii_rotamer( rotset.rotamer( ii ) );

		//count the number of acceptors and donors
		//Size nhbonders = ii_rotamer->type().Hpos_polar().size() + ii_rotamer->type().accpt_pos().size();

		Size n_to_add(0);
		utility::vector1< int > add_to_trie( ii_rotamer->natoms(), 0 );
		for ( Size jj = 1; jj <= ii_rotamer->natoms(); ++jj ) {
			if ( ii_rotamer->atom_type_set()[ ii_rotamer->atom( jj ).type() ].is_acceptor() ) {
				//std::cout << "acc=" << jj << " ";
				add_to_trie[ jj ] = 1;
			} //else if ( ii_rotamer->atom_type_set()[ ii_rotamer->atom( jj ).type() ].is_hydrogen() &&
				else if ( ii_rotamer->atom_is_hydrogen( jj ) &&
					ii_rotamer->atom_type_set()[ ii_rotamer->atom( ii_rotamer->type().atom_base( jj ) ).type() ].is_donor()) {
				add_to_trie[ jj ] = 1;
				add_to_trie[ ii_rotamer->type().atom_base( jj ) ] = 1;
				//std::cout << "don=" << jj << " donb= " << ii_rotamer->type().atom_base( jj ) << " ";
			}
		}
		//std::cout << "rotamer trie for residue: " << ii_rotamer->type().name() << std::endl;
		for ( Size jj = 1; jj <= ii_rotamer->natoms(); ++jj ) {
			if ( add_to_trie[ jj ] == 1 ){
				++n_to_add;
				//std::cout << jj << " ";
			}
		}
		//std::cout << std::endl;

		if ( n_to_add == 0 ) {
			// What happens if there are NO hydrogen bonding atoms?  It would be nice if we didn't have to create
			// a trie at all, but I'm pretty sure the indexing logic requires that we have a place-holder rotamer.
			add_to_trie[ 1 ] = 1; ++n_to_add;
		}
		rotamer_descriptors[ ii ].natoms( n_to_add );

		Size count_added_atoms = 0;
		for ( Size jj = 1; jj <= ii_rotamer->nheavyatoms(); ++jj ) {
			if ( add_to_trie[ jj ] == 0 ) continue;

			HBAtom newatom;
			HBCPData cpdata;

			newatom.xyz( ii_rotamer->atom(jj).xyz() );
			newatom.is_hydrogen( false );
			newatom.is_backbone( ii_rotamer->atom_is_backbone( jj ) ) ;

			//Following preserves hbond_sc, hbond_bb_sc, as preferred by other developers for protein/DNA.
			newatom.is_protein( ii_rotamer->is_protein() );
			newatom.is_dna( ii_rotamer->is_DNA() );

			if ( ii_rotamer->atom_type_set()[ ii_rotamer->atom( jj ).type() ].is_acceptor() ) {
				newatom.hb_chem_type( get_hb_acc_chem_type( jj, *ii_rotamer ));
				newatom.orientation_vector( create_acc_orientation_vector( *ii_rotamer, jj ));

				cpdata.is_sc( ! ii_rotamer->type().atom_is_backbone( jj ) );

				cpdata.avoid_sc_hbonds( ! cpdata.is_sc() &&
					ii_rotamer->type().is_protein() &&
					hbond_set.acc_bbg_in_bb_bb_hbond( resid ) );
			}


			RotamerDescriptorAtom< HBAtom, HBCPData > rdatom( newatom, cpdata );
			rotamer_descriptors[ ii ].atom( ++count_added_atoms, rdatom );

			for ( Size kk = ii_rotamer->attached_H_begin( jj ),
					kk_end = ii_rotamer->attached_H_end( jj );
					kk <= kk_end; ++kk ) {
				if ( add_to_trie[ kk ] == 0 ) continue;

				HBAtom newhatom;
				newhatom.xyz( ii_rotamer->atom(kk).xyz() );
				newhatom.orientation_vector( create_don_orientation_vector( *ii_rotamer, kk ));

				newhatom.hb_chem_type( get_hb_don_chem_type( kk, *ii_rotamer ));
				newhatom.is_hydrogen( true );
				newhatom.is_backbone( ii_rotamer->atom_is_backbone( kk ) ) ;

				//Following preserves hbond_sc, hbond_bb_sc, as preferred by other developers for protein/DNA.
				newhatom.is_protein( ii_rotamer->is_protein() );
				newhatom.is_dna( ii_rotamer->is_DNA() );

				HBCPData hcpdata;
				hcpdata.is_sc( ! ii_rotamer->type().atom_is_backbone( kk ) );

				hcpdata.avoid_sc_hbonds( ! hcpdata.is_sc() &&
					ii_rotamer->type().is_protein() &&
					hbond_set.don_bbg_in_bb_bb_hbond( resid ) );

				RotamerDescriptorAtom< HBAtom, HBCPData > hrdatom( newhatom, hcpdata );
				rotamer_descriptors[ ii ].atom( ++count_added_atoms, hrdatom );

			}
			rotamer_descriptors[ ii ].rotamer_id( ii );
		}
	}

	// Debug.  An inconsistent < operator can cause rare segfaults by making
	// the insertion-sort step in std::sort() run off the end of the array/vector.
	// (E.g. compare floats == within epsilon, and a == b, b == c, but a < c)
	// It seems to happen more often when re-sorting an already sorted list
	// (as happens when the RotamerTrie is constructed, below).

	//if( resid == 273 ) {
	//	std::cout << "First sort\n";
	//	sort( rotamer_descriptors.begin(), rotamer_descriptors.end() );
	//	for( Size i = 1; i <= rotamer_descriptors.size(); ++i) {
	//		for( Size j = i+1; j <= rotamer_descriptors.size(); ++j) {
	//			if( (rotamer_descriptors[i] < rotamer_descriptors[j]) && (rotamer_descriptors[j] < rotamer_descriptors[i]) ) {
	//				std::cout << "Both items are less; i = " << i << ", j = " << j << "\n";
	//			}
	//			if(rotamer_descriptors[j] < rotamer_descriptors[i]) {
	//				std::cout << "Not strictly sorted; i = " << i << ", j = " << j << "\n";
	//			}
	//		}
	//	}
	//	std::cout << "Second sort\n";
	//	sort( rotamer_descriptors.begin(), rotamer_descriptors.end() );
	//	std::cout << "Survived!\n";
	//}
	//else {
		sort( rotamer_descriptors.begin(), rotamer_descriptors.end() );
	//}

	return new RotamerTrie< HBAtom, HBCPData >( rotamer_descriptors, atomic_interaction_cutoff());

}

///@brief code to evaluate a hydrogen bond energy for the trie that
/// didn't belong in the header itself -- it certainly does enough work
/// such that inlining it would not likely produce a speedup.
Energy
HBondEnergy::drawn_out_heavyatom_hydrogenatom_energy(
	hbtrie::HBAtom const & at1, // atom 1 is the heavy atom, the acceptor
	hbtrie::HBAtom const & at2, // atom 2 is the hydrogen atom, the donor
	bool flipped // is at1 from residue 1?
) const
{
	HBEvalType hbe_type = HBeval_lookup( at1.hb_acc_chem_type(), at2.hb_don_chem_type() );
	if (hbe_type == hbe_BB) { // refine backbone/backbone bond types by separation
		Size acc_res = flipped ? res2_ : res1_;
		Size don_res = flipped ? res1_ : res2_;
		hbe_type = hbe_classify_BB_by_separation( acc_res, don_res );
	}

	Energy hbenergy;
	hb_energy_deriv( hbe_type, at2.xyz(), at2.orientation_vector(),
		at1.xyz(), at1.orientation_vector(), hbenergy,
		false /*evaluate_derivative*/, DUMMY_DERIV2D );

	if ( hbenergy >= MAX_HB_ENERGY ) return 0.0; // no hbond

	Real envweight( 1.0 );
	if ( use_hb_env_dep_ ) envweight = ( flipped ? get_environment_dependent_weight( hbe_type, res2_nb_, res1_nb_ ) :
																	           get_environment_dependent_weight( hbe_type, res1_nb_, res2_nb_ ));
	if ( ! use_hb_env_dep_DNA_ ) {
		// avoid neighbor-based attenuation for hbonds involving DNA residues
		if ( at1.is_dna() || at2.is_dna() ) envweight = 1.0;
	}
	Energy weighted_energy( 0.0 );

	if ( (!at1.is_protein() && !at1.is_dna() ) ||
			 (!at2.is_protein() && !at2.is_dna() ) ) {
		//use a generic rule to assign hbonds as "backbone", etc.
		bool const & atm1_is_bb(  at1.is_backbone() );
		bool const & atm2_is_bb(  at2.is_backbone() );
		if ( atm1_is_bb && atm2_is_bb ){
			if ( std::abs( int(res1_) - int(res2_) ) <= hbonds::GENERIC_SHORTRANGE_SEQ_CUTOFF) {
				weighted_energy = weights_[ hbond_sr_bb ] * hbenergy * envweight;
			} else {
				weighted_energy = weights_[ hbond_lr_bb ] * hbenergy * envweight;
			}
		} else if (atm1_is_bb || atm2_is_bb ) {
			weighted_energy = weights_[ hbond_sc ] * hbenergy * envweight;
			if ( std::abs( int(res1_) - int(res2_) ) <= hbonds::GENERIC_SHORTRANGE_SEQ_CUTOFF) {
				weighted_energy = weights_[ hbond_sr_bb_sc ] * hbenergy * envweight;
			} else {
				weighted_energy = weights_[ hbond_lr_bb_sc ] * hbenergy * envweight;
			}
		} else {
			weighted_energy = weights_[ hbond_sc ] * hbenergy * envweight;
		}
		return weighted_energy;
	}

	// Original, rosetta++ style assignments, mainly protein-centric.
	switch ( hbe_type ) {
		case hbe_BBTURN:
		case hbe_BBHELIX:
			weighted_energy = weights_[ hbond_sr_bb ] * hbenergy * envweight;
			break;
		case hbe_BBOTHER:
			weighted_energy = weights_[ hbond_lr_bb ] * hbenergy * envweight;
			break;
		case hbe_SP2B:
		case hbe_SP3B:
		case hbe_RINGB:
			weighted_energy = weights_[ hbond_bb_sc ] * hbenergy * envweight;
			break;
		case hbe_BSC:
			weighted_energy = weights_[ hbond_bb_sc ] * hbenergy * envweight;
			break;
		case hbe_SP2SC:
		case hbe_SP3SC:
		case hbe_RINGSC:
			weighted_energy = weights_[ hbond_sc ] * hbenergy * envweight;
			break;
		default:
			std::cout << "Warning: energy from unexpected HB type ignored " <<
				hbe_type << std::endl;
			utility_exit();
	} // switch ( hbe_type )
	//std::cout << "TVT Found an hbond: hbenergy: " << hbenergy << " envweight: " << envweight << " total: " << weighted_energy << std::endl;
	//std::cout << at1 << " and " << at2 << std::endl;
	return weighted_energy;
}

} // hbonds
} // scoring
} // core

