// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/HBondEnergy.hh
/// @brief  Hydrogen bond energy method class declaration
/// @author Phil Bradley
/// @author Andrew Leaver-Fay


#ifndef INCLUDED_core_scoring_hbonds_HBondEnergy_HH
#define INCLUDED_core_scoring_hbonds_HBondEnergy_HH

// Unit Headers
#include <core/scoring/hbonds/HBondEnergy.fwd.hh>

// Package headers
#include <core/scoring/hbonds/hbtrie/HBAtom.hh>
#include <core/scoring/hbonds/hbtrie/HBondTrie.fwd.hh>
#include <core/scoring/hbonds/constants.hh>

#include <core/scoring/methods/ContextDependentTwoBodyEnergy.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>

#include <core/scoring/TenANeighborGraph.hh>

// Project headers
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/EnergyMap.hh>

namespace core {
namespace scoring {
namespace hbonds {

///
class HBondEnergy : public methods::ContextDependentTwoBodyEnergy  {
public:
	typedef methods::ContextDependentTwoBodyEnergy  parent;
public:

	///
	HBondEnergy( methods::EnergyMethodOptions const & options );

	///
	HBondEnergy( HBondEnergy const & src );

	/// clone
	virtual
	methods::EnergyMethodOP
	clone() const;

	///
	virtual
	void
	setup_for_packing( pose::Pose & pose, pack::task::PackerTask const & ) const;

	// Creates a rotamer trie for the input set of rotamers and stores the trie
	// in the rotamer set.
	virtual
	void
	prepare_rotamers_for_packing(
		pose::Pose const & pose,
		pack::rotamer_set::RotamerSet & set ) const;

	// Updates the cached rotamer trie for a residue if it has changed during the course of
	// a repacking
	virtual
	void
	update_residue_for_packing( pose::Pose & pose, Size resid ) const;

	///
	virtual
	void
	setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const;

	///
	virtual
	void
	setup_for_derivatives( pose::Pose & pose, ScoreFunction const & ) const;


	/////////////////////////////////////////////////////////////////////////////
	// scoring
	/////////////////////////////////////////////////////////////////////////////

	/// note that this only evaluates sc-sc and sc-bb energies
	virtual
	void
	residue_pair_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const &,
		TwoBodyEnergyMap & emap
	) const;

	///@brief Evaluates the interaction between the backbone of rsd1 and the
	/// backbone of rsd2 and accumulates the unweighted energy.
	virtual
	void
	backbone_backbone_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;


	///@brief Evaluates the interaction between the backbone of rsd1 and the
	/// sidechain of rsd2 and accumulates the unweighted energy.
	virtual
	void
	backbone_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;

	///@brief Evaluates the interaction between the sidechain of rsd1 and the
	/// sidechain of rsd2 and accumulates the unweighted energy.
	virtual
	void
	sidechain_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;



	virtual
	void
	evaluate_rotamer_pair_energies(
		pack::rotamer_set::RotamerSet const & set1,
		pack::rotamer_set::RotamerSet const & set2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
	) const;


	//@brief overrides default rotamer/background energy calculation and uses
	// the trie-vs-trie algorithm instead
	virtual
	void
	evaluate_rotamer_background_energies(
		pack::rotamer_set::RotamerSet const & set,
		conformation::Residue const & residue,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		utility::vector1< pack::PackerEnergy > & energy_vector
	) const;


	///
	virtual
	void
	finalize_total_energy(
		pose::Pose & pose,
		ScoreFunction const &,
		EnergyMap & totals
	) const;

	/// f1 and f2 are zeroed
	virtual
	void
	eval_atom_derivative(
		id::AtomID const & atom_id,
		pose::Pose const & pose,
		kinematics::DomainMap const &,
		ScoreFunction const &,
		EnergyMap const & weights,
		Vector & F1,
		Vector & F2
	) const;


	virtual
	Distance
	atomic_interaction_cutoff() const;

	Real
	hydrogen_interaction_cutoff2() const;

	///@brief HBondEnergy is context sensitive
	virtual
	void indicate_required_context_graphs(
		utility::vector1< bool > & context_graphs_required ) const;

	virtual
	bool
	defines_intrares_energy( EnergyMap const & /*weights*/ ) const;

	virtual
	void
	eval_intrares_energy(
		conformation::Residue const & rsd,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;

	inline
	Energy heavyatom_heavyatom_energy(
		hbtrie::HBAtom const & at1,
		hbtrie::HBAtom const & at2,
		DistanceSquared & d2
	) const
	{
		d2 = at1.xyz().distance_squared( at2.xyz() );
		return 0.0;
	}

	inline
	Energy heavyatom_hydrogenatom_energy(
		hbtrie::HBAtom const & at1, // atom 1 is the heavy atom, the acceptor unless it's a placeholder atom
		hbtrie::HBAtom const & at2, // atom 2 is the hydrogen atom, the donor
		bool flipped = false
	) const
	{
		DistanceSquared d2 = at1.xyz().distance_squared( at2.xyz() );
		if ( d2 > MAX_R2 || d2 < MIN_R2 || at1.non_hbonding_atom() ) return 0.0;

		return drawn_out_heavyatom_hydrogenatom_energy( at1, at2, flipped );
	}

	Energy
	drawn_out_heavyatom_hydrogenatom_energy(
		hbtrie::HBAtom const & at1, // atom 1 is the heavy atom, the acceptor
		hbtrie::HBAtom const & at2, // atom 2 is the hydrogen atom, the donor
		bool flipped
	) const;

	inline
	Energy hydrogenatom_heavyatom_energy(
		hbtrie::HBAtom const & at1,
		hbtrie::HBAtom const & at2
	) const
	{
		return heavyatom_hydrogenatom_energy( at2, at1, true );
	}

	inline
	Energy hydrogenatom_hydrogenatom_energy(
		hbtrie::HBAtom const &,
		hbtrie::HBAtom const &
	) const
	{
		return 0.0;
	}

private:

	hbtrie::HBondRotamerTrieOP
	create_rotamer_trie(
		pack::rotamer_set::RotamerSet const & rotset,
		pose::Pose const & pose
	) const;

	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////

private:

	bool exclude_DNA_DNA_;
	bool use_hb_env_dep_;
	bool use_hb_env_dep_DNA_;
	bool smooth_hb_env_dep_;
	bool decompose_bb_hb_into_pair_energies_;

	// Used in the "const" evaluate_rotamer_pair_energies and evaluate_rotamer_background_energies
	// methods to keep track of the sfxn weights and the neighbor counts for the two input residues
	// so that the data may be retrieved from within the trie-vs-trie and trie-vs-path calls.
	mutable EnergyMap weights_;
	mutable int res1_;
	mutable int res2_;
	mutable int res1_nb_;
	mutable int res2_nb_;
};

} // hbonds
} // scoring
} // core

#endif


