// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is made available under the Rosetta Commons license.
// See http://www.rosettacommons.org/license
// (C) 199x-2007 University of Washington
// (C) 199x-2007 University of California Santa Cruz
// (C) 199x-2007 University of California San Francisco
// (C) 199x-2007 Johns Hopkins University
// (C) 199x-2007 University of North Carolina, Chapel Hill
// (C) 199x-2007 Vanderbilt University

/// @file   core/scoring/hbonds/EnergyMethodOptions.cc
/// @brief  HBondOptions class, holds the options for the hbond energy function
/// @author Matthew O'Meara

/// @detail
/// To add an additional option for hydrogen bonds do the following:
///
/// In HBondOptions.hh:
/// 1) add it to the default constructor
/// 2) add it to the copy constructor
/// 3) add a getter and a setter
/// 4) add it to operator==
/// 5) add it to the private data
/// 6) add it to HBondOptions::show

#include <core/scoring/hbonds/HBondOptions.hh>
#include <string>
#include <iostream>

namespace core {
namespace scoring {
namespace hbonds {


  HBondOptions::HBondOptions():
    exclude_DNA_DNA_( true ),
    use_hb_env_dep_ ( true ),
		use_hb_env_dep_DNA_( true ),
    smooth_hb_env_dep_( true ),
    decompose_bb_hb_into_pair_energies_( false )
  {}

  /// copy constructor
  HBondOptions::HBondOptions( HBondOptions const & src ):
    ReferenceCount( src )
  {
    *this = src;
  }

  HBondOptions::~HBondOptions(){}

  /// copy operator
  HBondOptions const &
  HBondOptions::operator=( HBondOptions const & src )
  {
    exclude_DNA_DNA_ = src.exclude_DNA_DNA_;
    use_hb_env_dep_ = src.use_hb_env_dep_;
		use_hb_env_dep_DNA_ = src.use_hb_env_dep_DNA_;
    smooth_hb_env_dep_ = src.smooth_hb_env_dep_;
    decompose_bb_hb_into_pair_energies_ = src.decompose_bb_hb_into_pair_energies_;
    return *this;
  }

  ///
  bool const &
  HBondOptions::exclude_DNA_DNA() const
  {
    return exclude_DNA_DNA_;
  }

  ///
  void
  HBondOptions::exclude_DNA_DNA( bool const & setting )
  {
    exclude_DNA_DNA_ = setting;
  }

  ///
  bool const &
  HBondOptions::use_hb_env_dep_DNA() const
  {
    return use_hb_env_dep_DNA_;
  }

  ///
  void
  HBondOptions::use_hb_env_dep_DNA( bool const & setting )
  {
    use_hb_env_dep_DNA_ = setting;
  }

  ///
  bool const &
  HBondOptions::use_hb_env_dep() const
  {
    return use_hb_env_dep_;
  }

  ///
  void
  HBondOptions::use_hb_env_dep( bool const & setting )
  {
    use_hb_env_dep_ = setting;
  }

  ///
  bool const &
  HBondOptions::smooth_hb_env_dep() const
  {
    return smooth_hb_env_dep_;
  }

  ///
  void
  HBondOptions::smooth_hb_env_dep( bool const & setting )
  {
    smooth_hb_env_dep_ = setting;
  }

  ///
  bool const &
  HBondOptions::decompose_bb_hb_into_pair_energies() const
  {
    return decompose_bb_hb_into_pair_energies_;
  }

  ///
  void
  HBondOptions::decompose_bb_hb_into_pair_energies( bool const & setting )
  {
    decompose_bb_hb_into_pair_energies_ = setting;
  }

  bool
  operator==( HBondOptions const & a, HBondOptions const & b )
  {
    return ( ( a.exclude_DNA_DNA_                    == b.exclude_DNA_DNA_ ) &&
	     ( a.use_hb_env_dep_     	             == b.use_hb_env_dep_ ) &&
	     ( a.use_hb_env_dep_DNA_                 == b.use_hb_env_dep_DNA_ ) &&
	     ( a.smooth_hb_env_dep_                  == b.smooth_hb_env_dep_ ) &&
	     ( a.decompose_bb_hb_into_pair_energies_ == b.decompose_bb_hb_into_pair_energies_ ) );
  }

  bool
  operator!=( HBondOptions const & a, HBondOptions const & b )
  {
    return !( a == b );
  }

  std::ostream &
  operator<< ( std::ostream & out,  const HBondOptions & options ){
    options.show( out );
    return out;
  }

  ///
  void
  HBondOptions::show( std::ostream & out ) const
  {
    out <<"HBondOptions::show: exclude_DNA_DNA: "
	<<( exclude_DNA_DNA_ ? "true" : "false" ) << std::endl;
    out <<"HBondOptions::show: use_hb_env_dep: "
	<<( use_hb_env_dep_ ? "true" : "false" ) << std::endl;
    out <<"HBondOptions::show: use_hb_env_dep_DNA: "
	<<( use_hb_env_dep_DNA_ ? "true" : "false" ) << std::endl;
    out <<"HBondOptions::show: smooth_hb_env_dep: "
	<<( smooth_hb_env_dep_ ? "true" : "false " ) << std::endl;
    out <<"HBondOptions::show: decompose_bb_hb_into_pair_energies: "
	<<( decompose_bb_hb_into_pair_energies_ ? "true" : "false" ) << std::endl;
  }

}
}
}
