// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/hbonds/HBondSet.hh
/// @brief  Hydrogen bond set class implementation
/// @author Andrew Leaver-Fay (aleaverfay@gmail.com)

// Unit Headers
#include <core/scoring/hbonds/HBondSet.hh>


// Package headers
#include <core/scoring/hbonds/types.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyGraph.hh>
#include <core/scoring/TenANeighborGraph.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <core/conformation/Residue.hh>
#include <core/id/AtomID.hh>
#include <core/pose/PDBInfo.hh>

//#include <core/options/util.hh> // HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK HACK


// Utility headers
//#include <utility/exit.hh>
// #include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <numeric/xyzVector.hh>

// ObjexxFCL headers
#include <ObjexxFCL/formatted.o.hh>

// C++
#include <map> // what is the right header for std::pair ?

// option key includes

#include <core/options/keys/dna.OptionKeys.gen.hh>


//#include <set>

namespace core {
namespace scoring {
namespace hbonds {


HBondSet::HBondSet():
	atom_map_init_( false ),
	exclude_DNA_DNA_( false ), // default
	exclude_bb_sc_( true ),
	use_hb_env_dep_( true ), // default
	use_hb_env_dep_DNA_( true ), // default
	smooth_hb_env_dep_( true ) // default
	// do not check this hack in!!!
	//exclude_bb_sc_( options::option[ options::OptionKeys::dna::specificity::exclude_bb_sc_hbonds ] )
{}

HBondSet::HBondSet( Size nres )
	: atom_map_init_( false ),
		exclude_DNA_DNA_( false ), // default
		exclude_bb_sc_( true ),
		use_hb_env_dep_( true ), // default
		use_hb_env_dep_DNA_( true ), // default
		smooth_hb_env_dep_( true ) // default
{
	resize_bb_donor_acceptor_arrays( nres );
}

/// copy ctor
HBondSet::HBondSet( HBondSet const & src ):
	util::datacache::CacheableData()
{
	for ( Size i=1; i<= src.hbonds_.size(); ++i ) {
		hbonds_.push_back( new HBond( *src.hbonds_[i] ) );
	}
	backbone_backbone_donor_ = src.backbone_backbone_donor_;
	backbone_backbone_acceptor_ = src.backbone_backbone_acceptor_;
	nbrs_ = src.nbrs_;
	// set this flag since we still have to setup the atommap
	atom_map_init_ = false;
	exclude_DNA_DNA_ = src.exclude_DNA_DNA_;
	exclude_bb_sc_ = src.exclude_bb_sc_;
	use_hb_env_dep_ = src.use_hb_env_dep_;
	use_hb_env_dep_DNA_ = src.use_hb_env_dep_DNA_;
	smooth_hb_env_dep_ = src.smooth_hb_env_dep_;
}




/// @brief clone this object
util::datacache::CacheableDataOP
HBondSet::clone() const
{
	return new HBondSet( *this );
}


///////////////////////////////////////////////////////////////////////////////

/// \brief  Add a new hbond to the list
/// updates the "hbchk" array as necessary
void
HBondSet::append_hbond(
	Size const dhatm,
	conformation::Residue const & don_rsd,
	Size const aatm,
	conformation::Residue const & acc_rsd,
	HBEvalType const & hbe_type,
	Real const energy,
	Real const weight,
	HBond::Deriv const & deriv
)
{
	atom_map_init_ = false;

	// note that the derivative may not have been evaluated
	bool const dhatm_is_protein_backbone
		( don_rsd.is_protein() && don_rsd.atom_is_backbone( dhatm ) );
	bool const  aatm_is_protein_backbone
		( acc_rsd.is_protein() && acc_rsd.atom_is_backbone(  aatm ) );

	bool const dhatm_is_backbone
		( don_rsd.atom_is_backbone( dhatm ) );
	bool const  aatm_is_backbone
		( acc_rsd.atom_is_backbone(  aatm ) );

	Size const don_pos( don_rsd.seqpos() );
	Size const acc_pos( acc_rsd.seqpos() );

	hbonds_.push_back( new HBond( dhatm, dhatm_is_protein_backbone, don_rsd.is_protein(), don_rsd.is_DNA(),
																dhatm_is_backbone, don_pos,
																aatm ,  aatm_is_protein_backbone, acc_rsd.is_protein(),
																acc_rsd.is_DNA(), aatm_is_backbone, acc_pos,
																hbe_type, energy, weight, deriv ) );

	// update hbcheck
	// note: these dimension checks could be removed completely & replaced by pose-aware dimensioning
	if ( dhatm_is_protein_backbone || aatm_is_protein_backbone ) {
		Size const max_pos( std::max( acc_pos, don_pos ) );
		if ( max_pos > backbone_backbone_donor_.size() ||
					max_pos > backbone_backbone_acceptor_.size() ) {
			// this could be more efficient if we require specifying nres ahead of time
			// Specify nres in setup_for_residue_pair_energy
			resize_bb_donor_acceptor_arrays( max_pos );
		}
		if ( dhatm_is_protein_backbone && aatm_is_protein_backbone ) {
			if ( exclude_bb_sc_ ) {
				backbone_backbone_donor_   [ don_pos ] = true;
				backbone_backbone_acceptor_[ acc_pos ] = true;
			}
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
bool
hbond_energy_comparer( HBondOP const & a, HBondOP const & b )
{
	return ( a->energy() * a->weight() < b->energy() * b->weight() );
}


///////////////////////////////////////////////////////////////////////////////
void
HBondSet::sort_by_weighted_energy()
{
	std::sort( hbonds_.begin(), hbonds_.end(), hbond_energy_comparer );
}



///////////////////////////////////////////////////////////////////////////////
/// \brief  Setup the mapping from atoms to hbonds
void
HBondSet::setup_atom_map() const // lazy updating
{
	if ( atom_map_init_ ) return;
	atom_map_init_ = true;
	for ( Size i=1; i<= hbonds_.size(); ++i ) {
		if ( !allow_hbond(i) ) continue;
		HBond const & hb( hbond(i) );
		AtomID const don_atom( hb.don_hatm(), hb.don_res() );
		AtomID const acc_atom( hb.acc_atm() , hb.acc_res()  );
		for ( Size r=1; r<= 2; ++r ) {
			AtomID const & atom( r == 1 ? don_atom : acc_atom );
			HBondAtomMap::iterator iter( atom_map_.find( atom ) );
			if ( iter == atom_map_.end() ) {
				atom_map_.insert
					( std::make_pair( atom, utility::vector1< HBondCOP >() ));
				iter = atom_map_.find( atom );
			} // atom not already present
			iter->second.push_back( &hb );
		} // repeat for donor and acceptor
	} // i=1.nhbonds
}


///
void
HBondSet::get_residue_residue_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Real & scE,
	Real & bb_scE
) const
{
	scE    = 0.0;
	bb_scE = 0.0;

	Real sr_bbE, lr_bbE; // unused

	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;

	// make sure that setup_for_residue_pair_energies has been called!
	assert( nbrs_.size() > 0 );

	get_residue_residue_hbond_energy
		( rsd1, rsd2, nbrs_[ rsd1.seqpos() ], nbrs_[ rsd2.seqpos() ],
			backbone_backbone_donor_[ rsd1.seqpos() ],
			backbone_backbone_acceptor_[ rsd1.seqpos() ],
			backbone_backbone_donor_[ rsd2.seqpos() ],
			backbone_backbone_acceptor_[ rsd2.seqpos() ],
			true /*exclude_bb_bb*/,
			use_hb_env_dep_, use_hb_env_dep_DNA_, smooth_hb_env_dep_,
			scE, sr_bbE, lr_bbE, bb_scE );


	assert( std::abs( sr_bbE ) < 1e-3 && std::abs( lr_bbE ) < 1e-3 );
}

void
HBondSet::get_residue_residue_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Real & sr_bbE,
	Real & lr_bbE,
	Real & scE,
	Real & bb_scE
) const
{
	scE    = 0.0;
	bb_scE = 0.0;
	sr_bbE = 0.0;
	lr_bbE = 0.0;
	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;

	// make sure that setup_for_residue_pair_energies has been called!
	assert( nbrs_.size() > 0 );

	get_residue_residue_hbond_energy
		( rsd1, rsd2, nbrs_[ rsd1.seqpos() ], nbrs_[ rsd2.seqpos() ],
			backbone_backbone_donor_[ rsd1.seqpos() ],
			backbone_backbone_acceptor_[ rsd1.seqpos() ],
			backbone_backbone_donor_[ rsd2.seqpos() ],
			backbone_backbone_acceptor_[ rsd2.seqpos() ],
			false /*exclude_bb_bb*/,
			use_hb_env_dep_, use_hb_env_dep_DNA_, smooth_hb_env_dep_,
			scE, sr_bbE, lr_bbE, bb_scE);
} // get_residue_residue_energy

///
void
HBondSet::get_residue_residue_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	Real & scE,
	Real & bb1_sc2E,
	Real & bb2_sc1E
) const
{
	scE = 0.0;
	bb1_sc2E = bb2_sc1E = 0.0;
	Real sr_bbE, lr_bbE; // unused
	if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) return;
	get_residue_residue_hbond_energy
		( rsd1, rsd2, nbrs_[ rsd1.seqpos() ], nbrs_[ rsd2.seqpos() ],
			backbone_backbone_donor_[ rsd1.seqpos() ],
			backbone_backbone_acceptor_[ rsd1.seqpos() ],
			backbone_backbone_donor_[ rsd2.seqpos() ],
			backbone_backbone_acceptor_[ rsd2.seqpos() ],
			true /*exclude_bb_bb*/,
			use_hb_env_dep_, use_hb_env_dep_DNA_, smooth_hb_env_dep_,
			scE, sr_bbE, lr_bbE, bb1_sc2E, bb2_sc1E );

	assert( std::abs( sr_bbE ) < 1e-3 && std::abs( lr_bbE ) < 1e-3 );
} // get_residue_residue_energy


///
void
HBondSet::setup_for_residue_pair_energies(
	pose::Pose const & pose,
	bool const calculate_derivative /*= false*/,
	bool const backbone_only /*= true*/
)
{

	// now fill the hbond set with only bb-bb hbonds
	// in the process we fill the arrays that keep track of which protein bb groups are making a
	// bb-bb hbond
	//
	// sc-bb hbonds with these groups are disallowed
	//
	fill_hbond_set( pose, calculate_derivative, *this, backbone_only );

	// stash the nbr info
	TenANeighborGraph const & tenA_neighbor_graph
		( pose.energies().tenA_neighbor_graph() );

	nbrs_.resize( pose.total_residue() );
	for ( Size i=1; i<= pose.total_residue(); ++i ) {
		//nbrs_[i] = tenA_neighbor_graph.get_node(i)->num_neighbors_counting_self();

		{ // hacky thing here until we update to current svn
			nbrs_[i] = 1;
			for ( graph::Graph::EdgeListConstIter
					ir  = tenA_neighbor_graph.get_node( i )->const_edge_list_begin(),
					ire = tenA_neighbor_graph.get_node( i )->const_edge_list_end();
					ir != ire; ++ir ) {
				Size const neighbor_id( (*ir)->get_other_ind( i ) );
				chemical::ResidueType const & nbr_rsd( pose.residue_type( neighbor_id ) );
				if ( nbr_rsd.is_protein() ) {
					nbrs_[i] += 1;
				} else if ( nbr_rsd.is_DNA() ) {
					nbrs_[i] += 1;
// 					nbrs_[i] += 3;
				} else {
					nbrs_[i] += 1;
				}
			}
// 			std::cout << "nbrdiff: old= " << tenA_neighbor_graph.get_node(i)->num_neighbors_counting_self() << " new= " <<
// 				nbrs_[i] << std::endl;
		}
	}

	// ensure full size for the the HbondSet bb_acc/bb_don arrays
	// the bb arrays ought to be resized once at the beginning of this function
	// and not resized to 0 for later push_back operations.... such a waste.
	resize_bb_donor_acceptor_arrays( pose.total_residue() );

	// this is a bug:
	//std::fill( backbone_backbone_donor_.begin(), backbone_backbone_donor_.end(), false );
	//std::fill( backbone_backbone_acceptor_.begin(), backbone_backbone_acceptor_.end(), false );

}

void HBondSet::show(pose::Pose & pose, std::ostream & out) const
{
	core::Size don_pos, acc_pos, don_res, acc_res;
	std::string don_name, acc_name, don_atom_name, acc_atom_name;
	core::Real hbE;

	// This version is formatted for easy parsing by R, Excel, etc.

	out << "# donor atom acceptor atom energy" << std::endl;
	for ( core::Size i=1; i<=(core::Size)nhbonds(); ++i ) {
		if ( allow_hbond((int)i) ) {
			don_pos = hbond((int)i).don_res();
			don_res = pose.pdb_info()->number(don_pos);
			don_name = pose.residue((int)don_pos).type().name3();
			core::id::AtomID const datm ( hbond((int)i).don_hatm(), don_pos );
			don_atom_name = pose.residue((int)don_pos).atom_name(datm.atomno());

			acc_pos = hbond((int)i).acc_res();
			acc_res = pose.pdb_info()->number(acc_pos);
			acc_name = pose.residue((int)acc_pos).type().name3();
			core::id::AtomID const aatm ( hbond((int)i).acc_atm(), acc_pos );
			acc_atom_name = pose.residue((int)acc_pos).atom_name(aatm.atomno());
			hbE = hbond((int)i).energy();

			out << "# " << I(4,don_res) << I(4,don_name) << I(5,don_atom_name) << I(4,acc_res)
			    << I(4,acc_name) << I(5,acc_atom_name) << F(6,2,hbE) << std::endl;
		}
	}
	out << "# end hbond info " << std::endl;
}

utility::vector1< HBondCOP > HBondSet::empty_list_of_hbonds_;

}
}
}

