// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/hbonds/HBondSet.hh
/// @brief  Hydrogen bond set class declaration
/// @author Phil Bradley


#ifndef INCLUDED_core_scoring_hbonds_HBondSet_HH
#define INCLUDED_core_scoring_hbonds_HBondSet_HH


// Unit Headers
#include <core/scoring/hbonds/HBondSet.fwd.hh>


// Package headers
#include <core/scoring/hbonds/types.hh>


// Project headers
#include <core/pose/Pose.fwd.hh>
#include <core/conformation/Residue.hh>
#include <core/id/AtomID.hh>
#include <core/util/datacache/CacheableData.hh>
#include <core/pose/datacache/CacheableDataType.hh>


// Utility headers
//#include <utility/exit.hh>
// #include <utility/pointer/access_ptr.hh>
#include <utility/pointer/owning_ptr.hh>
#include <utility/pointer/ReferenceCount.hh>

#include <numeric/xyzVector.hh>

// C++
#include <map> // what is the right header for std::pair ?
//#include <set>



namespace core {
namespace scoring {
namespace hbonds {

class HBond : public utility::pointer::ReferenceCount {

public:
	typedef std::pair< Vector, Vector > Deriv;


public:
	HBond(
		Size const dhatm,
		bool const dhatm_is_protein_backbone,
		bool const dres_is_protein,
		bool const dres_is_dna,
		bool const dhatm_is_backbone,
		Size const dres,
		Size const aatm,
		bool const aatm_is_protein_backbone,
		bool const ares_is_protein,
		bool const ares_is_dna,
		bool const aatm_is_backbone,
		Size const ares,
		HBEvalType const hbe_type,
		Real const energy_in, // unweighted
		Real const weight_in,
		Deriv const & deriv_in
	):
		don_hatm_( dhatm ),
		don_hatm_is_protein_backbone_( dhatm_is_protein_backbone ),
		don_res_is_protein_( dres_is_protein ),
		don_res_is_dna_( dres_is_dna ),
		don_hatm_is_backbone_( dhatm_is_backbone ),
		don_res_( dres ),
		acc_atm_( aatm ),
		acc_atm_is_protein_backbone_( aatm_is_protein_backbone ),
		acc_res_is_protein_( ares_is_protein ),
		acc_res_is_dna_( ares_is_dna ),
		acc_atm_is_backbone_( aatm_is_backbone ),
		acc_res_( ares ),
		eval_type_( hbe_type ),
		energy_( energy_in ),
		weight_( weight_in ),
		deriv_( deriv_in )
	{}

	///
	Size
	don_res() const
	{
		return don_res_;
	}

	///
	Size
	don_hatm() const
	{
		return don_hatm_;
	}

	/// needed for silly allow logic
	bool
	don_hatm_is_protein_backbone() const
	{
		return don_hatm_is_protein_backbone_;
	}

	bool
	don_res_is_protein() const
	{
		return don_res_is_protein_;
	}

	bool
	don_res_is_dna() const
	{
		return don_res_is_dna_;
	}

	/// needed for silly allow logic
	bool
	don_hatm_is_backbone() const
	{
		return don_hatm_is_backbone_;
	}

	///
	Size
	acc_res() const
	{
		return acc_res_;
	}

	///
	Size
	acc_atm() const
	{
		return acc_atm_;
	}

	/// needed for silly allow logic
	bool
	acc_atm_is_protein_backbone() const
	{
		return acc_atm_is_protein_backbone_;
	}

	bool
	acc_res_is_protein() const
	{
		return acc_res_is_protein_;
	}

	bool
	acc_res_is_dna() const
	{
		return acc_res_is_dna_;
	}

	/// needed for silly allow logic
	bool
	acc_atm_is_backbone() const
	{
		return acc_atm_is_backbone_;
	}

	/// NOTE: this is unweighted energy, see weight() for the weight
	Real
	energy() const
	{
		return energy_;
	}

	///
	Real
	weight() const
	{
		return weight_;
	}


	///
	Deriv const &
	deriv() const
	{
		return deriv_;
	}


	///
	HBEvalType const &
	eval_type() const
	{
		return eval_type_;
	}

	///
	bool
	atom_is_donorH( id::AtomID const & atom ) const
	{
		return ( atom.rsd() == don_res_ && atom.atomno() == don_hatm_ );
	}


	///
	bool
	atom_is_acceptor( id::AtomID const & atom ) const
	{
		return ( atom.rsd() == acc_res_ && atom.atomno() == acc_atm_ );
	}


	///////
	// data
private:

	Size const don_hatm_;
	bool const don_hatm_is_protein_backbone_;
	bool const don_res_is_protein_;
	bool const don_res_is_dna_;
	bool const don_hatm_is_backbone_;
	Size const don_res_;
	Size const acc_atm_;
	bool const acc_atm_is_protein_backbone_;
	bool const acc_res_is_protein_;
	bool const acc_res_is_dna_;
	bool const acc_atm_is_backbone_;
	Size const acc_res_;
	HBEvalType const eval_type_;
	Real const energy_;
	Real const weight_;
	Deriv const deriv_;

};

typedef utility::pointer::owning_ptr< HBond > HBondOP;
typedef utility::pointer::owning_ptr< HBond const > HBondCOP;


//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////
// HBondSet
//////////////////////////////////////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////////////////////////////////////

class HBondSet : public util::datacache::CacheableData {

public:
	HBondSet();
	HBondSet( Size );

	HBondSet( HBondSet const & src );


	// typedefs
	typedef id::AtomID AtomID;

public:

	/// @brief Clone this object
	util::datacache::CacheableDataOP
	clone() const;

	/// \brief  Number of hbonds
	Size
	nhbonds() const
	{
		return hbonds_.size();
	}


	// for accessing the nbrs, allows hacky non-tenA_nbr count
	int
	nbrs( Size const seqpos ) const
	{
		return nbrs_[ seqpos ];
	}

	void
  set_nbrs( Size const seqpos, Size value )
  {
    nbrs_[ seqpos ] = value;
  }

	/// \brief  Access hbond
	HBond const &
	hbond( Size const number ) const
	{
		return *(hbonds_[ number ]);
	}

	/// \brief  Add a new hbond to the list
	/// updates the "hbchk" array as necessary
	void
	append_hbond(
		Size const dhatm,
		conformation::Residue const & don_rsd,
		Size const aatm,
		conformation::Residue const & acc_rsd,
		HBEvalType const & hbe_type,
		Real const energy,
		Real const weight,
		HBond::Deriv const & deriv
	);

	/// \brief  Is this hbond allowed under the bb-bb exclusion scheme?
	bool
	allow_hbond( Size const index ) const
	{
		return allow_hbond( *hbonds_[ index ] );
	}

	bool
	allow_hbond( HBond const & hbond ) const
	{
		// donor is backbone atom already making bb-bb hbond
		// acceptor is sidechain
		if ( hbond.don_hatm_is_protein_backbone() &&
				 !hbond.acc_atm_is_protein_backbone() &&
				 backbone_backbone_donor_[ hbond.don_res() ] ) return false;

		if ( hbond.acc_atm_is_protein_backbone() &&
				 !hbond.don_hatm_is_protein_backbone() &&
				 backbone_backbone_acceptor_[ hbond.acc_res() ] ) return false;

		return true;
	}

	/// @brief is the backbone bone acceptor group in a bb/bb hydrogen bond?
	bool
	acc_bbg_in_bb_bb_hbond( Size const residue ) const
	{
		return backbone_backbone_acceptor_[ residue ];
	}

	/// @brief is the backbone bone donor group in a bb/bb hydrogen bond?
	bool
	don_bbg_in_bb_bb_hbond( Size const residue ) const
	{
		return backbone_backbone_donor_[ residue ];
	}

	/// @Manually set the state of backbone-backbone acceptor. Used for symmetry.
  void
  set_backbone_backbone_acceptor( Size const residue, bool state )
  {
    backbone_backbone_acceptor_[ residue ] = state;
  }

	/// @Manually set the state of backbone-backbone donor. Used for symmetry.
  void
  set_backbone_backbone_donor( Size const residue, bool state )
  {
    backbone_backbone_donor_[ residue ] = state;
  }

	/// @brief  Setup the mapping from atoms to hbonds
	void
	setup_atom_map() const;

	/// \brief  Get a vector of all the hbonds involving this atom
	utility::vector1< HBondCOP > const &
	atom_hbonds( AtomID const & atom ) const
	{
		setup_atom_map();
		HBondAtomMap::const_iterator iter( atom_map_.find( atom ) );
		if ( iter == atom_map_.end() ) {
			return empty_list_of_hbonds_;
		} else return iter->second;
	}


	/// \brief  Delete all the data
	void
	clear()
	{
		atom_map_init_ = false;
		hbonds_.clear();
		backbone_backbone_donor_.clear();
		backbone_backbone_acceptor_.clear();
		nbrs_.clear();
	}

	/// \brief Resize bb info arrays
	void
	resize_bb_donor_acceptor_arrays( Size const new_dimension )
	{
		backbone_backbone_donor_.resize( new_dimension, false );
		backbone_backbone_acceptor_.resize( new_dimension, false );
	}

	void
	copy_bb_donor_acceptor_arrays( HBondSet const & src ) {
		backbone_backbone_donor_ = src.backbone_backbone_donor_;
		backbone_backbone_acceptor_ = src.backbone_backbone_acceptor_;
	}

	/// @brief get residue-residue hbond energies, separated into components for short-range backbone,
	/// long-range backbone, sidechain-sidechain, and backbone-sidechain interactions.
	void
	get_residue_residue_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real & sr_bbE,
		Real & lr_bbE,
		Real & scE,
		Real & bb_scE
	) const;

	/// @brief get residue-residue hbond energies for sidechain-sidechain and backbone-sidechain interactions.
	void
	get_residue_residue_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real & scE,
		Real & bb_scE
	) const;

	///
	void
	get_residue_residue_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		Real & scE,
		Real & bb1_sc2E,
		Real & bb2_sc1E
	) const;


	///
	void
	setup_for_residue_pair_energies(
		pose::Pose const & pose,
		bool const calculate_derivative = false,
		bool const backbone_only = true
	);

	///
	void
	sort_by_weighted_energy();

	///
	bool
	exclude_DNA_DNA() const
	{
		return exclude_DNA_DNA_;
	}

	///
	void
	exclude_DNA_DNA( bool const setting )
	{
		exclude_DNA_DNA_ = setting;
	}

	///
	bool
	use_hb_env_dep() const
	{
		return use_hb_env_dep_;
	}

	///
	void
	use_hb_env_dep( bool const setting )
	{
		use_hb_env_dep_ = setting;
	}

	bool use_hb_env_dep_DNA() const { return use_hb_env_dep_DNA_; }
	void use_hb_env_dep_DNA( bool setting ) { use_hb_env_dep_DNA_ = setting; }

	///
	bool
	smooth_hb_env_dep() const
	{
		return smooth_hb_env_dep_;
	}

	///
	void
	smooth_hb_env_dep( bool const setting )
	{
		smooth_hb_env_dep_ = setting;
	}

	/// @brief various show functions
	void show(pose::Pose & pose, std::ostream & out) const;

	/// @brief This function proveded mostly to simplify usage of PyRosetta
	void show(pose::Pose & pose) const { show(pose, std::cout); };


private:
	typedef std::map< AtomID, utility::vector1< HBondCOP > > HBondAtomMap;

	////////
	// data ---- IF YOU ADD DATA ALSO ADD IT TO THE COPY C-TOR
private:
	utility::vector1< HBondOP > hbonds_;
	utility::vector1< bool > backbone_backbone_donor_;
	utility::vector1< bool > backbone_backbone_acceptor_;
	utility::vector1< int > nbrs_;
	mutable HBondAtomMap atom_map_;
	mutable bool atom_map_init_;
	bool exclude_DNA_DNA_;
	bool exclude_bb_sc_;
	bool use_hb_env_dep_;
	bool use_hb_env_dep_DNA_;
	bool smooth_hb_env_dep_;

	static utility::vector1< HBondCOP > empty_list_of_hbonds_;
};

typedef utility::pointer::owning_ptr< HBondSet > HBondSetOP; //belongs in HBondSet.fwd.hh



} // namespace hbonds
} // namespace scoring
} // namespace core


#endif // INCLUDED_core_scoring_ScoreFunction_HH

