// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author


// Unit headers
#include <core/scoring/hbonds/hbonds.hh>

// Package headers
#include <core/scoring/hbonds/constants.hh>
#include <core/scoring/hbonds/HBondSet.hh>
#include <core/scoring/hbonds/hbonds_geom.hh>

// // Project headers
#include <core/conformation/Residue.hh>
#include <core/graph/Graph.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/EnergyGraph.hh>
#include <core/scoring/TenANeighborGraph.hh>
#include <core/scoring/ScoreType.hh>

#include <ObjexxFCL/formatted.o.hh>

//#include <core/scoring/Energies.hh>

// // Numeric headers
// #include <numeric/numeric.functions.hh>

/// PB temp hack
//bool const exclude_DNA_DNA( true );//false );

using namespace ObjexxFCL;

namespace core {
namespace scoring {
namespace hbonds {

///////////////////////////////////////////////////////////////////////////////
// void
// hbond_energy(
// 	pose::Pose & pose, // non-const since we modify stored energies
// 	ScoreFunction const & scorefxn
// )
// {
// 	if ( !nonzero_hbond_weight( scorefxn ) ) return;

// 	HBondSet hbond_set;
// 	fill_hbond_set( pose, false /* calc-deriv */, hbond_set );
// 	Real sr_bb_hbenergy, lr_bb_hbenergy, bb_sc_hbenergy, sc_hbenergy;
// 	Energies::Energy1D hbenergy( pose.total_residue(), 0.0 );
// 	get_hbond_energies( hbond_set, sr_bb_hbenergy, lr_bb_hbenergy,
// 											bb_sc_hbenergy, sc_hbenergy, pose.total_residue(),
// 											hbenergy );

// 	pose.energies().set_total_energy( scoring::hbond_sr_bb, sr_bb_hbenergy );
// 	pose.energies().set_total_energy( scoring::hbond_lr_bb, lr_bb_hbenergy );
// 	pose.energies().set_total_energy( scoring::hbond_bb_sc, bb_sc_hbenergy );
// 	pose.energies().set_total_energy( scoring::hbond_sc, sc_hbenergy );
// 	pose.energies().set_twobody_residue_energies( scoring::hbond, hbenergy );
// }

///////////////////////////////////////////////////////////////////////////////
/**
	 This routine fills an hbond-set with hbonds. All hbonds are included,
	 even ones which might be excluded later based on the backbone-hbond
	 exclusion.

	 The pose must have an update energies object, eg it must be scored.
**/

void
fill_hbond_set(
	pose::Pose const & pose,
	bool const calculate_derivative,
	HBondSet & hbond_set,
	bool const backbone_only // = false
)
{
	// clear old data
	hbond_set.clear();

	// need to know which residues are neighbors
	// and what the neighbor-numbers are for each residue since some of the
	// weights are environment-dependent.
	EnergyGraph const & energy_graph( pose.energies().energy_graph() );
	TenANeighborGraph const & tenA_neighbor_graph( pose.energies().tenA_neighbor_graph() );

	bool const exclude_DNA_DNA( hbond_set.exclude_DNA_DNA() );

	// loop over all nbr-pairs
	for ( Size res1 = 1; res1 <= pose.total_residue(); ++res1 ) {
		int const nb1 = tenA_neighbor_graph.get_node( res1 )->num_neighbors_counting_self_static();
		conformation::Residue const & rsd1( pose.residue( res1 ) );

		for ( graph::Graph::EdgeListConstIter
				iru = energy_graph.get_node(res1)->const_upper_edge_list_begin(),
				irue = energy_graph.get_node(res1)->const_upper_edge_list_end();
				iru != irue; ++iru ) {

			int const res2( (*iru)->get_second_node_ind() );

			conformation::Residue const & rsd2( pose.residue( res2 ) );
			if ( exclude_DNA_DNA && rsd1.is_DNA() && rsd2.is_DNA() ) continue;

			int const nb2 = tenA_neighbor_graph.get_node( res2 )->num_neighbors_counting_self_static();

			// rsd1 as donor, rsd2 as acceptor
			get_residue_residue_hbonds_1way( rsd1, rsd2, nb1, nb2,
				calculate_derivative, backbone_only, hbond_set );

			// rsd2 as donor, rsd1 as acceptor
			get_residue_residue_hbonds_1way( rsd2, rsd1, nb2, nb1,
				calculate_derivative, backbone_only, hbond_set );

		} // nbrs of res1
	} // res1
}

///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
void
use_generic_rule_to_fill_hb_energies(
		Real const & energy_weight,
		Size const & ares,
		Size const & dres,
		bool const & aatm_is_bb,
		bool const & datm_is_bb,
		Real & scE,
		Real & sr_bbE,
		Real & lr_bbE,
		Real & bb_scE,
		Real & sc1_bb2_scdon,
		Real & sc2_bb1_scacc )
{
	//				std::cout << "CHECKING NON-PROTEIN " << aatm_is_bb << " " << datm_is_bb << " " << energy_weight << std::endl;
	if ( aatm_is_bb && datm_is_bb ){
		if ( std::abs( int(ares) - int(dres) ) <= GENERIC_SHORTRANGE_SEQ_CUTOFF) {
			sr_bbE += energy_weight;
		} else {
			lr_bbE += energy_weight;
		}
	} else if (datm_is_bb ) {
		bb_scE += energy_weight;
		sc2_bb1_scacc += energy_weight;
	} else if (aatm_is_bb ) {
		bb_scE += energy_weight;
		sc1_bb2_scdon += energy_weight;
	} else {
		scE += energy_weight;
	}
}



/// Mike, note to self - merge this with above function later.
void
get_hbond_energies_excl(
	HBondSet const & hbond_set,
	Real & sr_bb_hbenergy, // short range bb-bb total
	Real & lr_bb_hbenergy, // long range bb-bb total
	Real & bb_sc_hbenergy, // sc-bb total
	Real & sc_hbenergy, // sc-sc total
	utility::vector1< core::Size > exclude_list
)
{

	// zero the energy totals
	sr_bb_hbenergy = 0.0;
	lr_bb_hbenergy = 0.0;
	bb_sc_hbenergy = 0.0;
	sc_hbenergy = 0.0;

	bool exclude=false;
	for ( Size i = 1; i <= hbond_set.nhbonds(); ++i ) {
		HBond const & hbond( hbond_set.hbond(i) );
		int const dres  = hbond.don_res();
		int const ares  = hbond.acc_res();
		exclude = false;
		for ( Size k = 1; k <= exclude_list.size(); k ++ ){
			if( exclude_list[k] == (core::Size)dres ){ exclude = true; break; };
			if( exclude_list[k] == (core::Size)ares ){ exclude = true; break; };
		}
		if( exclude ) continue;

		//int const dhatm = hbond.don_hatm();
		//int const aatm  = hbond.acc_atm();
		HBEvalType const hbe_type = hbond.eval_type();

		Real hbE = hbond.energy() /*raw energy*/ * hbond.weight() /*env-dep-wt*/;

		// hbond_set has compiled "hbchk" arrays in the process of filling
		if ( hbond_set.allow_hbond(i) ) {

			// Following is general for non-protein residues.
			if  ( (!hbond.don_res_is_protein() && !hbond.don_res_is_dna()) ||
						(!hbond.acc_res_is_protein() && !hbond.acc_res_is_dna()) ) 	{
				static Real dummy( 0.0 );
				use_generic_rule_to_fill_hb_energies( hbE, hbond.acc_res(), hbond.don_res(),
																							hbond.acc_atm_is_backbone(), hbond.don_hatm_is_backbone(),
																							sc_hbenergy, sr_bb_hbenergy, lr_bb_hbenergy, sc_hbenergy,
																							dummy, dummy );
				continue;
 			}

			//Following is protein-centric. For example, OH...O bonds get counted
			// as "sidechain" even though they're possibly backbone hydrogen bonds
			// in nucleic acids.
			switch ( get_hbond_weight_type( hbe_type ) ) {
			case hbw_SR_BB:
				sr_bb_hbenergy += hbE;
				break;
			case hbw_LR_BB:
				lr_bb_hbenergy += hbE;
				break;
			case hbw_BB_SC:
				bb_sc_hbenergy += hbE;
				break;
			case hbw_SC:
				sc_hbenergy += hbE;
				break;
			default:
				std::cout << "Warning: energy from unexpected HB type ignored " <<
					hbe_type << std::endl;
				break;
			}

			//hbE *= 0.5f;
			//hbenergy[ dres ] += hbE; // note hbenergy is utility::vector1
			//hbenergy[ ares ] += hbE;
		} // allow_hbond(i) ?

	} // i=1,nhbonds
}


void
get_hbond_energies(
	HBondSet const & hbond_set,
	Real & sr_bb_hbenergy, // short range bb-bb total
	Real & lr_bb_hbenergy, // long range bb-bb total
	Real & bb_sc_hbenergy, // sc-bb total
	Real & sc_hbenergy // sc-sc total

)
{

	// zero the energy totals
	sr_bb_hbenergy = 0.0;
	lr_bb_hbenergy = 0.0;
	bb_sc_hbenergy = 0.0;
	sc_hbenergy = 0.0;

	for ( Size i = 1; i <= hbond_set.nhbonds(); ++i ) {
		HBond const & hbond( hbond_set.hbond(i) );
		//int const dres  = hbond.don_res();
		//int const ares  = hbond.acc_res();
		//int const dhatm = hbond.don_hatm();
		//int const aatm  = hbond.acc_atm();
		HBEvalType const hbe_type = hbond.eval_type();

		Real hbE = hbond.energy() /*raw energy*/ * hbond.weight() /*env-dep-wt*/;

		// hbond_set has compiled "hbchk" arrays in the process of filling
		if ( hbond_set.allow_hbond(i) ) {

			// Following is general for non-protein residues.
			if  ( (!hbond.don_res_is_protein() && !hbond.don_res_is_dna()) ||
						(!hbond.acc_res_is_protein() && !hbond.acc_res_is_dna()) ) 	{
				static Real dummy( 0.0 );
				use_generic_rule_to_fill_hb_energies( hbE, hbond.acc_res(), hbond.don_res(),
																							hbond.acc_atm_is_backbone(), hbond.don_hatm_is_backbone(),
																							sc_hbenergy, sr_bb_hbenergy, lr_bb_hbenergy, sc_hbenergy,
																							dummy, dummy );
				continue;
 			}

			//Following is protein-centric. For example, OH...O bonds get counted
			// as "sidechain" even though they're possibly backbone hydrogen bonds
			// in nucleic acids.
			switch ( get_hbond_weight_type( hbe_type ) ) {
			case hbw_SR_BB:
				sr_bb_hbenergy += hbE;
				break;
			case hbw_LR_BB:
				lr_bb_hbenergy += hbE;
				break;
			case hbw_BB_SC:
				bb_sc_hbenergy += hbE;
				break;
			case hbw_SC:
				sc_hbenergy += hbE;
				break;
			default:
				std::cout << "Warning: energy from unexpected HB type ignored " <<
					hbe_type << std::endl;
				break;
			}

			//hbE *= 0.5f;
			//hbenergy[ dres ] += hbE; // note hbenergy is utility::vector1
			//hbenergy[ ares ] += hbE;
		} // allow_hbond(i) ?

	} // i=1,nhbonds
}

///////////////////////////////////////////////////////////////////////////////

/// Get the appropriate scorefxn weight for this type of hbond

Real
hb_eval_type_weight(
	HBEvalType const & hbe_type,
	EnergyMap const & weights
)
{

	switch ( get_hbond_weight_type( hbe_type ) ) {
	case hbw_SR_BB:
		return weights[ scoring::hbond_sr_bb ];
	case hbw_LR_BB:
		return weights[ scoring::hbond_lr_bb ];
	case hbw_BB_SC:
		return weights[ scoring::hbond_bb_sc ];
	case hbw_SC:
		return weights[ scoring::hbond_sc ];
	case hbw_H2O:
		return weights[ scoring::h2o_hbond ];
	default:
		std::cout << "Warning: energy from unexpected HB type ignored " <<
			hbe_type << std::endl;
		break;
	}
	return 0.0;
}


/// @brief Computes both the energy and the environment weight for a pair of atoms
/// the unweighted energy returned is set to 0 if no hydrogen bond exists.
inline
void
get_hbenergy_for_atom_pair(
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd,
	Size const /*hatm*/, // apparently, unneeded
	Size const datm,
	Vector const & hatm_xyz,
	Vector const & datm_xyz,
	Size const aatm,
	int const don_nb,
	int const acc_nb,
	bool const evaluate_derivative,
	HBEvalType & hbe_type,
	Energy & unweighted_energy,
	Energy & environment_weight,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep,
	std::pair< Vector, Vector > & deriv
)
{
	hbe_type = hbond_evaluation_type( datm, don_rsd, aatm, acc_rsd );

	environment_weight = 1; // overwritten if a real hydrogen bond is discovered

	int const base ( acc_rsd.atom_base( aatm ) );
	int const base2( acc_rsd.abase2( aatm ) );
	assert( base2 > 0 && base != base2 );

	if ( evaluate_derivative ) {
		hb_energy_deriv( hbe_type, datm_xyz, hatm_xyz,
			acc_rsd.atom(aatm ).xyz(),
			acc_rsd.atom(base ).xyz(),
			acc_rsd.atom(base2).xyz(),
			unweighted_energy, true, deriv );
	} else {
		hb_energy_deriv( hbe_type, datm_xyz, hatm_xyz,
			acc_rsd.atom(aatm ).xyz(),
			acc_rsd.atom(base ).xyz(),
			acc_rsd.atom(base2).xyz(),
			unweighted_energy );
	}

	if ( unweighted_energy >= MAX_HB_ENERGY ) { unweighted_energy = 0; return; } // no hbond

	if (false) {
		std::cout << "HBOND: " << don_rsd.name1() << don_rsd.seqpos() <<
			" " << don_rsd.atom_name( datm ) << " --- " <<
			acc_rsd.name1() << acc_rsd.seqpos() << " " << acc_rsd.atom_name( aatm ) << " ==> " << unweighted_energy
							<< std::endl;
		}

	if ( use_hb_env_dep ) environment_weight = ( get_environment_dependent_weight( hbe_type, don_nb, acc_nb, smooth_hb_env_dep ) );

	if ( ! use_hb_env_dep_DNA ) {
		// avoid neighbor-based attentuation for hbonds involving DNA residues
		if ( don_rsd.is_DNA() || acc_rsd.is_DNA() ) environment_weight = 1.0;
	}

// 			std::cout << "hbond: " << evaluate_derivative << ' ' <<
// 				don_rsd.seqpos() << ' ' << hatm /*don_rsd.atom_name( hatm )*/ <<' '<<
// 				acc_rsd.seqpos() << ' ' << aatm /*acc_rsd.atom_name( aatm )*/ <<
// 				std::endl;
}


void
get_residue_residue_hbonds_1way(
	// input
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd,
	int const don_nb,
	int const acc_nb,
	bool const evaluate_derivative,
	bool const backbone_only,
	// output
	HBondSet & hbond_set
)
{

	assert( don_rsd.seqpos() != acc_rsd.seqpos() ); // otherwise include in allow

	// <f1,f2> -- derivative vectors
	std::pair< Vector, Vector > deriv( Vector(0.0), Vector(0.0 ) );

	//bk cycle through polar hydrogens in don_aa, acceptors in acc_aa
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_polar().begin(),
					hnume = don_rsd.Hpos_polar().end(); hnum != hnume; ++hnum ) {
		Size const hatm( *hnum );

		if ( backbone_only && !don_rsd.atom_is_backbone( hatm ) ) continue;

		int const datm( don_rsd.atom_base( hatm ) );
		assert( !backbone_only || don_rsd.atom_is_backbone( datm ) );

		Vector const & hatm_xyz( don_rsd.atom( hatm ).xyz() );
		Vector const & datm_xyz( don_rsd.atom( datm ).xyz() );

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos().begin(),
						anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {

			Size const aatm( *anum );

			if ( backbone_only && !acc_rsd.atom_is_backbone( aatm ) ) continue;

			// early exit -- too much time is spent in determining the eval type
			// of non-existant hbonds.  This square distance is calculated a second time
			// inside hb_energy_deriv, but this inefficiency is more than made up for.
			if ( hatm_xyz.distance_squared( acc_rsd.xyz( aatm ) ) > MAX_R2 ) continue;

			Real unweighted_energy( 0.0 ), environment_weight( 0.0 );
			HBEvalType hbe_type;

			get_hbenergy_for_atom_pair( don_rsd, acc_rsd,
				hatm, datm, hatm_xyz, datm_xyz, aatm,
				don_nb, acc_nb, evaluate_derivative,
				hbe_type, unweighted_energy, environment_weight,
				hbond_set.use_hb_env_dep(), hbond_set.use_hb_env_dep_DNA(),
				hbond_set.smooth_hb_env_dep(), deriv );

			if ( unweighted_energy < 0.0 ) {
				hbond_set.append_hbond( hatm, don_rsd, aatm, acc_rsd,
					hbe_type, unweighted_energy, environment_weight,
					deriv );
			}
		} // loop over acceptors
	} // loop over donors

}

///////////////////////////////////////////////////////////////////////////////

/// residue-residue hbond energy, used in fullatom_energy
/**
	 compiles list of hbonds
	 optionally evaluates the derivative
**/


inline
Energy
get_sc_sc_hbond_energies_1way(
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd,
	int const don_nb,
	int const acc_nb,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep
)
{
	assert( don_rsd.seqpos() != acc_rsd.seqpos() ); // otherwise include in allow

	// <f1,f2> -- dummy derivative vectors
	std::pair< Vector, Vector > deriv( Vector(0.0), Vector(0.0 ) );
	Energy total_energy( 0.0 );

	//bk cycle through polar hydrogens in don_aa, acceptors in acc_aa
	for ( chemical::AtomIndices::const_iterator
					hnum  = don_rsd.Hpos_polar_sc().begin(),
					hnume = don_rsd.Hpos_polar_sc().end(); hnum != hnume; ++hnum ) {
		Size const hatm( *hnum );
		int const datm( don_rsd.atom_base( hatm ) );

		Vector const & hatm_xyz( don_rsd.atom( hatm ).xyz() );
		Vector const & datm_xyz( don_rsd.atom( datm ).xyz() );

		for ( chemical::AtomIndices::const_iterator
						anum  = acc_rsd.accpt_pos_sc().begin(),
						anume = acc_rsd.accpt_pos_sc().end(); anum != anume; ++anum ) {
			Size const aatm( *anum );

			// early exit -- too much time is spent in determining the eval type
			// of non-existant hbonds.  This square distance is calculated a second time
			// inside hb_energy_deriv, but this inefficiency is more than made up for.
			if ( hatm_xyz.distance_squared( acc_rsd.xyz( aatm ) ) > MAX_R2 ) continue;

			Real unweighted_energy( 0.0 ), environment_weight( 0.0 );
			HBEvalType hbe_type;

			get_hbenergy_for_atom_pair( don_rsd, acc_rsd,
				hatm, datm, hatm_xyz, datm_xyz, aatm,
				don_nb, acc_nb, false /*evaluate_derivative*/,
				hbe_type, unweighted_energy, environment_weight,
				use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep, deriv );


			//if ( unweighted_energy < 0 ) {
			//	std::cerr << "SCSC Hbond: " << don_rsd.name() << " " << acc_rsd.name() << " " << don_rsd.atom_name( datm ) << " " << acc_rsd.atom_name( aatm ) << " " << unweighted_energy * environment_weight << std::endl;
			//}
			total_energy += unweighted_energy * environment_weight;
		} // loop over sidechain acceptors
	} // loop over sidechain donors
	return total_energy;
}


///////////////////////////////////////////////////////////////////////////////
// called by packer, rotamer trials, perhaps?
//
void
get_residue_residue_hbond_energy(
	// input
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	int const nb1,
	int const nb2,
	bool const bb1_is_donor,
	bool const bb1_is_acceptor,
	bool const bb2_is_donor,
	bool const bb2_is_acceptor,
	bool const exclude_bb_bb,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep,
	// energies -- zeroed
	Real & scE,
	Real & sr_bbE,
	Real & lr_bbE,
	Real & bb_scE
)
{
	scE = 0.0;
	sr_bbE = 0.0;
	lr_bbE = 0.0;
	bb_scE = 0.0;

	Real tmp1(0.0), tmp2(0.0);

	get_residue_residue_hbond_energy_1way
		( rsd1, rsd2, nb1, nb2, bb1_is_donor, bb2_is_acceptor,
			exclude_bb_bb,
			use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep,
			scE, sr_bbE, lr_bbE, bb_scE,
			tmp1, tmp2 );

	get_residue_residue_hbond_energy_1way
		( rsd2, rsd1, nb2, nb1, bb2_is_donor, bb1_is_acceptor,
			exclude_bb_bb,
			use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep,
			scE, sr_bbE, lr_bbE, bb_scE,
			tmp1, tmp2 );

}


/// @details useful for initializing an OTF interaction graph.
void
get_residue_residue_hbond_energy(
	// input
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	int const nb1,
	int const nb2,
	bool const bb1_is_donor,
	bool const bb1_is_acceptor,
	bool const bb2_is_donor,
	bool const bb2_is_acceptor,
	bool const exclude_bb_bb,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep,
	// energies -- zeroed
	Real & scE,
	Real & sr_bbE,
	Real & lr_bbE,
	Real & bb1_sc2_E,
	Real & bb2_sc1_E
)
{
	scE = 0.0;
	sr_bbE = 0.0;
	lr_bbE = 0.0;
	bb1_sc2_E = bb2_sc1_E = 0.0;

	Real tmp_bb_scE(0.0);

	get_residue_residue_hbond_energy_1way
		( rsd1, rsd2, nb1, nb2, bb1_is_donor, bb2_is_acceptor,
			exclude_bb_bb,
			use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep,
			scE, sr_bbE, lr_bbE, tmp_bb_scE, bb2_sc1_E, bb1_sc2_E );

	get_residue_residue_hbond_energy_1way
		( rsd2, rsd1, nb2, nb1, bb2_is_donor, bb1_is_acceptor,
			exclude_bb_bb,
			use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep,
			scE, sr_bbE, lr_bbE, tmp_bb_scE, bb1_sc2_E, bb2_sc1_E );
}

Energy
get_sc_sc_hbond_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	int const nb1,
	int const nb2,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep
)
{
	Energy total( 0.0 );
	if ( rsd1.Hpos_polar_sc().size() > 0 && rsd2.accpt_pos_sc().size() > 0 ) {
		total += get_sc_sc_hbond_energies_1way( rsd1, rsd2, nb1, nb2, use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep );
	}
	if ( rsd2.Hpos_polar_sc().size() > 0 && rsd1.accpt_pos_sc().size() > 0 ) {
		total += get_sc_sc_hbond_energies_1way( rsd2, rsd1, nb2, nb1, use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep );
	}

	return total;

}


///////////////////////////////////////////////////////////////////////////////
void
eval_atom_hbond_derivative(
	HBond const & hbond,
	Real const & sign_factor,
	EnergyMap const & weights,
	Vector & f1,
	Vector & f2
)
{

	if ( ( hbond.acc_res_is_protein() || hbond.acc_res_is_dna()) &&
			 ( hbond.don_res_is_protein() || hbond.don_res_is_dna())  ) {
		//Standard protein stuff...
		Real const weight
			( sign_factor * hbond.weight() *
				hb_eval_type_weight( hbond.eval_type(), weights ) );
		f1 += weight * hbond.deriv().first;
		f2 += weight * hbond.deriv().second;
	} else {
		if ( hbond.acc_atm_is_backbone() && hbond.don_hatm_is_backbone() ) {
			if ( std::abs( int(hbond.acc_res()) - int(hbond.don_res()) ) <= GENERIC_SHORTRANGE_SEQ_CUTOFF) {
				f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_sr_bb ] * hbond.deriv().first;
				f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_sr_bb ] * hbond.deriv().second;
			} else {
				f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_lr_bb ] * hbond.deriv().first;
				f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_lr_bb ] * hbond.deriv().second;
			}
		} else if (hbond.acc_atm_is_backbone() || hbond.don_hatm_is_backbone() ) {
			f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_bb_sc ] * hbond.deriv().first;
			f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_bb_sc ] * hbond.deriv().second;
			if ( std::abs( int(hbond.acc_res()) - int(hbond.don_res()) ) <= GENERIC_SHORTRANGE_SEQ_CUTOFF) {
				f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_sr_bb_sc ] * hbond.deriv().first;
				f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_sr_bb_sc ] * hbond.deriv().second;
			} else {
				f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_lr_bb_sc ] * hbond.deriv().first;
				f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_lr_bb_sc ] * hbond.deriv().second;
			}
		} else {
			f1 += sign_factor * hbond.weight() * weights[ scoring::hbond_sc ] * hbond.deriv().first;
			f2 += sign_factor * hbond.weight() * weights[ scoring::hbond_sc ] * hbond.deriv().second;
		}
	}

}


/// \brief  Get the f1 and f2 contributions from all hbonds involving this atom

void
get_atom_hbond_derivative(
	id::AtomID const & atom,
	HBondSet const & hbond_set,
	EnergyMap const & weights,
	Vector & f1,
	Vector & f2
)
{
	f1 = Vector(0.0);
	f2 = Vector(0.0);

	utility::vector1< HBondCOP > const & hbonds
		( hbond_set.atom_hbonds( atom ) );

	for ( Size i=1; i<= hbonds.size(); ++i ) {
		HBond const & hbond( *hbonds[ i ] );
		Real sign_factor( 0.0 );
		if ( hbond.atom_is_donorH( atom ) ) sign_factor = 1.0;
		else {
			assert( hbond.atom_is_acceptor( atom ) );
			sign_factor = -1;
		}
		// get the appropriate type of hbond weight

		eval_atom_hbond_derivative( hbond, sign_factor, weights, f1, f2 );
	}
}


///////////////////////////////////////////////////////////////////////////////
/// residue-residue hbond energy, used in packing and rotamer trials
/**
	 this just computes energies, does not tabulate list of hbonds
	 used in rotamer-rotamer scoring, does not calculate derivatives
**/

void
get_residue_residue_hbond_energy_1way(
	// input
	conformation::Residue const & don_rsd,
	conformation::Residue const & acc_rsd,
	int const don_nb,
	int const acc_nb,
	bool const don_bb_hbond_exists,
	bool const acc_bb_hbond_exists,
	bool const exclude_bb_bb,
	bool const use_hb_env_dep,
	bool const use_hb_env_dep_DNA,
	bool const smooth_hb_env_dep,
	// energies -- not zeroed, just accumulated
	Real & scE,
	Real & sr_bbE,
	Real & lr_bbE,
	Real & bb_scE,
	Real & sc1_bb2_scdon, // does anyone use this? // apl -- I use this with the OTF code in the packer
	Real & sc2_bb1_scacc
)
{

	assert( don_rsd.seqpos() != acc_rsd.seqpos() ); // otherwise include in allow

	std::pair< Vector, Vector > deriv( Vector(0.0), Vector(0.0 ) );

	bool const don_rsd_is_protein( don_rsd.is_protein() );
	bool const acc_rsd_is_protein( acc_rsd.is_protein() );

	//bk cycle through polar hydrogens in don_aa, acceptors in acc_aa
	for ( chemical::AtomIndices::const_iterator
			hnum  = don_rsd.Hpos_polar().begin(),
			hnume = don_rsd.Hpos_polar().end(); hnum != hnume; ++hnum ) {
		int const hatm( *hnum );
		int const datm( don_rsd.atom_base( hatm ) );
		bool const datm_is_bb = don_rsd.atom_is_backbone( datm );

		Vector const & hatm_xyz( don_rsd.atom( hatm ).xyz() );
		Vector const & datm_xyz( don_rsd.atom( datm ).xyz() );

		for ( chemical::AtomIndices::const_iterator
				anum  = acc_rsd.accpt_pos().begin(),
				anume = acc_rsd.accpt_pos().end(); anum != anume; ++anum ) {
			Size const aatm( *anum );
			bool const aatm_is_bb = acc_rsd.atom_is_backbone(aatm); //jss to remove

			if ( exclude_bb_bb && datm_is_bb && aatm_is_bb ) continue;

			//bk don't allow sidechain backbone hydrogen bonds withs backbone atoms
			//bk that are already hydrogen bonded

			if ( ( acc_bb_hbond_exists && aatm_is_bb && acc_rsd_is_protein &&
					!( don_rsd_is_protein && datm_is_bb) ) ||
					( don_bb_hbond_exists && datm_is_bb && don_rsd_is_protein &&
					!( acc_rsd_is_protein && aatm_is_bb) ) ) {
				// disallowed hbond
				continue;
			}

			Real unweighted_energy( 0.0 ), environment_weight( 0.0 );
			HBEvalType hbe_type;

			get_hbenergy_for_atom_pair( don_rsd, acc_rsd,
				hatm, datm, hatm_xyz, datm_xyz, aatm,
				don_nb, acc_nb, false /*evaluate_derivative*/,
				hbe_type, unweighted_energy, environment_weight,
				use_hb_env_dep, use_hb_env_dep_DNA, smooth_hb_env_dep, deriv );

 			if ( false && unweighted_energy < -0.00001 ) {
 				std::cout << "HBSTATS "
 									<< acc_rsd.name1() << I(3, acc_rsd.seqpos()) << " "
 									<< acc_rsd.atom_name( aatm ) << " "
 									<< don_rsd.name1() << I(3, don_rsd.seqpos() )<< " "
 									<< don_rsd.atom_name( hatm ) << " "
 									<< don_rsd.atom_name( datm ) << " "
 									<< unweighted_energy << " " << environment_weight << "  --  "
									<< aatm_is_bb << " " << datm_is_bb << std::endl;
 			}

			Real const energy_weight( unweighted_energy * environment_weight );


			//Simple rule for RNA and other polymers?
			if ( (!don_rsd_is_protein && !don_rsd.is_DNA()) ||
					 (!acc_rsd_is_protein && !acc_rsd.is_DNA()) )	{
				use_generic_rule_to_fill_hb_energies(
					energy_weight, acc_rsd.seqpos(), don_rsd.seqpos(),
					aatm_is_bb, datm_is_bb,
					scE, sr_bbE, lr_bbE, bb_scE,
					sc1_bb2_scdon, sc2_bb1_scacc );
				continue;
 			}

			// accumulate the energies
			// cant use get_hbond_weight_type( hbe_type ) as above since
			// we are splitting cases more finely
			//
			// BELOW IS TOTALLY PROTEIN-CENTRIC (e.g., assumes any OH-O bond
			//  is a sidechain-sidechain interaction)
			// Why not use a more generic prescription, based on
			//  atom_is_backbone? (see above)
			switch ( hbe_type ) {
			case hbe_BBTURN:
			case hbe_BBHELIX:
				sr_bbE += energy_weight;
				break;
			case hbe_BBOTHER:
				lr_bbE += energy_weight;
				break;
			case hbe_SP2B:
			case hbe_SP3B:
			case hbe_RINGB:
				sc2_bb1_scacc += energy_weight;
				bb_scE += energy_weight;
				break;
			case hbe_BSC:
				sc1_bb2_scdon += energy_weight;
				bb_scE += energy_weight;
				break;
			case hbe_SP2SC:
			case hbe_SP3SC:
			case hbe_RINGSC:
				scE += energy_weight;
				break;
			default:
				std::cout << "Warning: energy from unexpected HB type ignored " <<
					hbe_type << std::endl;
				break;
			} // switch ( hbe_type )
		} // loop over acceptors
	} // loop over donors

}


///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////
// functions for environment-dependent weighting
///////////////////////////////////////////////////////////////////////////////
///////////////////////////////////////////////////////////////////////////////

////////////////////////
// lin jiang's approach:

//JSS note: this is half of the weight from one atom;
// the burial weight is the sum from donor and acceptor.
inline core::Real
burial_weight(int const nb)
{
	if ( nb < 7 ) return 0.1;
	if ( nb > 24 ) return 0.5;
	return (nb-2.75)*(0.5/21.25);
}

core::Real
hb_env_dep_burial_lin(int const nb1, int const nb2)
{
	return (burial_weight(nb1) + burial_weight(nb2));
}

////////////////////////////
// tanja kortemme's approach

void
burial3class_weight_initializer( FArray2D_double & burial )
{
	burial( 1, 1 ) = 0.2 ; burial( 1, 2 ) = 0.2 ; burial( 1, 3 ) = 0.55;
	burial( 2, 1 ) = 0.2 ; burial( 2, 2 ) = 0.55; burial( 2, 3 ) = 1.0 ;
	burial( 3, 1 ) = 0.55; burial( 3, 2 ) = 1.0 ; burial( 3, 3 ) = 1.0 ;
}


inline int
get_burial_3(
	int const neighbors,
	int const threshold_1,
	int const threshold_3
)
{
	//tk get burial measure, three possible classes:
	//tk 1: exposed, 2: intermediate, 3:buried
	if ( neighbors > threshold_1 ) {
		if ( neighbors >= threshold_3 ) return 3;
		else return 2;
	}
	else return 1;
}


core::Real
hb_env_dep_burial_tk(int const nb1, int const nb2)
{
	//tk assign weight based on CB neighbors of two interacting residues

	// local
	int const exposed_threshold = { 10 };
	int const buried_threshold = { 20 };

	static FArray2D_double const burial3class_weight
		( 3, 3, burial3class_weight_initializer );

	return burial3class_weight (get_burial_3( nb1, exposed_threshold,
																						buried_threshold ),
															get_burial_3( nb2, exposed_threshold,
																						buried_threshold ) );
}

///////////////////////////////////////////////////////////////////////////////
Real
get_environment_dependent_weight(
	HBEvalType const & hbe_type,
	int const don_nb,
	int const acc_nb,
	bool const smooth_env_dep_hb /*= false*/
)
{

	Real weight( 1.0 );
	if ( hbe_is_SC_type(hbe_type) ) {
		if ( smooth_env_dep_hb ) {
			weight = hb_env_dep_burial_lin( acc_nb, don_nb );
		} else {
			weight = hb_env_dep_burial_tk( acc_nb, don_nb );
		}
	}
	//	std::cout << "HB_ENV_WEIGHT: " << weight << std::endl;
	return weight;

}


bool
nonzero_hbond_weight( ScoreFunction const & scorefxn )
{
	return ( scorefxn.has_nonzero_weight( hbond_lr_bb ) ||
					 scorefxn.has_nonzero_weight( hbond_sr_bb ) ||
					 scorefxn.has_nonzero_weight( hbond_bb_sc ) ||
					 scorefxn.has_nonzero_weight( hbond_sr_bb_sc ) ||
					 scorefxn.has_nonzero_weight( hbond_lr_bb_sc ) ||
					 scorefxn.has_nonzero_weight( hbond_sc ) );
}


} // hbonds
} // scoring
} // core
