// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author


#include <core/scoring/hbonds/hbonds_geom.hh>

#include <core/scoring/hbonds/types.hh>
#include <core/scoring/hbonds/constants.hh>
#include <core/scoring/hbonds/create_poly.hh>

#include <core/conformation/Residue.hh>
#include <core/util/Tracer.hh>

#include <core/util/Tracer.hh>

#include <core/options/option.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/formatted.o.hh>

// C++ Headers
#include <cmath>
#include <iostream>


// option key includes

#include <core/options/keys/in.OptionKeys.gen.hh>

//Exception for NaN in hbonds.
#include <utility/excn/Exceptions.hh>

//Utility Headers
#include <utility/basic_sys_util.hh>

namespace core {
namespace scoring {
namespace hbonds {

static util::Tracer tr("core.scoring.hbonds");

Real DUMMY_DERIV;
HBond::Deriv DUMMY_DERIV2D;
HBond::Deriv const ZERO_DERIV2D( Vector(0.0,0.0,0.0),  Vector(0.0,0.0,0.0) );

///////////////////////////////////////////////////////////////////////////////
/// @begin classify_BB_by_separation
///
/// @brief
///  Returns the evaluation type of a backbone/backbone hydrogen bond ---
///  which geometric function is applied -- based on sequence separation.
/// @param  dres - [in] -
/// @param  ares - [in] -
/// @remarks
///  Used only for backbone atoms with residues on the same protein sequence.
///
/// @authors Jack Snoeyink Dec 2006
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////
hbonds::HBEvalType
hbe_classify_BB_by_separation(int const don_res, int const acc_res) {
	HBEvalType hbe;

	// JSS switch statements are implemented by table lookup, so are better than
	// if-else for cases like this.  Do not forget the "break;" statements, however!

	switch (don_res-acc_res) {
		case 0: // This probably means that someone called this without specifying residues
					hbe =	hbe_NONE;
					if ( true )
						tr << "Warning: donor/acceptor on same residue " << acc_res
											<< ".  Probably someone forgot to specify ares,dres for hbond_evaluation_type" << std::endl;
					break;
		case -2: // short separation formerly disallowed by allow_bond; now permitted. jss 23 jan 07
		case -1:
		case  1:
		case  2:
		case -3: // hbe = hbe_BBMINUS3; break;
		case  3: hbe = hbe_BBTURN; break;
		case -4: // hbe = hbe_BBMINUS4; break; // Suggestion for later types; put types used into HBEvalType enum
		case  4: hbe = hbe_BBHELIX; break;
		default: hbe = hbe_BBOTHER; break;
	}
	return hbe;
}

///////////////////////////////////////////////////////////////////////////////
HBDonChemType
get_hb_don_chem_type(
	int const datm,
	conformation::Residue const & don_rsd
)
{
	//assert( don_rsd.atom_type( datm ).is_donor() );
	if ( don_rsd.is_protein() && don_rsd.atom_is_backbone(datm) ) {
		return hbdon_BBN;
	} else {
		return hbdon_SC;
	}
}

///////////////////////////////////////////////////////////////////////////////
HBAccChemType
get_hb_acc_chem_type(
	int const aatm,
	conformation::Residue const & acc_rsd
)
{
	using namespace chemical;
	AtomType const & atom_type( acc_rsd.atom_type( aatm ) );
	assert( atom_type.is_acceptor() );

	if ( acc_rsd.is_protein() && acc_rsd.atom_is_backbone( aatm ) ) {
		return hbacc_BB;
	} else {
		Hybridization const & hybrid( atom_type.hybridization() );
		if ( hybrid == SP2_HYBRID ) {
			return hbacc_SP2;
		} else if ( hybrid == SP3_HYBRID ) {
			return hbacc_SP3;
		} else if ( hybrid == RING_HYBRID ) {
			return hbacc_RING;
		} else {
			utility_exit_with_message( "unknown hybridization in hbevaltype: " + acc_rsd.name1() + I(3,acc_rsd.seqpos()) + " " + acc_rsd.atom_name( aatm) );
		}
	}
	return hbacc_NO;
}


////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_evaluation_type
///
/// @brief
///  This function returns the evaluation type of a hydrogen bond ---
///  which determines which geometric hbond scoring function is to be applied.
///
/// @detailed
/// The type of a hydrogen bond determine which polynomials or scoring functions are applied
/// An hbond scoring function uses parameters  r=AHdis, xD=theta, xH=psi (historical names)
/// a burial number, and chi and returns energy and optional derivative information.
///////////////////////////////////////////////////////////////////////////////


hbonds::HBEvalType
hbond_evaluation_type(
	int const datm,
	conformation::Residue const & don_rsd,
	int const aatm,
	conformation::Residue const & acc_rsd,
	bool const ignore_sequence_separation /* = false */
)
{
	HBEvalType hbe = HBeval_lookup( get_hb_acc_chem_type( aatm, acc_rsd ),
																	get_hb_don_chem_type( datm, don_rsd ) );

	if (hbe == hbe_BB) { // refine backbone/backbone bond types by separation
		if (ignore_sequence_separation) return hbe_BBOTHER;
		hbe = hbe_classify_BB_by_separation( don_rsd.seqpos(), acc_rsd.seqpos() );
	}

	return hbe;

}


hbonds::HBEvalType
hbond_evaluation_type_HOHdonor(
	int const aatm,
	conformation::Residue const & acc_rsd
)
{
	HBEvalType hbe = HBeval_lookup( get_hb_acc_chem_type( aatm, acc_rsd ),
																	hbdon_SC /*HOH donor*/);
	return hbe;
}


// Fade intervals for current hydrogen bond parameters
// These specify the ranges where the corresponding polynomial is linearly interpolated to zero
// Eventually, these will be attached to the polynomials.
static FadeInterval const fade_rBB(MIN_R, MIN_R+0.1, R_INTERP_EDGE, MAX_R); // used to adjust xD,xH
static FadeInterval const fade_rshort(MIN_R, MIN_R+0.1, INTERP_MIN, INTERP_MAX);
static FadeInterval const fade_rlong(INTERP_MIN, INTERP_MAX, INTERP_MAX, MAX_R);
static FadeInterval const fade_xD(MIN_xD, ANGLE_INTERP_EDGE, 1, 1); // xD=theta should fade r,xH sooner!
static FadeInterval const fade_xH(MIN_xH, ANGLE_INTERP_EDGE, 1, 1); // fades r,xD
// Since C++ does not yet have unnamed (lambda) functions in the standard,
// I use #define to set up named hbond scoring polynomials that are called
// as name(/*in*/ double x, /*out*/ double &value, double &deriv);
// The idea is to auto-generate the create_poly#() by matlab scripts that
// fit polynomials; if you enter coodinates by hand and have typos,
// the error message may be cryptic becaue of the #defines (but at least
// you'll get them -- the old fortran-style allowed disagreements between
// degree and # of coefficients without complaint.)
//
// Polynomial coefficients are listed from most to least significant:
// e.g., create_poly8(poly_name, xmin, xmax, coefs on x^7..x^0 );
// This is the same order as MATLAB so polynomials can be made by polyfit
// and displayed by x=xmin:0.01:xmax; plot(x,polyval(coeffs, x));
//
// Each score polynomial is clipped to an interval [xmin,xmax], which
// is currently important only to the distance polynomials.
// Outside the interval, the derivative = 0.0 and value = POLY_CLIPVAL;
// If you want the polynomial to always be non-positive, you can find the
// interval by combining roots(coeffs) information with MATLAB plots.
//
// Formerly there was a scale and offset applied at evaluation time;
// those must be built into the polynomial now (WYSIWYG),
// but this is easily done with the MATLAB polynomial fitting scripts.

// define polynomials for use in hbond score calculation
create_poly8(POLY_AHdisBBHelix, MIN_R, 2.8, // 1.78218633, 2.6757, // these are the roots to eliminate positive values.
						 12.93768086,-221.0155722,1604.391304,-6409.335773,15200.86425,-21375.00216,16475.98811,-5361.55644)
create_poly8(POLY_AHdisBBOther, MIN_R, 2.745, // 1.6971523, 2.679339, // non-positive interval
						 13.58980244,-224.0452428,1568.933094,-6044.257847,13820.1498,-18730.96076,13912.92238,-4361.995425)
create_poly5(POLY_AHdisSP2, MIN_R, 2.5, // 1.6941, 2.5, // non-positive interval
						 10.98727738,-100.2401419,340.9733405,-511.6111233,285.0061262)
create_poly5(POLY_AHdisSP3, MIN_R, 2.5, // 1.755, 2.521385, // non-positive interval
						 7.011735538,-68.99968829,251.820931,-403.3593133,238.7378958)
create_poly8(POLY_xDBBHelix, MIN_xD, MAX_xD, // 0.3746, 1.04,
						 223.5268153,-757.7254095,1019.593508,-689.2232431,240.1436064,-37.84119583,0.85868904,0.278181985)
create_poly8(POLY_xDBBOther, MIN_xD, MAX_xD, // 0.76, 1.09,
						 111.9877946,-380.3066184,514.7650204,-352.4092342,124.6219703,-19.94401946,0.149314979,0.635771774)
create_poly3(POLY_xDSP2short, MIN_xD, MAX_xD, // 0.7071, 1.01,
						 -0.562582503,-0.746682668,0.809265171)
create_poly3(POLY_xDSP2long, MIN_xD, MAX_xD, // 0.0, 1.01,
						 0.094962885,-0.254313172,0.0)
create_poly3(POLY_xDSP3short, MIN_xD, MAX_xD, // 0.61566, 1.01,
						 -0.100140144,-1.139139041,0.739279186)
create_poly3(POLY_xDSP3long, MIN_xD, MAX_xD, // 0.0, 1.01,
						 0.089380221,-0.207503776,0.0)
create_poly8(POLY_xHBBHelix, MIN_xH, MAX_xH, // 0.156, 1.03,
						 54.80664331,-196.8196655,295.9418886,-232.105602,96.99124565,-20.60918361,1.573169816,0.000745458)
create_poly8(POLY_xHBBOther, MIN_xH, MAX_xH, // 0.61566, 1.07,
						 43.94483847,-144.3836033,193.5865176,-132.4469355,47.28137288,-8.945888012,-0.227035135,0.791902995)
create_poly3(POLY_xHSP2short, MIN_xH, MAX_xH, // 0.0, 1.08,
						 1.720984644,-1.855254573,0.0)
create_poly3(POLY_xHSP2long, MIN_xH, MAX_xH, // 0.0, 1.01,
						 0.439598249,-0.444673076,0.0)
create_poly3(POLY_xHSP3, MIN_xH, MAX_xH, // 0.0, 1.06,
						 1.761487842,-1.876959406,0.0)
create_poly7(POLY_xHRing, MIN_xH, MAX_xH, // 0.7608, 1.089,e
						 37.744316,-117.731674,143.0759275,-86.2258835,26.7448175,-4.4699705,0.3658455)

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_compute_energy
///
/// @brief
///js   This function takes a distance and cosines of 2 angles, and returns the
///js   energy a of hydrogen bond, and the derivatives of that energy with
///js   respect to the 3 variables.
///
/// @detailed
/// JSS the parameters r=AHdis, xD=theta, xH=psi (historical names)
/// are actually combined as follows,
/// where Pa is a polynomial defined for param a; these can vary by hbond type
/// and Fa is the fading function defined for param a
/// E = Pr*FxD*FxH + Fr*PxD*FxH + Fr*FxD*PxH;
/// dE/dr = dPr/dr *FxD*FxH + dFr/dr *(PxD*FxH+FxD*PxH)
/// dE/dxD & dE/dxH are similar.
/// For sidechains, the PxD,PxH split into short and long, so
/// PxD = PSxD*FSr + PLxD*FLr; PxH = PSxH*FSr + PLxH*FLr;
/// complicating the derivative w.r.t. r
/// dE/dr = dPr/dr *FxD*FxH + dFr/dr *(PxD*FxH+FxD*PxH) +
///              Fr*(FxH*(PSxD*dFSr/dr+PLxD*dFLr/dr)+FxD*(PSxH*dFSr/dr+PLxH*dFLr/dr))
///
/// This "fading" was Jack Schonbrun's, although he had not identified it
/// as such, and had some minor problems with the derivatives in overlapping fade regions.
/// It may be easier to define the polynomials so that the energy can be
/// E = Pr*PxD*PxH
/// dE/dr = dPr/dr *PxD*PxH; etc.
/// Or to define the polynomials so that they live in wells whose boundaries have
/// positive values that outweigh the negative contributions from the other two terms.
/// Positive energy values are treated as "no bond" when this function returns.
/// This is not done yet, so as to reproduce the current rosetta hb_energy functions.
///
/// @param  hbe - [in] - hbond_evaluation_type
/// @param  AHdis - [in] - acceptor proton distance
/// @param  xD - [in] - -cos(180-theta), where theta is defined by Tanja K.
/// @param  xH - [in] - cos(180-phi), where phi is defined by Tanja K.
/// @param  energy - [out] -
/// @param  dE_dr - [out (optional)] -
/// @param  dE_dxD - [out (optional)] -
/// @param  dE_dxH - [out (optional)] -
///
/// @global_read
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
/////////////////////////////////////////////////////////////////////////////////

void
hbond_compute_energy(
	HBEvalType hbe,
	Real const AHdis, // acceptor proton distance
	Real const xD, // -cos(180-theta), where theta is defined by Tanja K.
	Real const xH, // cos(180-phi), where phi is defined by Tanja K.
	Real & energy,
	Real & dE_dr,
	Real & dE_dxD,
	Real & dE_dxH
)
{
	using namespace hbonds;

	energy = MAX_HB_ENERGY + 1.0f;
	dE_dr = dE_dxD = dE_dxH = 0.0;

	if ( std::abs(xD) > 1.0 || std::abs(xH) > 1.0 ) {
		if ( true )
			tr << "WARNING:: invalid angle value in hbond_compute_energy:" <<
				" xH = " << SS( xH ) << " xD = " << SS( xD ) << std::endl;
		return;
	}
	if ( AHdis > MAX_R || AHdis < MIN_R || xH < MIN_xH || xD < MIN_xD ||
			 xH > MAX_xH || xD > MAX_xD ) {
		return;
	}

	double const dAHdis = static_cast<double>(AHdis);
	double const dxD = static_cast<double>(xD);
	double const dxH = static_cast<double>(xH);

	double  FSr,  FLr(0.0),  FxD,  FxH; // values of fading intervals
	double dFSr, dFLr(0.0), dFxD, dFxH; // derivatives of fading intervals
	double  Pr(0.0),  PSxD(0.0),  PSxH(0.0),  PLxD(0.0),  PLxH(0.0); // values of polynomials
	double dPr(0.0), dPSxD(0.0), dPSxH(0.0), dPLxD(0.0), dPLxH(0.0); // derivatives of polynomials

	// fade functions handle the interpolation at edges and for AHdis in [INTERP_MIN,INTERP_MAX]
	fade_xD.value_deriv(xD, FxD, dFxD);
	fade_xH.value_deriv(xH, FxH, dFxH);
	if ( hbe_is_BB_type(hbe) ) { // if we are a BB/BB type
		fade_rBB.value_deriv(AHdis, FSr, dFSr);
		// long range is initialized to zero
	}	else{
		fade_rshort.value_deriv(AHdis, FSr, dFSr);
		fade_rlong.value_deriv(AHdis, FLr, dFLr);
	}

	// here we evaluate the polynomials that pertain to the hbond_evaluation_type hbe
	switch (hbe) {
		case hbe_NONE: break;
		case hbe_BBHELIX:  //hbstat = 1, polys 1,5,11
			POLY_AHdisBBHelix(dAHdis, Pr,dPr);
			POLY_xDBBHelix(dxD, PSxD,dPSxD);
			POLY_xHBBHelix(dxH, PSxH,dPSxH);
			break;

		case hbe_BBOTHER: //hbstat = 2, polys 2,6,11
			POLY_AHdisBBOther(dAHdis, Pr,dPr);
			POLY_xDBBOther(dxD, PSxD,dPSxD);
			POLY_xHBBOther(dxH, PSxH,dPSxH);
			break;

		case hbe_BB:  //hbstat = 3, polys 3,7,13,8,14
		case hbe_BBTURN:
		case hbe_BSC:
		case hbe_SP2B:
		case hbe_SP2SC:
			POLY_AHdisSP2(dAHdis,Pr,dPr);
			POLY_xDSP2short(dxD, PSxD,dPSxD);
			POLY_xHSP2short(dxH, PSxH,dPSxH);
			POLY_xDSP2long(dxD, PLxD,dPLxD);
			POLY_xHSP2long(dxH, PLxH,dPLxH);
			break;

		case hbe_SP3B: //hbstat = 4, polys 4,9,15,10,15
		case hbe_SP3SC:
			POLY_AHdisSP3(dAHdis, Pr,dPr);
			POLY_xDSP3short(dxD, PSxD,dPSxD);
			POLY_xHSP3(dxH, PSxH,dPSxH);
			POLY_xDSP3long(dxD, PLxD,dPLxD);
			PLxH = PSxH; dPLxH = dPSxH;
			break;

		case hbe_RINGB: //hbstat = 5, polys 3,7,16,8,16
		case hbe_RINGSC:
			POLY_AHdisSP2(dAHdis, Pr,dPr);
			POLY_xDSP2short(dxD, PSxD,dPSxD);
			POLY_xHRing(dxH, PSxH,dPSxH);
			POLY_xDSP2long(dxD, PLxD,dPLxD);
			PLxH = PSxH; dPLxH = dPSxH;
			break;
	}
	energy =  Pr*FxD*FxH +  FSr*(PSxD*FxH + FxD*PSxH) +  FLr*(PLxD*FxH + FxD*PLxH);
	if (&dE_dxH == &DUMMY_DERIV) return;  // NOTE: if any deriv parameter omitted, we don't compute derivatives.

	dE_dr =  dPr*FxD*FxH + dFSr*(PSxD*FxH + FxD*PSxH) + dFLr*(PLxD*FxH + FxD*PLxH);
	dE_dxD = dFxD*(Pr*FxH + FLr*PLxH + FSr*PSxH) + FxH*(FSr*dPSxD + FLr*dPLxD);
	dE_dxH = dFxH*(Pr*FxD + FLr*PLxD + FSr*PSxD) + FxD*(FSr*dPSxH + FLr*dPLxH);
}



////////////////////////////////////////////////////////////////////////////////
/// @begin hb_energy_deriv
///
/// @brief
///car Evaluate the energy and derivative components for a hydrogen bond
///
/// @detailed
///car Energy is assumed to be a sum of independent functions of distance,
///car angle at the donor atom, and angle at the acceptor atom. This form
///car of the hydrogen bond potential was selected by Tanja Kortemme
///car The math for computing the derivative of the angular components of
///car the potential was derived by Bill Wedemeyer.
///
///ora used also to calculate derivatives for RB movements if docking_flag T
///ora important NOTE: if in docking mode minimization is done NOT in tr space, this
///ora       should be specified and the docking_specific part should be skipped
///
/// @param  hbe_type - [in] - hydrogen bond evaluation type from hbonds_ns.h
/// @param  donor_res - [in] -
/// @param  acceptor_res - [in] -
/// @param  Dxyz - [in/out] - donor
/// @param  Hxyz - [in/out] - proton
/// @param  Axyz - [in/out] - acceptor
/// @param  Bxyz - [in/out] - acceptor base
/// @param  B2xyz - [in/out] - 2nd acceptor base for ring acceptors
/// @param  energy - [out] -
/// @param  deriv - [out (optional)] - xyz,f1/f2
/// @param  deriv_type [in (optional)] - deriv is NORMAL(default), DOCK_ACC_DON, DOCK_DON_ACC
///
/// @global_read
///   When docking derivatives are computed (DOCK_ACC_DON or DOCK_DON_ACC), center_of_rotation MUST BE DEFINED!
///
/// @global_write
///
/// @remarks
///
/// @references
///
/// @authors
///
/// @last_modified
////////////////////////////////////////////////////////////////////////////////
// Overload to allow non-standard derivative calculations for geometric solvation
void
hb_energy_deriv(
	hbonds::HBEvalType const hbe_type, // hb evalation type
	Vector const & Hxyz, // proton coords
	Vector const & HDunit, // unit vector toward donor
	Vector const & Axyz, // acceptor coords
	Vector const & BAunit, // unit vector towards base
	Real & energy, // returned energy
	bool const evaluate_deriv,
	HBond::Deriv & deriv
) {
	HBDerivType const deriv_type = ( evaluate_deriv ? hbderiv_ABE_GO :  hbderiv_NONE );
	hb_energy_deriv( hbe_type, Hxyz, HDunit, Axyz, BAunit, energy, deriv_type, deriv );
}
///////////////////////////////////////////////////////////////////////////////////////
void
hb_energy_deriv(
	hbonds::HBEvalType const hbe_type, // hb evalation type
	Vector const & Hxyz, // proton coords
	Vector const & HDunit, // unit vector toward donor
	Vector const & Axyz, // acceptor coords
	Vector const & BAunit, // unit vector towards base
	Real & energy, // returned energy
	HBDerivType const deriv_type, // hb derivative type
	HBond::Deriv & deriv
)
{
	using namespace hbonds;

	//  angle definitions:  JSS the angle names are really bad. A-H-D = xD and B-A-H = xH
	//   cos(180-theta) = cos(thetaD) = xD    angle to donor
	//   cos(180-psi) = cos(thetaH) = xH       angle to proton
	//   raw angle in radians for ring nitrogen improper dihedral

	//    energy  - total energy from this hbond
	//    dE_dr   - deriv w/respect to distance
	//    dE_dxD  - deriv w/respect to cos(thetaD)
	//    dE_dxH  - deriv w/respect to cos(thetaH)

	energy = MAX_HB_ENERGY + 1.0f;
	deriv.first = Vector( 0.0 );
	deriv.second = Vector( 0.0 );

	//Objexx: Local arrays declared static for speed
	//car A->H unit vector, distance
	Vector AH;
	//AH(1) = Hxyz(1) - Axyz(1);
	//AH(2) = Hxyz(2) - Axyz(2);
	//AH(3) = Hxyz(3) - Axyz(3);
	AH = Hxyz - Axyz;
	//Real const AHdis2 = AH(1) * AH(1) + AH(2) * AH(2) + AH(3) * AH(3);
	Real const AHdis2( AH.length_squared() );

	//CAR AFTER FUNCTION EVALUATION IS WRITTEN, CHECK THAT THIS IS CORRECT
//	if (AHdis2 > 20.0) return; // jss debug
//	if (runlevel_ns::benchmark) // jss debug
//		tr<<"hbed1 "<<F(10,4,AHdis2)  <<
////     " t:"<< SS(hbe_type) <<
//      "  H" << F(9,2,Hxyz(1))<<","<<F(9,2,Hxyz(2))<<","<<F(9,2,Hxyz(3))<<
//      "  A"<<F(9,2,Axyz(1))<<","<<F(9,2,Axyz(2))<<","<<F(9,2,Axyz(3))<<
//  std::endl;

	if ( AHdis2 > MAX_R2 ) return;
	if ( AHdis2 < MIN_R2 ) return;
	Real const AHdis = std::sqrt(AHdis2);
	Real const inv_AHdis = 1.0f / AHdis;
	Vector AHunit;
	//AHunit(1) = AH(1) * inv_AHdis;
	//AHunit(2) = AH(2) * inv_AHdis;
	//AHunit(3) = AH(3) * inv_AHdis;
	AHunit = AH * inv_AHdis;


	//BW cosines of angle at donor, proton
	Real const xD =            /* cos(180-theta) = cos(thetaD) */
		//AHunit(1) * HDunit(1) + AHunit(2) * HDunit(2) + AHunit(3) * HDunit(3);
		dot( AHunit, HDunit );

	if ( xD < MIN_xD ) return;
	if ( xD > MAX_xD ) return;

	Real const xH =            /* cos(180-psi) = cos(thetaH) */
		//BAunit(1) * AHunit(1) + BAunit(2) * AHunit(2) + BAunit(3) * AHunit(3);
		dot( BAunit, AHunit );

	if ( xH < MIN_xH ) return;
	if ( xH > MAX_xH ) return;

	if ( deriv_type == hbderiv_NONE ) {
		// NOTE: early return with energy if no derivatives
		hbond_compute_energy( hbe_type, AHdis, xD, xH, energy );
		return;
	}

	//JSS the rest happens only if we want deriviative information
	Real dE_dxH, dE_dxD, dE_dr;
	hbond_compute_energy(hbe_type, AHdis,xD,xH,energy,dE_dr,dE_dxD,dE_dxH);


	if (energy >= MAX_HB_ENERGY) return;

	// deriv is a pair of vectors:
	Vector & f1( deriv.first );
	Vector & f2( deriv.second );

	//car distance-dependent gradient component.
	//car see also comments in minimize.cc
	//car  dr/dphi = Eab x (V-Vb) . (V' - V)/|V-V'|
	//db  (first cross product is the displacement of V upon a rotation dphi
	//db   around the unit vector Eab, Vb is the coordinates of the second atom
	//db   in the bond)

	//car  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / |V-V'|
	//car  dEr/dphi = dEr/dr * (Eab  x (V-Vb) . (V' - V)] / r
	//car  rearrange...
	//car  dEr/dphi = dEr/dr * [Eab X Vb . (V' - V) + Eab . (V' x V)] / r

	//car Eab and Eab X Vb are calulated in dfunc_vdw and dependent on the torison
	//car angle with respect to which the derivative is being taken
	//car f1 and f2 are independent of the torsion angle and here we're precomputing
	//car increments to f1 and f2 for each hbond
	//car f1 = dEr/dr * (V'xV) / r
	//car f2 = dEr/dr * (V'-V) / r
	//car here, V' = H
	//car       V  = A
	//car       dE/dr * 1/r = prefactor

	//car Eab and Eab X Vb are calulated in dfunc_vdw, here

	//car V'x V
	Vector HxA;
	//HxA(1) = Hxyz(2) * Axyz(3) - Hxyz(3) * Axyz(2);
	//HxA(2) = Hxyz(3) * Axyz(1) - Hxyz(1) * Axyz(3);
	//HxA(3) = Hxyz(1) * Axyz(2) - Hxyz(2) * Axyz(1);
	HxA = cross( Hxyz, Axyz );

	// in/decrements to the f1 and f2 of the angle moving donor/acceptor
	Real prefactor = inv_AHdis * dE_dr;
	//deriv[0] = prefactor * HxA(1); // deriv(1,1) // f1
	//deriv[1] = prefactor * HxA(2); // deriv(2,1)
	//deriv[2] = prefactor * HxA(3); // deriv(3,1)
	f1 = prefactor * HxA;

	//deriv[3] = prefactor * AH(1);  // deriv(1,2) // f2
	//deriv[4] = prefactor * AH(2);  // deriv(2,2)
	//deriv[5] = prefactor * AH(3);  // deriv(3,2)
	f2 = prefactor * AH;

	//car gradient component for xD (theta)
	//car (see comments below for xH gradient)
	if (deriv_type != hbderiv_ABE_GO_NO_xD ){

		Vector BD;
		prefactor = inv_AHdis * dE_dxD;
		//BD(1) = prefactor * ( HDunit(1) - xD * AHunit(1) );
		//BD(2) = prefactor * ( HDunit(2) - xD * AHunit(2) );
		//BD(3) = prefactor * ( HDunit(3) - xD * AHunit(3) );
		BD = prefactor * ( HDunit - xD * AHunit );

		Vector BDxA;
		//BDxA(1) = BD(2) * Axyz(3) - BD(3) * Axyz(2);
		//BDxA(2) = BD(3) * Axyz(1) - BD(1) * Axyz(3);
		//BDxA(3) = BD(1) * Axyz(2) - BD(2) * Axyz(1);
		BDxA = cross( BD, Axyz );


		//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor
		//deriv[0] += BDxA(1); // deriv(1,1) // f1,theta
		//deriv[1] += BDxA(2); // deriv(2,1)
		//deriv[2] += BDxA(3); // deriv(3,1)
		f1 += BDxA;

		//deriv[3] += BD(1);   // deriv(1,2) // f2,theta
		//deriv[4] += BD(2);   // deriv(2,2)
		//deriv[5] += BD(3);   // deriv(3,2)
		f2 += BD;

	}

	//BW gradient component for xH (psi)
	//car   (from code and paper by Bill Wedemeyer)
	//car  xH = (BAunit . AH)/AHdis
	//car  dxH/dphi = 1/AHdis * (BAunit . dAH/dphi - xH *dAHdis/dphi)
	//car    (note dBAunit/dphi = 0)
	//car
	//car  dAHdis/dphi = AHunit . dAH/dphi
	//car
	//car  substituting and rearranging....
	//car  dxH/dphi = 1/AHdis * dAH/dphi . (BAunit - xH*AHunit)
	//car           = 1/AHdis * dAH/dphi . BH
	//car
	//car note: BH = (BAunit - xH*AHunit) = component of BAunit that is
	//car       perpendicular to AHunit
	//car
	//car dAH/dphi = Eab x (V-Vb)  . (V' - V)/|V-V'|   from above
	//car dAH/dphi = Eab x (H-Vb)  . AHunit
	//car
	//car dExH/dphi = dExH/dxH * dxH/dphi
	//car           = dExH/dxH * 1/AHdis * dAH/dphi . BH
	//car           = dExH/dxH * 1/AHdis * (Eab x (H-Vb) .(H-A)/AHdis) . BH
	//car
	//car rearrange as with dEr/dr above to get f1 and f2 component

	//car f1 = dE/dxH *(1/AHdis) * BH x H
	//car f2 = dE/dxH *(1/AHdis) * BH

	if ( deriv_type != hbderiv_ABE_GO_NO_xH ){

		Vector BH;
		prefactor = inv_AHdis * dE_dxH;
		//BH(1) = prefactor * ( BAunit(1) - xH * AHunit(1) );
		//BH(2) = prefactor * ( BAunit(2) - xH * AHunit(2) );
		//BH(3) = prefactor * ( BAunit(3) - xH * AHunit(3) );
		BH = prefactor * ( BAunit - xH * AHunit );

		Vector BHxH;
		//BHxH(1) = BH(2) * Hxyz(3) - BH(3) * Hxyz(2);
		//BHxH(2) = BH(3) * Hxyz(1) - BH(1) * Hxyz(3);
		//BHxH(3) = BH(1) * Hxyz(2) - BH(2) * Hxyz(1);
		BHxH = cross( BH, Hxyz );

		//BW in/decrements to the f1 and f2 of the angle moving donor/acceptor
		//deriv[0] += BHxH(1); // deriv(1,1)
		//deriv[1] += BHxH(2); // deriv(2,1)
		//deriv[2] += BHxH(3); // deriv(3,1)
		f1 += BHxH;

		//deriv[3] += BH(1);   // deriv(1,2)
		//deriv[4] += BH(2);   // deriv(2,2)
		//deriv[5] += BH(3);   // deriv(3,2)
		f2 += BH;
	}

}


////////////////////////////////////////////////////////////////////////////////
/// @begin hb_energy_deriv
///
/// @remarks See comments on helper function above.
///
////////////////////////////////////////////////////////////////////////////////
// Overload to allow non-standard derivative calculations for geometric solvation
void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type -- determines what scoring function to use
	Vector const & Dxyz, // donor coords
	Vector const & Hxyz, // proton
	Vector const & Axyz, // acceptor
	Vector const & Bxyz, // acceptor base
	Vector const & B2xyz, // 2nd acceptor base for ring & SP3 acceptors
	Real & energy,
	bool const evaluate_deriv, // hb derivative type
	std::pair< Vector, Vector > & deriv
) {
	HBDerivType const deriv_type = ( evaluate_deriv ? hbderiv_ABE_GO :  hbderiv_NONE );
	hb_energy_deriv( hbe_type, Dxyz, Hxyz, Axyz, Bxyz, B2xyz, energy, deriv_type, deriv );
}

// You must supply 1.0 and 0.0 as arguments a and b.
// and no you can't short cut this, because the compiler will optimize it away!
bool is_finitenumber( double s, double  a, double b ){
	if ((a*s) != (s*cos(b)) )       return false; //  NAN!
	if ( s * 100.0 == s * 1000.00 ) return false; //  INF!
	return true;
}


void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type -- determines what scoring function to use
	Vector const & Dxyz, // donor coords
	Vector const & Hxyz, // proton
	Vector const & Axyz, // acceptor
	Vector const & Bxyz, // acceptor base
	Vector const & B2xyz, // 2nd acceptor base for ring & SP3 acceptors
	Real & energy,
	HBDerivType const deriv_type, // hb derivative type
	std::pair< Vector, Vector > & deriv
)
{
	using namespace hbonds;

//  angle definitions:  JSS the angle names are really bad. A-H-D = xD and B-A-H = xH
//   cos(180-theta) = cos(thetaD) = xD    angle to donor
//   cos(180-psi) = cos(thetaH) = xH       angle to proton
//   raw angle in radians for ring nitrogen improper dihedral

//    energy  - total energy from this hbond
//    dE_dr   - deriv w/respect to distance
//    dE_dxD  - deriv w/respect to cos(thetaD)
//    dE_dxH  - deriv w/respect to cos(thetaH)

//Objexx: Local arrays declared static for speed
//JSS all early exits are in helper above, so this version of the function is deprecated.
//These unit vectors are invariant for hbonded pairs and can be precalculated.
//car  H->D unit vector, dis2
	Vector HDunit;
	//HDunit(1) = Dxyz(1) - Hxyz(1);
	//HDunit(2) = Dxyz(2) - Hxyz(2);
	//HDunit(3) = Dxyz(3) - Hxyz(3);
	HDunit = Dxyz - Hxyz;
	//Real const HDdis2 =
	//  HDunit(1) * HDunit(1) + HDunit(2) * HDunit(2) + HDunit(3) * HDunit(3);
	Real const HDdis2( HDunit.length_squared() );

	// NaN check
	if ( ! is_finitenumber( HDdis2, 1.0, 0.0 ) ) {
		std::string const warning( "NANs occured in hbonding!" );
		static bool warn_on_std_err = true;
		if( warn_on_std_err ){
			std::cerr << "Hbond tripped: " << utility::timestamp() << std::endl; std::cerr.flush();
			warn_on_std_err = false;
		}
		bool fail_on_bad_hbond = options::option[ options::OptionKeys::in::file::fail_on_bad_hbond ]();

		if ( fail_on_bad_hbond ) {
			throw( utility::excn::EXCN_Msg_Exception( warning ) );
			utility_exit_with_message( warning );
		} else {
			tr.Error << warning << std::endl;
		}
	}

	if ( HDdis2 < 0.64 || HDdis2 > 1.5625 ) { // .8 to 1.25A
		if ( true ) {
			// this warning was runlevel dependent
			if ( tr.visible() )
			tr.Debug << "Warning: hb_energy_deriv has H(" << Hxyz(1) << "," <<
				Hxyz(2)<< "," << Hxyz(3) << ") D(" << Dxyz(1) << "," << Dxyz(2) <<
				"," << Dxyz(3) << ")  distance out of range " << std::sqrt( HDdis2 ) << std::endl;
		}
		energy = 0.0;
		deriv.first = Vector( 0.0 );
		deriv.second = Vector( 0.0 );
		return;
	}

	Real const inv_HDdis = 1.0f / std::sqrt( HDdis2 );
	//HDunit(1) *= inv_HDdis;
	//HDunit(2) *= inv_HDdis;
	//HDunit(3) *= inv_HDdis;
	HDunit *= inv_HDdis;

//car  B->A unit vector
	Vector BAunit;
	make_hbBasetoAcc_unitvector(hbe_type, Axyz, Bxyz, B2xyz, BAunit);
	hb_energy_deriv( hbe_type, Hxyz, HDunit, Axyz, BAunit, energy,
									 deriv_type, deriv );
}


Vector
create_acc_orientation_vector(
	conformation::Residue const & residue,
	int atom_id
)
{
	assert( residue.atom_type_set()[ residue.atom(atom_id).type() ].is_acceptor() );

	HBEvalType faux_evaltype( hbe_BB );
	HBAccChemType chemtype = get_hb_acc_chem_type( atom_id, residue );
	switch( chemtype ) {
		case hbacc_NO :
			utility_exit_with_message( "Error in hbonds_geom.cc: create_acc_orientation_vector with hbacc_NO" );
			break;
		case hbacc_BB :   faux_evaltype = hbe_BB; break;
		case hbacc_SP2 :  faux_evaltype = hbe_SP2B; break;
		case hbacc_SP3 :  faux_evaltype = hbe_SP3B; break;
		case hbacc_RING : faux_evaltype = hbe_RINGB; break;
	}
	Vector ovect;
	make_hbBasetoAcc_unitvector( faux_evaltype,
		residue.atom( atom_id ).xyz(),
		residue.atom( residue.atom_base( atom_id ) ).xyz(),
		residue.atom( residue.abase2( atom_id ) ).xyz(),
		ovect );
	return ovect;
}

/// @brief create a unit vector pointing from the hydrogen toward the donor
Vector
create_don_orientation_vector(
	conformation::Residue const & residue,
	int atom_id
)
{
	assert( residue.atom_type_set()[ residue.atom( residue.atom_base(atom_id)).type() ].is_donor() );

	Vector HDunit;
	HDunit = residue.atom( residue.atom_base( atom_id ) ).xyz() - residue.atom( atom_id ).xyz();
	Real const HDdis2( HDunit.length_squared() );
	Real const inv_HDdis = 1.0f / std::sqrt( HDdis2 );
	HDunit *= inv_HDdis;
	return HDunit;
}

///////////////////////////////////////////////////////////////////////////////
//
// construct for a hydrogen bond Acceptor, the unit vector from Base
// the BA unit vector depends on the hb evaluation type
//
void
make_hbBasetoAcc_unitvector(
	HBEvalType const hbe_type,
	Vector const & Axyz,
	Vector const & Bxyz,
	Vector const & B2xyz,
	Vector & BAunit
)
{
	switch ( hbe_type ) {
		case hbe_RINGSC:
		case hbe_RINGB: // RING uses midpoint between base and base2
			//BAunit(1) = Axyz(1) - 0.5f * ( Bxyz(1) + B2xyz(1) );
			//BAunit(2) = Axyz(2) - 0.5f * ( Bxyz(2) + B2xyz(2) );
			//BAunit(3) = Axyz(3) - 0.5f * ( Bxyz(3) + B2xyz(3) );
			BAunit = Axyz - Real(0.5) * ( Bxyz + B2xyz );
			break;
		case hbe_SP3SC:
		case hbe_SP3B: // SP3 uses base2
			//BAunit(1) = Axyz(1) - B2xyz(1);
			//BAunit(2) = Axyz(2) - B2xyz(2);
			//BAunit(3) = Axyz(3) - B2xyz(3);
			BAunit = Axyz - B2xyz;
			break;
		default: // SP2 uses base
			//BAunit(1) = Axyz(1) - Bxyz(1);
			//BAunit(2) = Axyz(2) - Bxyz(2);
			//BAunit(3) = Axyz(3) - Bxyz(3);
			BAunit = Axyz - Bxyz;
	}
	//Real const BAdis = BAunit(1) * BAunit(1) + BAunit(2) * BAunit(2) + BAunit(3) * BAunit(3);
	//assert(BAdis > 1e-3); //JSS base and base2 shouldn't be the same atoms
	//if (BAdis < 1e-2) //JSS base and base2 shouldn't be the same atoms
	//	if (runlevel_ns::benchmark || runlevel_ns::runlevel > runlevel_ns::standard)
	//		tr << "WARNING: hb_energy_deriv() has base and acceptor atoms at distance "<<BAdis<<std::endl;
	//Real const inv_BAdis = 1.0f / std::sqrt(BAdis);
	//BAunit(1) *= inv_BAdis;
	//BAunit(2) *= inv_BAdis;
	//BAunit(3) *= inv_BAdis;
	BAunit.normalize();
}

////////////////////////////////////////////////////////////////////////////////
/// @begin hbond_polynomial
///
/// @brief evaluate a polynomial and its derivative.
///
/// @detailed
///
/// @param  variable - [in] - evaluate polynomial(value)
/// @param  table - [in] - table of parameters -- JSS backwards: low to high
/// @param  value - [out] - returned output
/// @param  deriv - [out] - returned output
///
/// @global_read
///
/// @global_write
///
/// @remarks
/// No longer used as of changeset 11880 or so; I leave it in as an example of
/// how to use horner's rule to compute a polynomial and its derivative together.
/// This is the template for the create_poly# macros in hbonds_ns.h
///
/// @references
///
/// @authors
///
/// @last_modified Jack Snoeyink
/////////////////////////////////////////////////////////////////////////////////
//void
//hbond_polynomial(
//								 double const variable,
//								 int const table, // table of parameters
//								 double & value, // returned output
//								 double & deriv // returned output
//								 )
//{
//  using namespace hbonds;
//
//  double const internal_weight = { .5 };
//
//  // Polynomial value and derivative using Horner's rule
//  // value = Sum_(i = 1,...,N) [ coeff_i * variable^(i-1) ]
//  // deriv = Sum_(i = 2,...,N) [ ( i - 1 ) * coeff_i * variable^(i-2) ]
//  // JSS: Horner's rule for evaluating polynomials is based on rewriting the polynomial as:
//  // JSS: p(x)  = a0 + x*(a1 + x*(a2 + x*(...  x*(aN)...)))
//  // JSS: or value_k = a_k + x*value_k+1 for k = N-1 to 0
//  // JSS: and the derivative is
//  // JSS: deriv_k = value_k+1 + deriv_k+1 for k = N-1 to 1
//  int N = poly_degree(table); // offset for arrays
//  assert( N > 0 );
//  int l = coeff.index( N, table );
//  value = coeff[ l ];
//  deriv = 0.0;
//  while ( --N > 0 ) {
//    ( deriv *= variable ) += value;
//    ( value *= variable ) += coeff[ --l ];
//  }
//
//  value -= offset(table);  // JSS: normalization should be applied to the original polynomial.
//  double const normalize = internal_weight / scale(table);
//  value *= normalize;
//  deriv *= normalize;
//}

void
show_poly()
{
	using numeric::conversions::radians;

	// distance
	for ( Size i=120; i<= 300; i+=2  ){
		double const AHdis( 0.01 * i );
		double energy,deriv;
		tr << "DISTANCE_POLY " << F(9,3,AHdis);
		POLY_AHdisBBHelix( AHdis, energy, deriv ); tr << F( 9, 3, energy );
		POLY_AHdisBBOther( AHdis, energy, deriv ); tr << F( 9, 3, energy );
		POLY_AHdisSP2    ( AHdis, energy, deriv ); tr << F( 9, 3, energy );
		POLY_AHdisSP3    ( AHdis, energy, deriv ); tr << F( 9, 3, energy ) << std::endl;
	}

	// angle at acceptor
	for ( Size i=60; i<= 180; i+=2 ) {
		double const xH( std::cos( radians( 180.0 - i ) ) );
		double energy,deriv;
		tr << "ACC_ANGLE_POLY " << F( 9, 3, double(i) ) << F(9,3,xH);
		POLY_xHBBHelix ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHBBOther ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHSP2short( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHSP2long ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHSP3     ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHSP3     ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHRing    ( xH, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xHRing    ( xH, energy, deriv ); tr << F( 9, 3, energy ) << std::endl;
	}

	// angle at hydrogen
	// angle at acceptor
	for ( Size i=60; i<= 180; i+=2 ) {
		double const xD( std::cos( radians( 180.0 - i ) ) );
		double energy,deriv;
		tr << "HYD_ANGLE_POLY " << F( 9, 3, double(i) ) << F(9,3,xD);
		POLY_xDBBHelix ( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDBBOther ( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP2short( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP2long ( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP3short( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP3long ( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP2short( xD, energy, deriv ); tr << F( 9, 3, energy );
		POLY_xDSP2long ( xD, energy, deriv ); tr << F( 9, 3, energy ) << std::endl;
	}




}


} // hbonds
} // scoring
} // core
