// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

#ifndef INCLUDED_core_scoring_hbonds_hbonds_geom_HH
#define INCLUDED_core_scoring_hbonds_hbonds_geom_HH

#include <core/scoring/hbonds/types.hh>

#include <core/scoring/hbonds/HBondSet.hh>

#include <core/conformation/Residue.fwd.hh>



namespace core {
namespace scoring {
namespace hbonds {

//using namespace hbonds;

// class declarations

/////////////////////////////
/// class FadeInterval
/// stores an "fading interval" [a b c d] with a <= b <= c <= d
/// and computes the fraction of containment for x, which is defined to be
/// 0 if x is outside of (a,d), 1 if x is inside of [b,c],
/// and a continuous linear ramp otherwise.
/// i.e. (x-a)/(b-a) for x in (a,b), and (d-x)/(d-c) for x in (c,d)
/// This is used to ensure that hbond scoring as a sum Er + ExH + ExD goes to zero at the edges.
////////////////////////////
class FadeInterval {
private:
	Real const min0_;
	Real const fmin_;
	Real const fmax_;
	Real const max0_;
	double const dfade_min_;
	double const dfade_max_;

public:
	/// Constructor

	FadeInterval(
		Real const min0,
		Real const fmin,
		Real const fmax,
		Real const max0) :
		min0_(min0),
		fmin_(fmin),
		fmax_(fmax),
		max0_(max0),
		dfade_min_(1.0/static_cast<double>(fmin-min0)),
		dfade_max_(1.0/static_cast<double>(max0-fmax))
	{
		assert(min0 <= fmin && fmin <= fmax && fmax <= max0); // make sure dfades are not negative
		// if they are 1.0/0.0, they won't be used...
	};

	double value(Real const x) const {  //JSS  5 intervals --a-b---c-d--
		if (x < fmax_) {
			if (x >= fmin_) return 1.0;      // in [fmin, fmax]
			if (x <= min0_) return 0.0;      // <= min0
			return (x - min0_) * dfade_min_; // in (min0,fmin)
		} else {
			if (x >= max0_) return 0.0;      // >= max0
			return (max0_ - x) * dfade_max_; // in (fmax,max0)
		}
	}

	void value_deriv(Real const x, double &val, double &deriv) const {  //JSS  5 intervals --a-b---c-d--
		if (x < fmax_) {
			if (x >= fmin_) { val = 1.0; deriv = 0.0; return; }  // in [fmin, fmax]
			if (x <= min0_) { val = deriv = 0.0; return; }       // <= min0
			deriv = dfade_min_; val = (x - min0_) * dfade_min_;  // in (min0,fmin)
		} else {
			if (x >= max0_) { val = deriv = 0.0; return; }       // >= max0
			deriv = -dfade_max_; val = (max0_ - x) * dfade_max_; // in (fmax,max0)
		}
	}
};



// PHIL deleted several helper functions for determining
// hbevaltype of water hbonds. add back in later

hbonds::HBEvalType
hbe_classify_BB_by_separation(
	int const don_res,
	int const acc_res
);


HBAccChemType
get_hb_acc_chem_type(
	int const aatm,
	conformation::Residue const & acc_rsd
);

HBDonChemType
get_hb_don_chem_type(
	int const datm,
	conformation::Residue const & don_rsd
);

HBEvalType
hbond_evaluation_type(
	int const datm,
	conformation::Residue const & don_rsd,
	int const aatm,
	conformation::Residue const & acc_rsd,
	bool const ignore_sequence_separation = false
);

hbonds::HBEvalType
hbond_evaluation_type_HOHdonor(
	int const aatm,
	conformation::Residue const & acc_rsd
);

// hack?
extern Real DUMMY_DERIV;

void
hbond_compute_energy(
	HBEvalType hbe, // used internally & by geometric solvation
	Real const AHdis, // acceptor proton distance
	Real const xD, // -cos(180-theta), where theta is defined by Tanja K.
	Real const xH, // cos(180-phi), where phi is defined by Tanja K.
	Real & energy,
	Real & dE_dr = DUMMY_DERIV,
	Real & dE_dxD = DUMMY_DERIV,
	Real & dE_dxH = DUMMY_DERIV
);


// hbond evaluation type -- determines what scoring function to use
extern HBond::Deriv DUMMY_DERIV2D;
extern HBond::Deriv const ZERO_DERIV2D;

void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type
	Vector const & Hxyz, // proton
	Vector const & HDunit, // donor
	Vector const & Axyz, // acceptor
	Vector const & BAunit, // unit vector towards base
	Real & energy,
	bool const calculate_derivative = false,
	HBond::Deriv & deriv = DUMMY_DERIV2D
);

void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type
	Vector const & Hxyz, // proton
	Vector const & HDunit, // donor
	Vector const & Axyz, // acceptor
	Vector const & BAunit, // unit vector towards base
	Real & energy,
	HBDerivType const deriv_type,
	HBond::Deriv & deriv = DUMMY_DERIV2D
);


// hbond evaluation type -- determines what scoring function to use
void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type
	Vector const & Dxyz, // donor coords
	Vector const & Hxyz, // proton
	Vector const & Axyz, // acceptor
	Vector const & Bxyz, // acceptor base
	Vector const & B2xyz, // 2nd acceptor base for ring & SP3 acceptors
	Real & energy,
	bool const calculate_derivative = false,
	std::pair< Vector, Vector > & deriv = DUMMY_DERIV2D // xyz,f1/f2
);

void
hb_energy_deriv(
	HBEvalType const hbe_type, // hbond evaluation type
	Vector const & Dxyz, // donor coords
	Vector const & Hxyz, // proton
	Vector const & Axyz, // acceptor
	Vector const & Bxyz, // acceptor base
	Vector const & B2xyz, // 2nd acceptor base for ring & SP3 acceptors
	Real & energy,
	HBDerivType const deriv_type,
	std::pair< Vector, Vector > & deriv = DUMMY_DERIV2D // xyz,f1/f2
);

Vector
create_acc_orientation_vector(
	conformation::Residue const & residue,
	int atom_id
);

Vector
create_don_orientation_vector(
	conformation::Residue const & residue,
	int atom_id
);

// construct for a hydrogen bond Acceptor, the unit vector from Base
// the BA unit vector depends on the hb evaluation type -- though it is only dependent
// on the acceptor hybridization and is independent of the hydrogen atom's properties.
void
make_hbBasetoAcc_unitvector(
	HBEvalType const hbe_type,
	Vector const & Axyz,
	Vector const & Bxyz,
	Vector const & B2xyz,
	Vector & BAunit
);


void
show_poly();

}
}
}

#endif
