// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/types.hh
/// @brief  core::scoring package type declarations
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)


#ifndef INCLUDED_core_scoring_hbonds_types_HH
#define INCLUDED_core_scoring_hbonds_types_HH


// Project headers
#include <core/scoring/types.hh>


// ObjexxFCL headers
//#include <ObjexxFCL/CArray.fwd.hh>
//#include <ObjexxFCL/CPArray.fwd.hh>
//#include <ObjexxFCL/FArray.fwd.hh>


namespace core {
namespace scoring {
namespace hbonds {

/// this is not currently used, copied it from rosetta++
/// where it is also not used
///
enum HBondWeightType {
	hbw_SR_BB = 1,
	hbw_LR_BB,
	hbw_BB_SC,
	hbw_SC,
	hbw_H2O,
	hbw_NONE
};


//JSS Chemtypes determine which hbond scoring function applies to AHdist, theta, psi, chi, burial
//
// function for getting hbevaltype is in hbonds_geom.cc
//
// note that if you change this enum you should change all the case statements
// in hbonds.cc and elsewhere that depend on HBEvalType, eg the one below
//
enum HBEvalType {
	hbe_NONE, // no bond
	// BB/BB -- order must be preserved
	hbe_BB, // generic BB/BB (only refined are used)
	hbe_BBTURN, // BB/BB short range
	hbe_BBHELIX, // BB/BB short range
	hbe_BBOTHER, // keep BB/BB consecutive for hbond_compute_energy
	// BB donor, SC acceptor
	hbe_SP2B,
	hbe_SP3B,
	hbe_RINGB,
	// BB acceptor, SC donor
	hbe_BSC,
	//SC/SC
	hbe_SP2SC,
	hbe_SP3SC,
	hbe_RINGSC,
	// hbe_MAX should always be # of last
	hbe_MAX=hbe_RINGSC
};

enum HBDerivType {
	hbderiv_NONE, // no derivative
	hbderiv_ABE_GO, // standard hbond derivative calculation
	hbderiv_ABE_GO_NO_xD, // needed for geometric solvation craziness
	hbderiv_ABE_GO_NO_xH  // needed for geometric solvation craziness
};



inline
HBondWeightType
get_hbond_weight_type( HBEvalType const & hbe_type )
{
	switch ( hbe_type ) {
	case hbe_BBTURN:
	case hbe_BBHELIX:
		return hbw_SR_BB;
	case hbe_BBOTHER:
		return hbw_LR_BB;
	case hbe_SP2B:
	case hbe_SP3B:
	case hbe_RINGB:
	case hbe_BSC:
		return hbw_BB_SC;
	case hbe_SP2SC:
	case hbe_SP3SC:
	case hbe_RINGSC:
		return hbw_SC;
	default:
		return hbw_NONE;
	}
	return hbw_NONE;
}

inline
bool
hbe_is_BB_type( HBEvalType hbe )
{
	return ( hbe <= hbe_BBOTHER ) && ( hbe >= hbe_BB );
}

inline
bool
hbe_is_SC_type( HBEvalType hbe )
{
	return (hbe > hbe_BBOTHER);
}

// converting from atomno,rsd to HBDonChemType and HBAccChemType is
// handled by functions in hbonds_geom.hh

enum HBDonChemType	{
	hbdon_NO = 1,
	hbdon_BBN,
	hbdon_SC,
	hbdon_MAX=hbdon_SC
}; // existing donor classes


/** recommended donor classes:
enum HBDonChemType {
	hbdon_NO = 1,
	hbdon_BBN,
	hbdon_SCN,
	hbdon_SCOH,
	hbdon_HOH,
	hbdon_MAX=hbdon_HOH
};
**/

enum HBAccChemType {
	hbacc_NO = 1,
	hbacc_BB,
	hbacc_SP2,
	hbacc_SP3,
	hbacc_RING,
	hbacc_MAX=hbacc_RING
}; // existing acceptor classes


} // namespace hbonds
} // namespace scoring
} // namespace core


#endif // INCLUDED_core_scoring_types_HH
