// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoreFunction.hh
/// @brief  Score function class
/// @author Phil Bradley

/// NOTE-- this file includes both string and map, use .fwd.hh if
/// you can!

// Unit headers
#include <core/scoring/methods/EnergyMethodOptions.hh>

#include <core/mm/MMBondAngleResidueTypeParamSet.hh>

// C++ Headers

namespace core {
namespace scoring {
namespace methods {

EnergyMethodOptions::EnergyMethodOptions():
	etable_type_( FA_STANDARD_DEFAULT ), // hard-wired default, but you can set this with etable_type( string ),
	atom_vdw_atom_type_set_name_( chemical::CENTROID ), // can be set, see below
	exclude_protein_protein_hack_elec_( false ), // rosetta++ defaulted to true!
	exclude_DNA_DNA_   ( true ), // rosetta++ default
	use_hb_env_dep_    ( true ), // rosetta++ default
	use_hb_env_dep_DNA_( true ),
	smooth_hb_env_dep_ ( true ), // rosetta++ default
	decompose_bb_hb_into_pair_energies_( false ),
	bond_angle_residue_type_param_set_( NULL )
{}

/// copy constructor
EnergyMethodOptions::EnergyMethodOptions( EnergyMethodOptions const & src ):
	ReferenceCount( src )
{
	*this = src;
}

EnergyMethodOptions::~EnergyMethodOptions()
{ }

/// copy operator
EnergyMethodOptions const &
EnergyMethodOptions::operator=( EnergyMethodOptions const & src )
{
	etable_type_ = src.etable_type_;
	atom_vdw_atom_type_set_name_ = src.atom_vdw_atom_type_set_name_;
	method_weights_ = src.method_weights_;
	ss_weights_ = src.ss_weights_;
	exclude_protein_protein_hack_elec_ = src.exclude_protein_protein_hack_elec_;
	exclude_DNA_DNA_    = src.exclude_DNA_DNA_;
	use_hb_env_dep_     = src.use_hb_env_dep_;
	use_hb_env_dep_DNA_ = src.use_hb_env_dep_DNA_;
	smooth_hb_env_dep_  = src.smooth_hb_env_dep_;
	decompose_bb_hb_into_pair_energies_ = src.decompose_bb_hb_into_pair_energies_;
	bond_angle_central_atoms_to_score_ = src.bond_angle_central_atoms_to_score_;
	bond_angle_residue_type_param_set_ = src.bond_angle_residue_type_param_set_;
	return *this;
}

/// deprecated
utility::vector1<std::string> const &
EnergyMethodOptions::bond_angle_central_atoms_to_score() const
{
	if (bond_angle_residue_type_param_set_) {
		return bond_angle_residue_type_param_set_->central_atoms_to_score();
	}
	return bond_angle_central_atoms_to_score_;
}

/// deprecated
void
EnergyMethodOptions::bond_angle_central_atoms_to_score(
	utility::vector1<std::string> const & atom_names
)
{
	bond_angle_central_atoms_to_score_ = atom_names;
	if (bond_angle_residue_type_param_set_) {
		bond_angle_residue_type_param_set_->central_atoms_to_score(atom_names);
	}
}

core::mm::MMBondAngleResidueTypeParamSetOP
EnergyMethodOptions::bond_angle_residue_type_param_set()
{
	return bond_angle_residue_type_param_set_;
}

core::mm::MMBondAngleResidueTypeParamSetCOP
EnergyMethodOptions::bond_angle_residue_type_param_set() const
{
	return bond_angle_residue_type_param_set_;
}

void
EnergyMethodOptions::bond_angle_residue_type_param_set(
	core::mm::MMBondAngleResidueTypeParamSetOP param_set
)
{
	bond_angle_residue_type_param_set_ = param_set;
}

/// used inside ScoreFunctionInfo::operator==
bool
operator==( EnergyMethodOptions const & a, EnergyMethodOptions const & b )
{
	return ( ( a.etable_type_ == b.etable_type_ ) &&
					 ( a.atom_vdw_atom_type_set_name_ == b.atom_vdw_atom_type_set_name_ ) &&
					 ( a.exclude_DNA_DNA_ == b.exclude_DNA_DNA_ ) &&
					 ( a.method_weights_ == b.method_weights_ ) &&
					 ( a.ss_weights_ == b.ss_weights_ ) &&
					 ( a.exclude_protein_protein_hack_elec_ == b.exclude_protein_protein_hack_elec_ ) &&
					 ( a.exclude_DNA_DNA_ == b.exclude_DNA_DNA_ ) &&
					 ( a.use_hb_env_dep_ == b.use_hb_env_dep_ ) &&
					 ( a.use_hb_env_dep_DNA_ == b.use_hb_env_dep_DNA_ ) &&
					 ( a.smooth_hb_env_dep_ == b.smooth_hb_env_dep_ ) &&
					 ( a.decompose_bb_hb_into_pair_energies_ == b.decompose_bb_hb_into_pair_energies_ ) &&
					 ( a.bond_angle_central_atoms_to_score_ == b.bond_angle_central_atoms_to_score_ ) &&
					 ( a.bond_angle_residue_type_param_set_ == b.bond_angle_residue_type_param_set_ ) );
}

void
EnergyMethodOptions::show( std::ostream & out ) const
{
	if ( etable_type_.size() ) out << "EnergyMethodOptions::show: etable_type: " << etable_type_ <<'\n';
	for ( MethodWeights::const_iterator it=method_weights_.begin(), ite = method_weights_.end(); it != ite; ++it ) {
		out << "EnergyMethodOptions::show: method_weights: " << it->first;
		for ( Size i=1; i<= it->second.size(); ++i ) {
			out << ' ' << it->second[i];
		}
		out << '\n';
	}
	out << "EnergyMethodOptions::show: atom_vdw_atom_type_set_name: " << atom_vdw_atom_type_set_name_ << std::endl;
	out << "EnergyMethodOptions::show: exclude_protein_protein_hack_elec: "
	    << (exclude_protein_protein_hack_elec_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: exclude_DNA_DNA: "
	    << (exclude_DNA_DNA_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: use_hb_env_dep: "
	    << (use_hb_env_dep_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: use_hb_env_dep_DNA: "
	    << (use_hb_env_dep_DNA_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: smooth_hb_env_dep: "
	    << (smooth_hb_env_dep_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: decompose_bb_hb_into_pair_energies: "
	    << (decompose_bb_hb_into_pair_energies_ ? "true" : "false") << std::endl;
	out << "EnergyMethodOptions::show: bond_angle_central_atoms_to_score:";
	if (bond_angle_residue_type_param_set_) {
		out << "setting ignored";
	} else {
		for ( Size i=1; i <= bond_angle_central_atoms_to_score_.size(); ++i ) {
			out << " \"" << bond_angle_central_atoms_to_score_[i] << "\"";
		}
	}
	out << std::endl;
	out << "EnergyMethodOptions::show: bond_angle_residue_type_param_set: "
	    << (bond_angle_residue_type_param_set_ ? "in use" : "none") << std::endl;
	if (bond_angle_residue_type_param_set_) {
		out << "  central_atoms_to_score:";
		if (!bond_angle_residue_type_param_set_->central_atoms_to_score().size()) out << "all";
		for ( Size i=1; i <= bond_angle_residue_type_param_set_->central_atoms_to_score().size(); ++i ) {
			out << " \"" << bond_angle_residue_type_param_set_->central_atoms_to_score()[i] << "\"";
		}
		out << std::endl;
		out << "  use_residue_type_theta0: "
		    << (bond_angle_residue_type_param_set_->use_residue_type_theta0() ? "true" : "false") << std::endl;
	}
}


///
std::ostream &
operator<< ( std::ostream & out, EnergyMethodOptions const & options )
{
	options.show( out );
	return out;
}


}
}
}
