// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/ScoreFunction.hh
/// @brief  Score function class
/// @author Phil Bradley

/// NOTE-- this file includes both string and map, use .fwd.hh if
/// you can!


#ifndef INCLUDED_core_scoring_methods_EnergyMethodOptions_HH
#define INCLUDED_core_scoring_methods_EnergyMethodOptions_HH

#include <core/scoring/methods/EnergyMethodOptions.fwd.hh>

// Unit headers
#include <core/scoring/methods/EnergyMethod.fwd.hh>

#include <core/types.hh>
#include <core/scoring/ScoreType.hh>
#include <core/scoring/ScoringManager.fwd.hh> // FA_STANDARD_DEFAULT
#include <core/scoring/SecondaryStructureWeights.hh> /// REPLACE THIS WITH .fwd.hh

#include <core/chemical/ChemicalManager.fwd.hh> // CENTROID

#include <core/mm/MMBondAngleResidueTypeParamSet.fwd.hh>

/// Utility headers
#include <utility/exit.hh>
#include <utility/pointer/ReferenceCount.hh>

// C++ Headers
#include <string>
#include <map>

namespace core {
namespace scoring {
namespace methods {

/// add more options here
/// NOTE: If you add an option, make sure you also update the == comparison operator in this .hh file!
/// right now this class should be pretty light-weight since a copy is held inside ScoreFunctionInfo
///



class EnergyMethodOptions : public utility::pointer::ReferenceCount {
public:
	///
	EnergyMethodOptions();

	/// copy constructor
	EnergyMethodOptions( EnergyMethodOptions const & src );

	virtual
	~EnergyMethodOptions();

	/// copy operator
	EnergyMethodOptions const &
	operator=( EnergyMethodOptions const & src );

	///
	std::string const &
	etable_type() const
	{
		return etable_type_;
	}

	///
	void
	etable_type( std::string const & type )
	{
		etable_type_ = type;
	}

	///
	bool
	exclude_protein_protein_hack_elec() const
	{
		return exclude_protein_protein_hack_elec_;
	}

	///
	void
	exclude_protein_protein_hack_elec( bool const setting )
	{
		exclude_protein_protein_hack_elec_ = setting;
	}

	///
	bool
	exclude_DNA_DNA() const
	{
		return exclude_DNA_DNA_;
	}

	///
	void
	exclude_DNA_DNA( bool const setting )
	{
		exclude_DNA_DNA_ = setting;
	}

	/// @brief  This is used in the construction of the VDW_Energy's AtomVDW object
	std::string const &
	atom_vdw_atom_type_set_name() const
	{
		return atom_vdw_atom_type_set_name_;
	}

	///
	void
	atom_vdw_atom_type_set_name( std::string const & setting )
	{
		atom_vdw_atom_type_set_name_ = setting;
	}

	///
	bool
	use_hb_env_dep() const
	{
		return use_hb_env_dep_;
	}

	///
	void
	use_hb_env_dep( bool const setting )
	{
		use_hb_env_dep_ = setting;
	}

	bool use_hb_env_dep_DNA() const { return use_hb_env_dep_DNA_; }
	void use_hb_env_dep_DNA( bool setting ) { use_hb_env_dep_DNA_ = setting; }

	///
	bool
	smooth_hb_env_dep() const
	{
		return smooth_hb_env_dep_;
	}

	///
	void
	smooth_hb_env_dep( bool const setting )
	{
		smooth_hb_env_dep_ = setting;
	}

	///
	bool
	decompose_bb_hb_into_pair_energies() const
	{
		return decompose_bb_hb_into_pair_energies_;
	}

	///
	void
	decompose_bb_hb_into_pair_energies( bool const setting )
	{
		decompose_bb_hb_into_pair_energies_ = setting;
	}

	/// deprecated
	utility::vector1<std::string> const &
	bond_angle_central_atoms_to_score() const;

	/// deprecated
	void
	bond_angle_central_atoms_to_score(
		utility::vector1<std::string> const & atom_names
	);

	core::mm::MMBondAngleResidueTypeParamSetOP
	bond_angle_residue_type_param_set();

	core::mm::MMBondAngleResidueTypeParamSetCOP
	bond_angle_residue_type_param_set() const;

	void
	bond_angle_residue_type_param_set(
		core::mm::MMBondAngleResidueTypeParamSetOP param_set
	);

	void set_strand_strand_weights(
		int ss_lowstrand,
		int ss_cutoff
	) {
		ss_weights_.set_ss_cutoff   ( ss_cutoff    );
		ss_weights_.set_ss_lowstrand( ss_lowstrand );
	}

	///
	SecondaryStructureWeights const &
	secondary_structure_weights() const
	{
		return ss_weights_;
	}

	///
	SecondaryStructureWeights &
	secondary_structure_weights()
	{
		return ss_weights_;
	}

	///
	bool
	has_method_weights( ScoreType const & type ) const
	{
		return ( method_weights_.find( type ) != method_weights_.end() );
	}

	///
	utility::vector1< Real > const &
	method_weights( ScoreType const & type ) const
	{
		MethodWeights::const_iterator it( method_weights_.find( type ) );
		if ( it == method_weights_.end() ) {
			utility_exit_with_message( "EnergyMethodOptions::method_weights do not exist: "+name_from_score_type(type));
		}
		return it->second;
	}

	///
	void
	set_method_weights(
		ScoreType const & type,
		utility::vector1< Real > const & wts
	)
	{
		method_weights_[ type ] = wts;
	}

	/// used inside ScoreFunctionInfo::operator==
	friend
	bool
	operator==( EnergyMethodOptions const & a, EnergyMethodOptions const & b );

	/// used inside ScoreFunctionInfo::operator==
	friend
	inline
	bool
	operator!=( EnergyMethodOptions const & a, EnergyMethodOptions const & b )
	{
		return !( a == b );
	}

	///
	void
	show( std::ostream & out ) const;

private:
	/// expand this to a class and include ss weights inside
	typedef 	std::map< ScoreType, utility::vector1< Real > > MethodWeights;

private:

	/////////////////////////////////////////////////
	// SEE FOLLOWING NOTE!
	/////////////////////////////////////////////////
	std::string etable_type_;
	std::string atom_vdw_atom_type_set_name_;
	MethodWeights method_weights_;
	SecondaryStructureWeights ss_weights_;
	bool exclude_protein_protein_hack_elec_;
	bool exclude_DNA_DNA_;
	bool use_hb_env_dep_;
	bool use_hb_env_dep_DNA_;
	bool smooth_hb_env_dep_;
	bool decompose_bb_hb_into_pair_energies_;
	/// deprecated
	utility::vector1<std::string> bond_angle_central_atoms_to_score_;
	core::mm::MMBondAngleResidueTypeParamSetOP bond_angle_residue_type_param_set_;
	/// NOTE: If you add an option, make sure you also update the == comparison operator,
	/// the constructor, and the copy constructor in the .cc file!
};


std::ostream &
operator<< ( std::ostream & out, EnergyMethodOptions const & options );

}
}
}

#endif // INCLUDED_core_scoring_ScoreFunction_HH
