// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/ProClosureEnergy.cc
/// @brief  Proline ring closure energy method class implementation
/// @author Andrew Leaver-Fay

// Unit Headers
#include <core/scoring/methods/ProClosureEnergy.hh>
#include <core/scoring/methods/ProClosureEnergyCreator.hh>

// Package Headers
//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/constraints/DihedralConstraint.hh>
#include <core/scoring/constraints/HarmonicFunc.hh>

// Project headers
#include <core/conformation/Residue.hh>
#include <core/pose/Pose.hh>

// Utility headers
#include <numeric/conversions.hh>
#include <numeric/xyz.functions.hh>
#include <numeric/deriv/distance_deriv.hh>
#include <numeric/deriv/dihedral_deriv.hh>

// STL Headers
#include <string>

namespace core {
namespace scoring {
namespace methods {


/// @details This must return a fresh instance of the ProClosureEnergy class,
/// never an instance already in use
methods::EnergyMethodOP
ProClosureEnergyCreator::create_energy_method(
	methods::EnergyMethodOptions const &
) const {
	return new ProClosureEnergy;
}

ScoreTypes
ProClosureEnergyCreator::score_types_for_method() const {
	ScoreTypes sts;
	sts.push_back( pro_close );
	return sts;
}


using namespace numeric::constants::d;


/// ctor
ProClosureEnergy::ProClosureEnergy() :
	parent( new ProClosureEnergyCreator ),
	n_nv_dist_sd_( 0.1 ), // totally fictional

	/// measured from 4745 prolines from 1.25 A and higher resolution protein structures
	trans_chi4_mean_( 176.3 * numeric::constants::d::degrees_to_radians ),
	trans_chi4_sd_( 6.0158  * numeric::constants::d::degrees_to_radians ),
	cis_chi4_mean_( -2.9105 * numeric::constants::d::degrees_to_radians ),
	cis_chi4_sd_( 5.8239    * numeric::constants::d::degrees_to_radians ),

	bbN_( "N" ),
	scNV_( "NV" ),
	scCD_( "CD" ),
	bbC_( "C" ),
	bbO_( "O")
{}

ProClosureEnergy::~ProClosureEnergy()
{}

/// clone
EnergyMethodOP
ProClosureEnergy::clone() const
{
	return new ProClosureEnergy;
}

/////////////////////////////////////////////////////////////////////////////
// methods for ContextIndependentTwoBodyEnergies
/////////////////////////////////////////////////////////////////////////////
void
ProClosureEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const &,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{
	using namespace conformation;
	using namespace chemical;

	if (( rsd2.aa() == aa_pro && rsd1.is_bonded( rsd2 ) &&
			rsd1.seqpos() == rsd2.seqpos() - 1 ) ||
			(rsd1.aa() == aa_pro && rsd1.is_bonded( rsd2 ) &&
			rsd1.seqpos() - 1 == rsd2.seqpos() )) {
		Residue const & upper_res( rsd1.seqpos() > rsd2.seqpos() ? rsd1 : rsd2 );
		Residue const & lower_res( rsd1.seqpos() > rsd2.seqpos() ? rsd2 : rsd1 );

		Real chi4 = measure_chi4( lower_res, upper_res );
		emap[ pro_close ] += chi4E( chi4 );
	}
}

/// @brief Penalize the pucker-up residue type if its chi1 is positive;
/// penalize the pucker-down residue type if its chi1 is negative.  Only
/// applies this penalty when the other_residue is the next polymeric residue
/// after pro_residue (i+1), unless pro_residue is an upper_term,
/// in which case it applies the penalty for pro_residue's previous polymeric
/// residue.
void
ProClosureEnergy::bump_energy_full(
	conformation::Residue const & ,
	conformation::Residue const & ,
	pose::Pose const &,
	ScoreFunction const &,
	EnergyMap &
) const
{
	/*
	static const std::string prd_name( "PRD" );
	static const std::string pru_name( "PRU" );

	if ( pro_residue.aa() == chemical::aa_pro ) {
		if ( pro_residue.is_bonded( other_residue ) &&
				( ! pro_residue.is_upper_terminus() &&
				pro_residue.seqpos() + 1 == other_residue.seqpos() )
				||
				( pro_residue.is_upper_terminus() &&
				pro_residue.seqpos() == other_residue.seqpos() + 1 ) ) {
			if ( pro_residue.name3() == prd_name && pro_residue.chi( 1 ) < 0 ) {
				emap[ pro_close ] = 10;
			} else if ( pro_residue.name3() == pru_name && pro_residue.chi( 1 ) > 0 ) {
				emap[ pro_close ] = 10;
			}
		}
	}
	*/
}

/// @brief Penalize the pucker-up residue type if its chi1 is positive;
/// penalize the pucker-down residue type if its chi1 is negative.  Only
/// applies this penalty when the other_residue is the next polymeric residue
/// after pro_residue (i+1), unless pro_residue is an upper_term,
/// in which case it applies the penalty for pro_residue's previous polymeric
/// residue.
void
ProClosureEnergy::bump_energy_backbone(
	conformation::Residue const & ,
	conformation::Residue const & ,
	pose::Pose const & ,
	ScoreFunction const & ,
	EnergyMap &
) const
{
	//bump_energy_full( pro_residue, other_residue, pose, sfxn, emap );
}


bool
ProClosureEnergy::defines_intrares_energy( EnergyMap const & ) const
{
	return true;
}


///
void
ProClosureEnergy::eval_intrares_energy(
	conformation::Residue const & rsd,
	pose::Pose const & ,
	ScoreFunction const & ,
	EnergyMap & emap
) const
{
	/*
	static const std::string prd_name( "PRD" );
	static const std::string pru_name( "PRU" );
	*/

	if ( rsd.aa() == chemical::aa_pro ) {
		Distance const dist = rsd.xyz( bbN_ ).distance( rsd.xyz( scNV_ ) );
		emap[ pro_close ] = dist * dist / ( n_nv_dist_sd_ * n_nv_dist_sd_ );

		//if ( rsd.name3() == prd_name && rsd.chi( 1 ) < 0 ) {
		//	emap[ pro_close ] += 0.5;
		//} else if ( rsd.name3() == pru_name && rsd.chi( 1 ) > 0 ) {
		//	emap[ pro_close ] += 0.5;
		//}
	}

}

void
ProClosureEnergy::eval_atom_derivative(
	id::AtomID const & id,
	pose::Pose const & pose,
	kinematics::DomainMap const &,
	ScoreFunction const &,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
) const
{
	using namespace chemical;
	using namespace constraints;
	using namespace id;

	//std::cout << "proclose-eval_atom_derivative " << pose.residue(id.rsd()).name() << " " << pose.residue(id.rsd()).atom_name( id.atomno() ) << std::endl;
	if ( pose.residue( id.rsd() ).aa() == chemical::aa_pro ) {
		if ( pose.residue( id.rsd() ).atom_index( scNV_ ) ==  Size( id.atomno()) ) {

			Vector const & nv_pos( pose.residue( id.rsd() ).xyz( id.atomno() ));
			Vector const & n_pos( pose.residue( id.rsd() ).xyz( bbN_ ));
			/// Numeric deriv version to consolidate code.
			Vector f1( 0.0 ), f2( 0.0 );
			Distance dist( 0.0 );
			numeric::deriv::distance_f1_f2_deriv( nv_pos, n_pos, dist, f1, f2 );
			Real deriv( weights[ pro_close ] * 2 * dist / ( n_nv_dist_sd_ * n_nv_dist_sd_ ));
			F1 += deriv * f1;
			F2 += deriv * f2;

		} else if ( pose.residue( id.rsd() ).atom_index( bbN_ ) == Size( id.atomno()) ) {
			//std::cout << "evaluating N pro-closure energy derivative" << std::endl;
			Vector const & n_pos( pose.residue( id.rsd() ).xyz( id.atomno() ));
			Vector const & nv_pos( pose.residue( id.rsd() ).xyz( scNV_ ));

			{ /// Scope: Distance derivative.
			Vector f1( 0.0 ), f2( 0.0 );
			Distance dist( 0.0 );
			numeric::deriv::distance_f1_f2_deriv( n_pos, nv_pos, dist, f1, f2 );
			Real deriv( weights[ pro_close ] * 2 * dist / ( n_nv_dist_sd_ * n_nv_dist_sd_ ));
			F1 += deriv * f1;
			F2 += deriv * f2;
			}

			if ( id.rsd() != 1 && ! pose.residue( id.rsd() ).is_lower_terminus() &&
					pose.residue( id.rsd() ).is_bonded( pose.residue( id.rsd() - 1 ) ) ) {

				Vector f1( 0.0 ), f2( 0.0 );
				Real chi4( 0.0 );
				numeric::deriv::dihedral_p2_cosine_deriv(
					pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( scCD_ ), id.rsd() )),
					pose.xyz( id ),
					pose.xyz( AtomID( pose.residue( id.rsd()-1 ).atom_index( bbC_ ), id.rsd()-1 )),
					pose.xyz( AtomID( pose.residue( id.rsd()-1 ).atom_index( bbO_ ), id.rsd()-1 )),
					chi4, f1, f2 );
				Real deriv( weights[ pro_close ] * dchi4E_dchi4( chi4 ) );

				F1 += deriv * f1;
				F2 += deriv * f2;
			}

		} else if ( pose.residue( id.rsd() ).atom_index( scCD_ ) == Size( id.atomno()) ) {
			if ( id.rsd() != 1 && ! pose.residue( id.rsd() ).is_lower_terminus() &&
					pose.residue( id.rsd() ).is_bonded( pose.residue( id.rsd() - 1 ) ) ) {

				Vector f1( 0.0 ), f2( 0.0 );
				Real chi4( 0.0 );
				numeric::deriv::dihedral_p1_cosine_deriv(
					pose.xyz( id ),
					pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( bbN_ ), id.rsd() )),
					pose.xyz( AtomID( pose.residue( id.rsd()-1 ).atom_index( bbC_ ), id.rsd()-1 )),
					pose.xyz( AtomID( pose.residue( id.rsd()-1 ).atom_index( bbO_ ), id.rsd()-1 )),
					chi4, f1, f2 );
				Real deriv( weights[ pro_close ] * dchi4E_dchi4( chi4 ) );

				F1 += deriv * f1;
				F2 += deriv * f2;
			}
		}
	}

	/// Not an else-if since a proline may be followed by another proline
	if ( Size (id.rsd()) != pose.total_residue() &&
			! pose.residue( id.rsd() ).is_upper_terminus() &&
			pose.residue( id.rsd() + 1 ).aa() == aa_pro &&
			pose.residue( id.rsd() ).is_bonded( pose.residue( id.rsd() + 1 ))) {
		Vector f1( 0.0 ), f2( 0.0 );
		Real chi4( 0.0 );

		if ( pose.residue( id.rsd() ).atom_index( bbC_ ) ==  Size (id.atomno() )) {
			numeric::deriv::dihedral_p2_cosine_deriv(
				pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( bbO_ ), id.rsd() )),
				pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( bbC_ ), id.rsd() )),
				pose.xyz( AtomID( pose.residue( id.rsd() + 1 ).atom_index( bbN_ ), id.rsd() + 1 )),
				pose.xyz( AtomID( pose.residue( id.rsd() + 1 ).atom_index( scCD_ ), id.rsd() + 1 ) ),
				chi4, f1, f2 );
		} else if ( pose.residue( id.rsd() ).atom_index( bbO_ ) ==  Size( id.atomno() )) {
			numeric::deriv::dihedral_p1_cosine_deriv(
				pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( bbO_ ), id.rsd() )),
				pose.xyz( AtomID( pose.residue( id.rsd() ).atom_index( bbC_ ), id.rsd() )),
				pose.xyz( AtomID( pose.residue( id.rsd() + 1 ).atom_index( bbN_ ), id.rsd() + 1 )),
				pose.xyz( AtomID( pose.residue( id.rsd() + 1 ).atom_index( scCD_ ), id.rsd() + 1 ) ),
				chi4, f1, f2 );
		} else {
			return;
		}
		Real deriv( weights[ pro_close ] * dchi4E_dchi4( chi4 ) );

		F1 += deriv * f1;
		F2 += deriv * f2;
	}
}

/// @brief ProClosureEnergy Energy is context independent and thus
/// indicates that no context graphs need to
/// be maintained by class Energies
void
ProClosureEnergy::indicate_required_context_graphs(
	utility::vector1< bool > & /*context_graphs_required*/
)
const
{}

Real
ProClosureEnergy::measure_chi4(
	conformation::Residue const & lower_res,
	conformation::Residue const & upper_res
) const
{
	using namespace numeric;
	using namespace numeric::constants::d;

	assert( upper_res.aa() == chemical::aa_pro );
	assert( lower_res.is_bonded( upper_res ) );
	assert( lower_res.seqpos() == upper_res.seqpos() - 1 );

	Real chi4 = dihedral_radians(
		upper_res.xyz( scCD_ ),
		upper_res.xyz( bbN_ ),
		lower_res.xyz( bbC_ ),
		lower_res.xyz( bbO_ ));
	if ( chi4 < -pi_over_2 ) chi4 += pi_2; // wrap
	return chi4;
}

/// @details chi4 is in radians and should be in the range between -pi_over_2 and 3/2 pi
Real
ProClosureEnergy::chi4E(
	Real chi4
) const
{
	using namespace numeric::constants::d;

	if ( chi4 > pi_over_2 ) {
		Real diff = chi4 - trans_chi4_mean_;
		return diff * diff / ( trans_chi4_sd_ * trans_chi4_sd_ );
	} else {
		Real diff = chi4 - cis_chi4_mean_;
		return diff * diff / ( cis_chi4_sd_ * cis_chi4_sd_ );
	}
}


/// @details chi4 is in radians and should be in the range between -pi and pi
Real
ProClosureEnergy::dchi4E_dchi4(
	Real chi4
) const
{
	using namespace numeric::constants::d;
	if ( chi4 < -pi_over_2 ) chi4 += pi_2;

	if ( chi4 > pi_over_2 ) {
		Real diff = chi4 - trans_chi4_mean_;
		return 2 * diff / ( trans_chi4_sd_ * trans_chi4_sd_ );
	} else {
		Real diff = chi4 - cis_chi4_mean_;
		return 2 * diff / ( cis_chi4_sd_ * cis_chi4_sd_ );
	}
}


} // methods
} // scoring
} // core

