// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/ProClosureEnergy.hh
/// @brief  Proline ring closure energy method class declaration
/// @author Andrew Leaver-Fay


#ifndef INCLUDED_core_scoring_methods_ProClosureEnergy_HH
#define INCLUDED_core_scoring_methods_ProClosureEnergy_HH

// Unit headers
#include <core/scoring/methods/ProClosureEnergy.fwd.hh>

// Package headers
#include <core/scoring/methods/ContextIndependentTwoBodyEnergy.hh>


namespace core {
namespace scoring {
namespace methods {

///
class ProClosureEnergy : public ContextIndependentTwoBodyEnergy  {
public:
	typedef ContextIndependentTwoBodyEnergy  parent;

public:

	/// ctor
	ProClosureEnergy();

	// dstor
	~ProClosureEnergy();

	/// clone
	virtual
	EnergyMethodOP
	clone() const;

	/////////////////////////////////////////////////////////////////////////////
	// methods for ContextIndependentTwoBodyEnergies
	/////////////////////////////////////////////////////////////////////////////

	/// @brief Evaluate the interaction between a given residue pair
	virtual
	void
	residue_pair_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		TwoBodyEnergyMap & emap
	) const;

	/// @brief Penalize the pucker-up residue type if its chi1 is positive;
	/// penalize the pucker-down residue type if its chi1 is negative.  Only
	/// applies this penalty when the other_residue is the next polymeric residue
	/// after pro_residue (i+1), unless pro_residue is an upper_term,
	/// in which case it applies the penalty for pro_residue's previous polymeric
	/// residue.
	virtual
	void
	bump_energy_full(
		conformation::Residue const & pro_residue,
		conformation::Residue const & other_residue,
		pose::Pose const &,
		ScoreFunction const &,
		EnergyMap &
	) const;

	/// @brief Penalize the pucker-up residue type if its chi1 is positive;
	/// penalize the pucker-down residue type if its chi1 is negative.  Only
	/// applies this penalty when the other_residue is the next polymeric residue
	/// after pro_residue (i+1), unless pro_residue is an upper_term,
	/// in which case it applies the penalty for pro_residue's previous polymeric
	/// residue.
	virtual
	void
	bump_energy_backbone(
		conformation::Residue const & pro_residue,
		conformation::Residue const & other_residue,
		pose::Pose const &,
		ScoreFunction const &,
		EnergyMap &
	) const;


	virtual
	bool
	defines_intrares_energy( EnergyMap const & weights ) const;

	virtual
	void
	eval_intrares_energy(
		conformation::Residue const & rsd,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;

	void
	eval_atom_derivative(
		id::AtomID const & id,
		pose::Pose const & pose,
		kinematics::DomainMap const &, // domain_map,
		ScoreFunction const & /*sfxn*/, // needed for non-nblist minimization
		EnergyMap const & weights,
		Vector & F1,
		Vector & F2
	) const;

	/// @brief ProClosure Energy is context independent and thus
	/// indicates that no context graphs need to
	/// be maintained by class Energies
	virtual
	void indicate_required_context_graphs(
		utility::vector1< bool > & /*context_graphs_required*/
	) const;

	virtual
	Distance
	atomic_interaction_cutoff() const
	{ return 0.0; }

private:

	/// @brief measure in radians the chi4 between two residues;
	/// upper_residue must be a proline chi4 is wrapped to be in
	/// the range [-pi_over_2, 3/2*pi )
	Real
	measure_chi4(
		conformation::Residue const & lower_residue,
		conformation::Residue const & upper_residue
	) const;

	Real
	chi4E(
		Real chi4
	) const;

	Real
	dchi4E_dchi4(
		Real chi4
	) const;

	// data
private:
	Real const n_nv_dist_sd_; // coorinate variation stdev

	Real const trans_chi4_mean_;
	Real const trans_chi4_sd_;
	Real const cis_chi4_mean_;
	Real const cis_chi4_sd_;

	std::string const bbN_;
	std::string const scNV_;
	std::string const scCD_;
	std::string const bbC_;
	std::string const bbO_;

};

} // methods
} // scoring
} // core


#endif // INCLUDED_core_scoring_EtableEnergy_HH
