// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/ShortRangeTwoBodyEnergy.cc
/// @brief  Short-Range, Two-Body Energy Method base class implementation
/// @author Andrew Leaver-Fay

// Unit Headers
#include <core/scoring/methods/ShortRangeTwoBodyEnergy.hh>

// Package Headers
#include <core/scoring/EnergyMap.hh>
#include <core/scoring/methods/EnergyMethodCreator.hh>

// Project Headers
#include <core/conformation/Residue.hh>
#include <core/pack/rotamer_set/RotamerSet.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray2D.hh>


namespace core {
namespace scoring {
namespace methods {

ShortRangeTwoBodyEnergy::ShortRangeTwoBodyEnergy( EnergyMethodCreatorOP creator ) : parent( creator ) {}

ShortRangeTwoBodyEnergy::~ShortRangeTwoBodyEnergy() {}

void
ShortRangeTwoBodyEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap & emap
) const
{
	TwoBodyEnergyMap tbemap;
	residue_pair_energy( rsd1, rsd2, pose, sfxn, tbemap );
	emap.accumulate( tbemap, score_types() );
}

void
ShortRangeTwoBodyEnergy::backbone_backbone_energy(
	conformation::Residue const & ,
	conformation::Residue const & ,
	pose::Pose const & ,
	ScoreFunction const & ,
	TwoBodyEnergyMap &
) const
{}


void
ShortRangeTwoBodyEnergy::backbone_sidechain_energy(
	conformation::Residue const & ,
	conformation::Residue const & ,
	pose::Pose const & ,
	ScoreFunction const & ,
	TwoBodyEnergyMap &
) const
{}


void
ShortRangeTwoBodyEnergy::sidechain_sidechain_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	TwoBodyEnergyMap & emap
) const
{
	residue_pair_energy( rsd1, rsd2, pose, sfxn, emap );
}


void
ShortRangeTwoBodyEnergy::evaluate_rotamer_pair_energies(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
) const
{
	using namespace conformation;
	using namespace numeric;

	TwoBodyEnergyMap emap;

	for ( Size ii = 1; ii <= set1.get_n_residue_types(); ++ii ) {
		Size const ii_offset = set1.get_residue_type_begin( ii );
		Residue const & ii_example_rotamer( *set1.rotamer( ii_offset ));
		Vector const & ii_coord( ii_example_rotamer.atom( ii_example_rotamer.type().nbr_atom() ).xyz());
		Real const ii_radius( ii_example_rotamer.type().nbr_radius() );

		for ( Size jj = 1; jj <= set2.get_n_residue_types(); ++jj ) {
			Size const jj_offset = set2.get_residue_type_begin( jj );
			Residue const & jj_example_rotamer( *set2.rotamer( jj_offset ));
			Vector const & jj_coord( jj_example_rotamer.atom( jj_example_rotamer.type().nbr_atom() ).xyz());
			Real const jj_radius( jj_example_rotamer.type().nbr_radius() );

			if ( ii_coord.distance_squared( jj_coord ) < std::pow(ii_radius+jj_radius+atomic_interaction_cutoff(), 2 )) {
				for ( Size kk = 1, kke = set1.get_n_rotamers_for_residue_type( ii ); kk <= kke; ++kk ) {
					Size const kk_rot_id = ii_offset + kk - 1;
					for ( Size ll = 1, lle = set2.get_n_rotamers_for_residue_type( jj ); ll <= lle; ++ll ) {
						Size const ll_rot_id = jj_offset + ll - 1;

						emap.zero();
						residue_pair_energy( *set1.rotamer( kk_rot_id ), *set2.rotamer( ll_rot_id ), pose, sfxn, emap );
						energy_table( ll_rot_id, kk_rot_id ) += static_cast< pack::PackerEnergy > (weights.dot( emap ));
					}
				}
			}
		}
	}

}

void
ShortRangeTwoBodyEnergy::evaluate_rotamer_background_energies(
	pack::rotamer_set::RotamerSet const & set,
	conformation::Residue const & residue,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	utility::vector1< pack::PackerEnergy > & energy_vector
) const
{
	TwoBodyEnergyMap emap;
	for ( Size ii = 1, ii_end = set.num_rotamers(); ii <= ii_end; ++ii ) {
		emap.zero();
		residue_pair_energy( *set.rotamer( ii ), residue, pose, sfxn, emap );
		energy_vector[ ii ] += static_cast< pack::PackerEnergy > (weights.dot( emap ));
	}
}

void
ShortRangeTwoBodyEnergy::evaluate_rotamer_background_energy_maps(
	pack::rotamer_set::RotamerSet const & set,
	conformation::Residue const & residue,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & ,	// weights
	utility::vector1< EnergyMap > & emaps
) const
{
	TwoBodyEnergyMap emap;
	for ( Size ii = 1, ii_end = set.num_rotamers(); ii <= ii_end; ++ii ) {
		emap.zero();
		residue_pair_energy( *set.rotamer( ii ), residue, pose, sfxn, emap );
		emaps[ ii ] += emap;
	}
}

}
}
}

