// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/TwoBodyEnergy.hh
/// @brief  Two Body Energy Method base class declaration
/// @author Andrew Leaver-Fay

#ifndef INCLUDED_core_scoring_methods_TwoBodyEnergy_HH
#define INCLUDED_core_scoring_methods_TwoBodyEnergy_HH

// Unit Headers
#include <core/scoring/methods/TwoBodyEnergy.fwd.hh>

// Package headers
#include <core/scoring/methods/EnergyMethod.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

// Project headers
#include <core/pack/types.hh>
#include <core/conformation/Residue.fwd.hh>
#include <core/pose/Pose.fwd.hh>
#include <core/id/AtomID.fwd.hh>
#include <core/pack/rotamer_set/RotamerSet.fwd.hh>

// ObjexxFCL Headers
#include <ObjexxFCL/FArray2D.fwd.hh>


/// The Two Body Energy Method specifies an interface for all two body methods: both
/// long and short range, both context dependent and independent.  Any two body method
/// must implement this interface as well as the EnergyMethod interface.
///
/// In this inheritance heirarchy, a common interface is defined for both context
/// dependent and context independent two body energies.  Context dependent two
/// body energies require a Pose as context to evaluate the energy and so the interface
/// includes a Pose for all energy evaluation calls.
///
/// However, context independent methods do not require a pose, and could
/// conceivably be evaluated in the absence of one.  Is the design compromised
/// by demanding a common interface between context independent and context
/// dependent two-body methods?  Yes and no.  The alternatives are 1) to demand
/// separate, but nearly identical interfaces for ci2b and cd2b classes in which
/// case, the design is compromised by the presence of duplicated code, 2) to
/// use multiple inherritance so that the CI2B class could derive from a
/// two-body interface (shared with the CD2B class) as well as a contex-independent
/// interface (not shared with the CD2B class).  The problem is that most of the
/// "shared" interface involves evaluating energies, and for the CD classes, this
/// requires a Pose.  The truely shared interface between CI and CD classes would
/// be empty and problem 1 has resurfaced.  Moreover, multiple inherritance is
/// difficult to work with and should be avoided.
///
/// What is the drawback of requiring a pose for a CI2B method evaluation?  It makes
/// it less clear to the energy-method user (not the energy-method writer) that the
/// Pose does not play a role in the evaluation of the ci2b energy.  It also makes it
/// possible for the energy-method writer to define a context dependent two body method
/// while declaring it to be context independent: such an act would produce bizzare
/// (wrong) behavior as the innards of the ScoreFunction and the Energies class tried
/// to re-use cached scores that were no longer correct.  So the warning must be made:
///
/// DANGER! DANGER! DANGER!
/// If the EnergyMethod writer declares a method to be context independent, it
/// had better be!


namespace core {
namespace scoring {
namespace methods {

class TwoBodyEnergy : public EnergyMethod
{
public:
	typedef EnergyMethod parent;

public:
	TwoBodyEnergy( EnergyMethodCreatorOP );
	virtual ~TwoBodyEnergy();

	/// @brief Evaluate the interaction between a given residue pair
	/// accumulating the unweighted energies in an EnergyMap
	virtual
	void
	residue_pair_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const = 0;

	///@brief Evaluate the interaction between the backbone of rsd1 and the
	/// backbone of rsd2 and accumulate the unweighted energies.  The sum
	/// bb_bb(r1,r2) + bb_sc(r1,r2) + bb_sc(r2,r1) + sc_sc( r1,r2) must
	/// equal the weighted result of a call to residue_pair_energy.
	/// By default, bb_bb & bb_sc return 0 and sc_sc returns
	/// residue pair energy.
	virtual
	void
	backbone_backbone_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;


	///@brief Evaluate the interaction between the backbone of rsd1 and the
	/// sidechain of rsd2 and accumulate the unweighted energies.  The sum
	/// bb_bb(r1,r2) + bb_sc(r1,r2) + bb_sc(r2,r1) + sc_sc( r1,r2) must
	/// equal the unweighted result of a call to residue_pair_energy.
	/// By default, bb_bb & bb_sc return 0 and sc_sc returns
	/// residue pair energy.
	virtual
	void
	backbone_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;

	///@brief Evaluate the interaction between the sidechain of rsd1 and the
	/// sidechain of rsd2 and accumulate the unweighted energies.  The sum
	/// bb_bb(r1,r2) + bb_sc(r1,r2) + bb_sc(r2,r1) + sc_sc( r1,r2) must
	/// equal the unweighted result of a call to residue_pair_energy.
	/// By default, bb_bb & bb_sc return 0 and sc_sc returns
	/// residue pair energy.
	virtual
	void
	sidechain_sidechain_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const;

	/// @brief Two body energies are able to define intra-residue energies, and to do so
	/// only in the presence of certain non-zero weights.  The ScoreFunction will hand over its
	/// weight set as it asks whether the energy method defines an intraresidue energy or not.
	///
	/// For example, the Etable method defines intra-residue energies only when one or more
	/// of the fa_intra_{atr,rep,sol} weights are non-zero.
	virtual
	bool
	defines_intrares_energy( EnergyMap const & weights ) const = 0;

	/// @brief Evaluate the intra-residue energy for a given residue
	virtual
	void
	eval_intrares_energy(
		conformation::Residue const & rsd,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap & emap
	) const = 0;

	// @brief Low fidelity evaluation of sc/bb + sc/sc energy.
	// Do not define in derived class if that class should not be
	// used in the packer's bump-check phase.
	virtual
	void
	bump_energy_full(
		conformation::Residue const &,
		conformation::Residue const &,
		pose::Pose const &,
		ScoreFunction const &,
		EnergyMap &
	) const;

	// @brief Low fidelity evaluation of sc/bb energy.
	// Do not define in derived class if that class should not be
	// used in the packer's bump-check phase.
	virtual
	void
	bump_energy_backbone(
		conformation::Residue const &,
		conformation::Residue const &,
		pose::Pose const &,
		ScoreFunction const &,
		EnergyMap &
	) const;

	/// @brief Batch computation of rotamer intrares energies.  Need not be overriden in
	/// derived class -- by default, iterates over all rotamers,
	/// and calls derived class's intrares _energy method.
	virtual
	void
	evaluate_rotamer_intrares_energies(
		pack::rotamer_set::RotamerSet const & set,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		utility::vector1< pack::PackerEnergy > & energies
	) const;

	/// @brief Batch computation of rotamer intrares energy map.  Need not be overriden in
	/// derived class -- by default, iterates over all rotamers,
	/// and calls derived class's intrares _energy method.
	virtual
	void
	evaluate_rotamer_intrares_energy_maps(
		pack::rotamer_set::RotamerSet const & set,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		utility::vector1< EnergyMap > & emaps
	) const;

	/// @brief Batch computation of rotamer pair energies.  Need not be overriden in
	/// derived class -- by default, iterates over all pairs of rotamers,
	/// and calls derived class's residue_pair_energy method.
	virtual
	void
	evaluate_rotamer_pair_energies(
		pack::rotamer_set::RotamerSet const & set1,
		pack::rotamer_set::RotamerSet const & set2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
	) const;

	/// @brief Batch computation of rotamer/background energies.  Need not be overriden
	/// in derived class -- by default, iterates over all rotamers in the set, and calls
	/// derived class's residue_pair_energy method for each one against the background rotamr
	virtual
	void
	evaluate_rotamer_background_energies(
		pack::rotamer_set::RotamerSet const & set,
		conformation::Residue const & residue,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		utility::vector1< pack::PackerEnergy > & energy_vector
	) const;

	/// @brief Batch computation of rotamer/background energies.  Need not be overriden
	/// in derived class -- by default, iterates over all rotamers in the set, and calls
	/// derived class's residue_pair_energy method for each one against the background rotamr
	virtual
	void
	evaluate_rotamer_background_energy_maps(
		pack::rotamer_set::RotamerSet const & set,
		conformation::Residue const & residue,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		utility::vector1< EnergyMap > & emaps
	) const;

};

}
}
}

#endif
