// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
// :noTabs=false:tabSize=4:indentSize=4:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/rms/rms_util.hh
/// @brief  RMS stuff from rosetta++
/// @author James Thompson
/// @author Ian Davis
/// @date   Wed Aug 22 12:10:37 2007
///

#ifndef core_scoring_rms_util_HH
#define core_scoring_rms_util_HH

#include <core/pose/Pose.fwd.hh>

#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <core/id/types.hh>
#include <core/conformation/Residue.hh>

#include <numeric/model_quality/rms.hh>
#include <numeric/model_quality/maxsub.hh>
#include <numeric/xyzVector.hh>

#include <ObjexxFCL/FArray1D.hh>
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1Da.hh>
#include <ObjexxFCL/FArray2Da.hh>

#include <list>

namespace core {
namespace scoring {


/// @brief RMSD between residues, accounting for automorphisms (symmetries).
/// Does NOT include H atoms -- they add lots of extra symmetries.
core::Real
automorphic_rmsd(
	core::conformation::Residue const & rsd1,
	core::conformation::Residue const & rsd2,
	bool superimpose
);


//////////////////////////////////////////////////////////////////////////////
// Predicate functions to use with rmsd_no_super() and rmsd_with_super()

bool
is_protein_CA(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

bool
is_protein_CA_or_CB(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

bool
is_protein_backbone(
	core::pose::Pose const& pose1,
	core::pose::Pose const&,
	core::Size resno,
	core::Size atomno
);

bool
is_polymer_heavyatom(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

bool
is_ligand_heavyatom(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

bool
is_heavyatom(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

//////////////////////////////////////////////////////////////////////////////
// "Predicated" versions of RMSD calculations
// (templated functions must be defined in .hh, not .cc)


/// @brief Select atoms for RMS via a predicate function/functor.
/// @details Calculates minimal rms, allowing rotation/translation for best fit.
/// Parameter "predicate" should be a function pointer
/// [ or class that defines operator() ] with the following signature:
///
///   bool my_pred_func(Pose const & pose1, Pose const & pose2, Size resno, Size atomno);
///
/// It should return true if the atom should be included and false otherwise.
///
/// Example of use, to calculate C-alpha RMSD:
///   rmsd_with_super(pose1, pose2, is_protein_CA);
template< class T >
core::Real
rmsd_with_super(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	T* predicate
)
{
	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	std::vector< core::Vector > p1_coords;
	std::vector< core::Vector > p2_coords;

	for ( core::Size i = 1; i <= nres1; ++i ) {
		Size num_atoms ( pose1.residue(i).natoms() );
		if ( predicate == is_ligand_heavyatom ||
			predicate == is_polymer_heavyatom ||
			predicate == is_heavyatom ) {
			//			assert( pose1.residue(i).nheavyatoms() == pose2.residue(i).nheavyatoms() );
		} else if ( num_atoms > pose2.residue(i).natoms() ){
			num_atoms = pose2.residue(i).natoms();
		}
		for ( core::Size j = 1; j <= num_atoms; ++j ) {
			if( j <= pose1.residue(i).natoms() &&
			    j <= pose2.residue(i).natoms() ) {
				if ( predicate( pose1, pose2, i, j ) ) {
					p1_coords.push_back( pose1.residue(i).xyz(j) );
					p2_coords.push_back( pose2.residue(i).xyz(j) );
				}
			}
		}
	}
	assert( p1_coords.size() == p2_coords.size() );

	int const natoms = p1_coords.size();
	ObjexxFCL::FArray2D< core::Real > p1a( 3, natoms );
	ObjexxFCL::FArray2D< core::Real > p2a( 3, natoms );
	for ( int i = 0; i < natoms; ++i ) {
		for ( int k = 0; k < 3; ++k ) { // k = X, Y and Z
			p1a(k+1,i+1) = p1_coords[i][k];
			p2a(k+1,i+1) = p2_coords[i][k];
		}
	}

	return numeric::model_quality::rms_wrapper( natoms, p1a, p2a );
}


/// @brief Select a subset atoms for RMS via a predicate function/functor.
/// @details Calculates minimal rms, allowing rotation/translation for best fit.
///		Same as above function, but allows a subset of residues over which to
///		superposition to be passed in
/// Parameter "predicate" should be a function pointer
/// [ or class that defines operator() ] with the following signature:
///
///   bool my_pred_func(Pose const & pose1, Pose const & pose2, Size resno, Size atomno);
///
/// It should return true if the atom should be included and false otherwise.
///
/// Example of use, to calculate C-alpha RMSD:
///   rmsd_with_super(pose1, pose2, is_protein_CA, subset);
template< class T >
core::Real
rmsd_with_super_subset(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	ObjexxFCL::FArray1D_bool const & subset,
	T* predicate
)
{
	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	std::vector< core::Vector > p1_coords;
	std::vector< core::Vector > p2_coords;

	for ( core::Size i = 1; i <= nres1; ++i ) {
		if ( subset( i ) ) {
			Size num_atoms ( pose1.residue(i).natoms() );
			if ( predicate == is_ligand_heavyatom ||
				predicate == is_polymer_heavyatom ||
				predicate == is_heavyatom ) {
				assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
			} else if ( num_atoms > pose2.residue(i).natoms() ){
				num_atoms = pose2.residue(i).natoms();
			}
			for ( core::Size j = 1; j <= num_atoms; ++j ) {
				if ( predicate( pose1, pose2, i, j ) ) {
					p1_coords.push_back( pose1.residue(i).xyz(j) );
					p2_coords.push_back( pose2.residue(i).xyz(j) );
				}
			}
		}
	}
	assert( p1_coords.size() == p2_coords.size() );

	int const natoms = p1_coords.size();
	ObjexxFCL::FArray2D< core::Real > p1a( 3, natoms );
	ObjexxFCL::FArray2D< core::Real > p2a( 3, natoms );
	for ( int i = 0; i < natoms; ++i ) {
		for ( int k = 0; k < 3; ++k ) { // k = X, Y and Z
			p1a(k+1,i+1) = p1_coords[i][k];
			p2a(k+1,i+1) = p2_coords[i][k];
		}
	}

	return numeric::model_quality::rms_wrapper( natoms, p1a, p2a );
}


/// @brief Select atoms for RMS via a predicate function/functor.
/// @details Calculates minimal rms, NOT allowing rotation/translation --
/// uses current coordinates as-is.
/// Parameter "predicate" should be a function pointer
/// [ or class that defines operator() ] with the following signature:
///
///   bool my_pred_func(Pose const & pose1, Pose const & pose2, Size resno, Size atomno);
///
/// It should return true if the atom should be included and false otherwise.
///
/// Example of use, to calculate C-alpha RMSD:
///   rmsd_no_super(pose1, pose2, is_protein_CA);
template< class T >
core::Real
rmsd_no_super(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	T* predicate
)
{
	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	core::Real sum2( 0.0 );
	core::Size natoms( 0 );
	for ( core::Size i = 1; i <= nres1; ++i ) {
		Size num_atoms ( pose1.residue(i).natoms() );
		if ( predicate == is_ligand_heavyatom ||
			predicate == is_polymer_heavyatom ||
			predicate == is_heavyatom ) {
			assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
		} else if ( num_atoms > pose2.residue(i).natoms() ){
			num_atoms = pose2.residue(i).natoms();
		}
		for ( core::Size j = 1; j <= num_atoms; ++j ) {
			if ( predicate( pose1, pose2, i, j ) ) {
				core::Vector diff = pose1.residue(i).xyz(j) - pose2.residue(i).xyz(j);
				sum2 += diff.length_squared();
				natoms += 1;
			}
		}
	}
	return std::sqrt(sum2 / natoms);
}


/// @brief Select atoms for RMS via a predicate function/functor.
/// @details Calculates minimal rms over a subset of residues,
/// NOT allowing rotation/translation --
/// uses current coordinates as-is.
/// Parameter "predicate" should be a function pointer
/// [ or class that defines operator() ] with the following signature:
///
///   bool my_pred_func(Pose const & pose1, Pose const & pose2, Size resno, Size atomno);
/// the subset is a vector of all the residues, with true for those
/// over which to calculate the rms
///
/// It should return true if the atom should be included and false otherwise.
///
/// Example of use, to calculate C-alpha RMSD:
///   rmsd_no_super_subset(pose1, pose2, subset, is_protein_CA);
template< class T >
core::Real
rmsd_no_super_subset(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	ObjexxFCL::FArray1D_bool const & subset,
	T* predicate
)
{
	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	core::Real sum2( 0.0 );
	core::Size natoms( 0 );
	for ( core::Size i = 1; i <= nres1; ++i ) {
	if ( subset( i ) ) {
		Size num_atoms ( pose1.residue(i).natoms() );
			if ( predicate == is_ligand_heavyatom ||
				predicate == is_polymer_heavyatom ||
				predicate == is_heavyatom ) {
				assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
			} else if ( num_atoms > pose2.residue(i).natoms() ){
				num_atoms = pose2.residue(i).natoms();
			}
			for ( core::Size j = 1; j <= num_atoms; ++j ) {
				if ( predicate( pose1, pose2, i, j ) ) {
					core::Vector diff = pose1.residue(i).xyz(j) - pose2.residue(i).xyz(j);
					sum2 += diff.length_squared();
					natoms += 1;
				}
			}
		}
	}
	return std::sqrt(sum2 / natoms);
}

/// @brief function to return the biggest deviation between an atom in a pair of poses,
template< class T >
core::Real
biggest_residue_deviation_no_super(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	T* predicate
){

	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	core::Real biggest_dev2( 0.0 );
	for ( core::Size i = 1; i <= nres1; ++i ) {

		Real res_dev2(0.0);
		Size res_count_atoms(0);
		Size num_atoms ( pose1.residue(i).natoms() );
		if ( predicate == is_ligand_heavyatom ||
			predicate == is_polymer_heavyatom ||
			predicate == is_heavyatom ) {
			assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
		} else if ( num_atoms > pose2.residue(i).natoms() ){
			num_atoms = pose2.residue(i).natoms();
		}
		for ( core::Size j = 1; j <= num_atoms; ++j ) {
			if ( predicate( pose1, pose2, i, j ) ) {
				core::Vector diff = pose1.residue(i).xyz(j) - pose2.residue(i).xyz(j);
				res_dev2 += diff.length_squared();
				res_count_atoms++;
			}
		}
		res_dev2 /= res_count_atoms;
		if( res_dev2 > biggest_dev2 ) biggest_dev2 = res_dev2;
	}
	return std::sqrt( biggest_dev2 );
}


/// @brief function to return the biggest deviation between an atom in a pair of poses,
/// @brief as specified by the predicate and the subset
template< class T >
core::Real
biggest_residue_deviation_no_super_subset(
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	ObjexxFCL::FArray1D_bool const & subset,
	T* predicate
){

	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	assert( nres1 == nres2 );

	core::Real biggest_dev2( 0.0 );
	for ( core::Size i = 1; i <= nres1; ++i ) {
	if ( subset( i ) ) {
		Real res_dev2(0.0);
		Size res_count_atoms(0);
		Size num_atoms ( pose1.residue(i).natoms() );
			if ( predicate == is_ligand_heavyatom ||
				predicate == is_polymer_heavyatom ||
				predicate == is_heavyatom ) {
				assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
			} else if ( num_atoms > pose2.residue(i).natoms() ){
				num_atoms = pose2.residue(i).natoms();
			}
			for ( core::Size j = 1; j <= num_atoms; ++j ) {
				if ( predicate( pose1, pose2, i, j ) ) {
					core::Vector diff = pose1.residue(i).xyz(j) - pose2.residue(i).xyz(j);
					res_dev2 += diff.length_squared();
					res_count_atoms++;
				}
			}
			res_dev2 /= res_count_atoms;
			if( res_dev2 > biggest_dev2 ) biggest_dev2 = res_dev2;
		}
	}
	return std::sqrt( biggest_dev2 );
}

bool
is_nbr_atom(
	core::pose::Pose const & pose1,
	core::pose::Pose const & ,//pose2,
	core::Size resno,
	core::Size atomno
);

//////////////////////////////////////////////////////////////////////////////

core::Real
CA_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2
);

// compute rmsd for residues between start and end
core::Real
CA_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	Size start,
	Size end
);

// compute rmsd for residues between start and end
core::Real
CA_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	Size start,
	Size end,
	utility::vector1< Size > const& exclude
);

// compute rmsd for residues for residues in list
core::Real
CA_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	std::list<Size> residue_selection
);

core::Real
all_atom_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2
);

core::Real
all_atom_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	std::list< Size > residue_selection
);

core::Real
nbr_atom_rmsd(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2
);

void
fill_rmsd_coordinates(
	int & natoms,
	ObjexxFCL::FArray2D< core::Real > & p1a,
	ObjexxFCL::FArray2D< core::Real > & p2a,
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	std::string atom_name
);

template< class T >
void
fill_rmsd_coordinates(
	int & natoms,
	ObjexxFCL::FArray2D< core::Real > & p1a,
	ObjexxFCL::FArray2D< core::Real > & p2a,
	core::pose::Pose const & pose1,
	core::pose::Pose const & pose2,
	T* predicate
)
{
	core::Size const nres1 = pose1.total_residue();
	core::Size const nres2 = pose2.total_residue();
	//assert( nres1 == nres2 );

	std::vector< core::Vector > p1_coords;
	std::vector< core::Vector > p2_coords;

	for ( core::Size i = 1; i <= std::min( nres1, nres2 ); ++i ) {
		//assert( pose1.residue(i).natoms() == pose2.residue(i).natoms() );
		for ( core::Size j = 1; j <= pose1.residue(i).natoms(); ++j ) {
			if ( (*predicate)( pose1, pose2, i, j ) ) {
				p1_coords.push_back( pose1.residue(i).xyz(j) );
				p2_coords.push_back( pose2.residue(i).xyz(j) );
			}
		}
	}
	assert( p1_coords.size() == p2_coords.size() );

	natoms = p1_coords.size();
	p1a.dimension( 3, natoms );
	p2a.dimension( 3, natoms );
	for ( int i = 0; i < natoms; ++i ) {
		for ( int k = 0; k < 3; ++k ) { // k = X, Y and Z
			p1a(k+1,i+1) = p1_coords[i][k];
			p2a(k+1,i+1) = p2_coords[i][k];
		}
	}
}



// other model-quality related functions

/// @brief Calculates a C-alpha maxsub-based superposition between pose1 and pose2, returns
/// the number of residues superimposed past a certain threshold. See maxsub.hh and maxsub.cc
/// for more information.
int
CA_maxsub(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	Real rms = 4.0
);

int
CA_maxsub(
	const core::pose::Pose & pose1,
	const core::pose::Pose & pose2,
	std::list<Size> residue_selection,
	Real rms = 4.0
);


int xyz_maxsub(
	FArray2D< core::Real > p1a,
	FArray2D< core::Real > p2a,
	int natoms
);

/// @brief Calculate gdtmm based on the given sets of xyz coordinates in p1a and p2a.
core::Real
xyz_gdtmm(
	FArray2D< core::Real > p1a,
	FArray2D< core::Real > p2a,
	core::Real& m_1_1,
	core::Real& m_2_2,
	core::Real& m_3_3,
	core::Real& m_4_3,
	core::Real& m_7_4
);

/// @brief Calculate gdtmm based on the given sets of xyz coordinates in p1a and p2a.
core::Real
xyz_gdtmm(
	FArray2D< core::Real > p1a,
	FArray2D< core::Real > p2a
);


/// @brief Calculate gdtmm score based on the C-alpha positions in pose1 and pose2.
core::Real
CA_gdtmm(
	core::pose::Pose const& pose1,
	core::pose::Pose const& pose2
);

/// @brief Calculate gdtmm score based on the C-alpha positions in pose1 and pose2. Also returns the
/// five components of the gdtmm score.
core::Real
CA_gdtmm(
	core::pose::Pose const& pose1,
	core::pose::Pose const& pose2,
	core::Real& m_1_1,
	core::Real& m_2_2,
	core::Real& m_3_3,
	core::Real& m_4_3,
	core::Real& m_7_4
);

core::Real
CA_gdtmm(
	core::pose::Pose const& pose1,
	core::pose::Pose const& pose2,
	std::list<Size> residue_selection,
	core::Real& m_1_1,
	core::Real& m_2_2,
	core::Real& m_3_3,
	core::Real& m_4_3,
	core::Real& m_7_4
);

/// @brief Calculate gdtmm score based on the C-alpha positions in pose1 and pose2.
core::Real
CA_gdtmm(
	core::pose::Pose const& pose1,
	core::pose::Pose const& pose2,
	std::list<Size> residue_selection
);

/// @brief  Superimpose mod_pose onto ref_pose using the mapping of atoms from
/// mod_pose to ref_pose given by atom_map
Real
superimpose_pose(
	pose::Pose & mod_pose,
	pose::Pose const & ref_pose,
	id::AtomID_Map< id::AtomID > const & atom_map // from mod_pose to ref_pose
);

/// @brief Superimpose two poses by their calpha coordinates.  Ignores residues
/// that do not have atoms named "CA."
Real
calpha_superimpose_pose(
	pose::Pose & mod_pose,
	pose::Pose const & ref_pose
);

core::Real
CA_rmsd_symmetric(
  const core::pose::Pose & pose1,
  const core::pose::Pose & pose2
);

void
create_shuffle_map_recursive_rms(
	std::vector<int> sequence,
	int const N,
	std::vector< std::vector<int> > & map
);

} // end namespace scoring
} // end namespace core

#endif
