// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/HackElecEnergy.cc
/// @brief  Statistically derived rotamer pair potential class implementation
/// @author Phil Bradley
/// @author Andrew Leaver-Fay


// Unit headers
#include <core/scoring/hackelec/HackElecEnergy.hh>

// Package headers
//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/EnergyGraph.hh>
#include <core/scoring/Energies.hh>
#include <core/scoring/etable/count_pair/CountPairFunction.hh>
#include <core/scoring/etable/count_pair/CountPairFactory.hh>
#include <core/scoring/etable/count_pair/types.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>

//#include <core/scoring/ContextGraphTypes.hh>

// Project headers
#include <core/pose/Pose.hh>
#include <ObjexxFCL/formatted.o.hh>

#include <core/scoring/trie/CPDataCorrespondence.hh>
#include <core/scoring/trie/RotamerDescriptor.hh>
#include <core/scoring/trie/RotamerTrie.hh>
#include <core/scoring/trie/TrieCollection.hh>
#include <core/scoring/trie/TrieCountPairBase.hh>
#include <core/scoring/trie/trie.functions.hh>

#include <core/scoring/etable/etrie/CountPairData_1_1.hh>
#include <core/scoring/etable/etrie/CountPairData_1_2.hh>
#include <core/scoring/etable/etrie/CountPairData_1_3.hh>
#include <core/scoring/etable/etrie/CountPairDataGeneric.hh>

#include <core/scoring/etable/etrie/TrieCountPair1BC4.hh>
#include <core/scoring/etable/etrie/TrieCountPairAll.hh>
#include <core/scoring/etable/etrie/TrieCountPairNone.hh>

#include <core/pack/rotamer_set/RotamerSet.hh>
#include <core/pack/rotamer_set/RotamerSetFactory.hh>
#include <core/pack/task/PackerTask.hh>

// Utility headers


// C++

/////////////////////////////////////////////////////////////////////////////////////////
///
/// Hacky (hence the name) implementation of 10r dielectric model, cutoff at 5.5A
///
///

//
//     alternatives: WARSHEL (from ligand.cc)
//     E = 322.0637*q1*q2/r/e(r)
//     if ( r < 3 ) e(r) = 16.55
//     else         e(r) = 1 + 60*(1-exp(-0.1*r))
//     Warshel, A. Russell, S. T., Q. Rev. Biophys., 1984, 17, 283-422
//
//
//     sigmoidal dielectric: (hingerty 1985)
//
//     e(r) = D - (D-D0)/2 [ (rS)**2 + 2rS + 2]exp(-rS)
//     with eg:
//     D = 78, D0 = 1, S = 0.3565 (rouzina&bloomfield)
//     D = 80, D0 = 4, S = 0.4 (rohs)

namespace core {
namespace scoring {
namespace hackelec {

////////////////////////////////////////////////////////////////////////////

Real const HackElecEnergy::max_dis( 5.5 );
Real const HackElecEnergy::max_dis2( max_dis * max_dis );
Real const HackElecEnergy::min_dis( 1.5 );
Real const HackElecEnergy::min_dis2( min_dis * min_dis );


////////////////////////////////////////////////////////////////////////////

inline
Real
HackElecEnergy::eval_dhack_elecE_dr_over_r(
	Real const dis2,
	Real const q1,
	Real const q2
) const
{
	static Real const C0( 322.0637 );
	static Real const die( 10.0 ); // 10r dielectric
	static Real const dEfac( -2.0 * C0 / die );

	if ( dis2 > max_dis2 ) return 0.0;
	else if ( dis2 < min_dis2 ) return 0.0; // flat in this region

	return dEfac * q1 * q2 / ( dis2 * dis2 );
}

////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////
HackElecEnergy::HackElecEnergy( methods::EnergyMethodOptions const & options ):
	parent( new HackElecEnergyCreator ),
	exclude_DNA_DNA_( options.exclude_DNA_DNA() )
{
	//add_score_type( hack_elec );
}


////////////////////////////////////////////////////////////////////////////
HackElecEnergy::HackElecEnergy( HackElecEnergy const & src ):
	parent( src ),
	exclude_DNA_DNA_( src.exclude_DNA_DNA_ )
{
	add_score_type( hack_elec );
}

/// clone
methods::EnergyMethodOP
HackElecEnergy::clone() const
{
	return new HackElecEnergy( *this );
}

///
void
HackElecEnergy::setup_for_derivatives( pose::Pose & pose, ScoreFunction const & ) const
{
	pose.update_residue_neighbors();
}

///
void
HackElecEnergy::setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const
{
	pose.update_residue_neighbors();
}


// The HackElectEnergy method stores a vector of rotamer trie objects in the Energies
// object for use in rapid rotamer/background energy calculations.  Overrides default
// do-nothing behavior.
void
HackElecEnergy::setup_for_packing(
	pose::Pose & pose,
	pack::task::PackerTask const &
) const
{
	using namespace pack::rotamer_set;
	using namespace trie;

	TrieCollectionOP tries = new TrieCollection;
	tries->total_residue( pose.total_residue() );
	for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
		// Do not compute energy for virtual residues.
		if (pose.residue(ii).type().name() == "VRT")
			continue;
		//iwd  Temporary hack: also skip ligand residues
		//if ( !pose.residue(ii).is_polymer() )
		//	continue;

		// something of a hack: create a residue set, put inside it a single residue,
		// and then use the existing trie creation code -- single residue tries are
		// constructed with the same logic as are multiple-residue tries.
		RotamerSetFactory rsf;
		RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( ii ) );
		one_rotamer_set->set_resid( ii ); // change the interface so that the rotset initializes its seqpos at creation
		one_rotamer_set->add_rotamer( pose.residue( ii ) );
		RotamerTrieBaseOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );
		tries->trie( ii, one_rotamer_trie );
	}
	pose.energies().data().set( util::HACKELEC_TRIE_COLLECTION, tries );

}

// @brief Creates a rotamer trie for the input set of rotamers and stores the trie
// in the rotamer set.
void
HackElecEnergy::prepare_rotamers_for_packing(
	pose::Pose const & pose,
	pack::rotamer_set::RotamerSet & set
) const
{

	trie::RotamerTrieBaseOP rottrie = create_rotamer_trie( set, pose );
	set.store_trie( methods::hackelec_method, rottrie );
}


// @brief Updates the cached rotamer trie for a residue if it has changed during the course of
// a repacking
void
HackElecEnergy::update_residue_for_packing(
	pose::Pose & pose,
	Size resid
) const
{
	using namespace pack::rotamer_set;
	using namespace trie;

	RotamerSetFactory rsf; // change RSF to have a static method create_rotamer_set
	RotamerSetOP one_rotamer_set = rsf.create_rotamer_set( pose.residue( resid ) );
	one_rotamer_set->set_resid( resid ); // change the interface so that the rotset initializes its seqpos at creation
	one_rotamer_set->add_rotamer( pose.residue( resid ) );
	trie::RotamerTrieBaseOP one_rotamer_trie = create_rotamer_trie( *one_rotamer_set, pose );

	// grab non-const & of the cached tries and replace resid's trie with a new one.
	TrieCollection & trie_collection
		( static_cast< TrieCollection & > (pose.energies().data().get( util::HACKELEC_TRIE_COLLECTION )));
	trie_collection.trie( resid, one_rotamer_trie );

}

/////////////////////////////////////////////////////////////////////////////
// scoring
/////////////////////////////////////////////////////////////////////////////

///
void
HackElecEnergy::residue_pair_energy(
	conformation::Residue const & rsd1,
	conformation::Residue const & rsd2,
	pose::Pose const &,
	ScoreFunction const &,
	TwoBodyEnergyMap & emap
) const
{
	using namespace etable::count_pair;

	Real score(0.0);

	if ( rsd1.seqpos() == rsd2.seqpos() ) {
		// pass
	} else if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) {
		//std::cout << "PHOSCHARGE: " << rsd1.atomic_charge(rsd1.atom_index("O1P")) << std::endl;
		// pass
	} else if ( rsd1.is_bonded( rsd2 ) || rsd1.is_pseudo_bonded( rsd2 ) ) {
		// assuming only a single bond right now -- generalizing to arbitrary topologies
		// also assuming crossover of 4, should be closest (?) to classic rosetta
		CountPairFunctionOP cpfxn =
			CountPairFactory::create_count_pair_function( rsd1, rsd2, CP_CROSSOVER_4 );

		for ( Size i=1, i_end = rsd1.natoms(); i<= i_end; ++i ) {
			Vector const & i_xyz( rsd1.xyz(i) );
			Real const i_charge( rsd1.atomic_charge(i) );
			if ( i_charge == 0.0 ) continue;
			for ( Size j=1, j_end = rsd2.natoms(); j<= j_end; ++j ) {
				Real const j_charge( rsd2.atomic_charge(j) );
				if ( j_charge == 0.0 ) continue;
				Real weight(1.0);
				if ( cpfxn->count( i, j, weight ) ) {
					score += weight *
						eval_atom_atom_hack_elecE( i_xyz, i_charge, rsd2.xyz(j), j_charge);
				}
			}
		}
	} else {
		// no countpair!
		for ( Size i=1, i_end = rsd1.natoms(); i<= i_end; ++i ) {
			Vector const & i_xyz( rsd1.xyz(i) );
			Real const i_charge( rsd1.atomic_charge(i) );
			if ( i_charge == 0.0 ) continue;
			for ( Size j=1, j_end = rsd2.natoms(); j<= j_end; ++j ) {
				Real const j_charge( rsd2.atomic_charge(j) );
				if ( j_charge == 0.0 ) continue;
				score += eval_atom_atom_hack_elecE( i_xyz, i_charge, rsd2.xyz(j), j_charge );
			}
		}
	}
	emap[ hack_elec ] = score;
	//std::cout << rsd1.seqpos() << ' ' << rsd2.seqpos() << ' ' << score << std::endl;
}

void
HackElecEnergy::evaluate_rotamer_pair_energies(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
) const
{
	assert( set1.resid() != set2.resid() );

	using namespace methods;
	using namespace trie;
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table1( energy_table );
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table2( energy_table );

	temp_table1 = 0; temp_table2 = 0;

	// save weight information so that its available during tvt execution
	hackelec_weight_ = weights[ hack_elec ];

	/// this will later retrieve a stored rotamer trie from inside the set;
	//EtableRotamerTrieBaseOP trie1 = create_rotamer_trie( set1, pose );
	//EtableRotamerTrieBaseOP trie2 = create_rotamer_trie( set2, pose );

	RotamerTrieBaseCOP trie1 = set1.get_trie( hackelec_method );
	RotamerTrieBaseCOP trie2 = set2.get_trie( hackelec_method );

	// figure out which trie countPairFunction needs to be used for this set
	TrieCountPairBaseOP cp = get_count_pair_function_trie( set1, set2, pose, sfxn );
	initialize_count_pair_data_trie( set1, set2, pose, trie1, trie2 );

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge (and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_trie( *trie2, *cp, *this, temp_table1, temp_table2 );

	/// add in the energies calculated by the tvt alg.
	energy_table += temp_table1;
	//std::cout << "FINISHED evaluate_rotamer_pair_energies" << std::endl;

	/*
	// There should be a way to turn this on without recompiling...
	// debug
	ObjexxFCL::FArray2D< pack::PackerEnergy > temp_table3( energy_table );
	temp_table3 = 0;
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set1.num_rotamers(); ii <= ii_end; ++ii ) {
		for ( Size jj = 1, jj_end = set2.num_rotamers(); jj <= jj_end; ++jj ) {
			emap.zero();
			residue_pair_energy( *set1.rotamer( ii ), *set2.rotamer( jj ), pose, sfxn, emap );
			temp_table3( jj, ii ) += weights.dot( emap );
			if ( std::abs( temp_table1( jj, ii ) - temp_table3( jj, ii )) > 0.001 ) {
				std::cout << "HackElecE: Residues " << set1.resid() << " & " << set2.resid() << " rotamers: " << ii << " & " << jj;
				std::cout << " tvt/reg discrepancy: tvt= " <<  temp_table1( jj, ii ) << " reg= " << temp_table3( jj, ii );
				std::cout << " delta: " << temp_table1( jj, ii ) - temp_table3( jj, ii ) << std::endl;
			}
		}
	}
	std::cout << "Finished RPE calcs for residues " << set1.resid() << " & " << set2.resid() << std::endl;
	*/


}

void
HackElecEnergy::evaluate_rotamer_background_energies(
	pack::rotamer_set::RotamerSet const & set,
	conformation::Residue const & residue,
	pose::Pose const & pose,
	ScoreFunction const & sfxn,
	EnergyMap const & weights,
	utility::vector1< pack::PackerEnergy > & energy_vector
) const
{
	//iwd  Temporary hack:  for ligands, call base class implementation
	//if ( !residue.is_polymer() ) {
	//	ShortRangeTwoBodyEnergy::evaluate_rotamer_background_energies(set, residue, pose, sfxn, weights, energy_vector);
	//	return;
	//}

	using namespace methods;
	using namespace trie;

	// allocate space for the trie-vs-trie algorithm
	utility::vector1< pack::PackerEnergy > temp_vector1( set.num_rotamers(), 0.0 );
	utility::vector1< pack::PackerEnergy > temp_vector2( set.num_rotamers(), 0.0 );

	// save weight information so that its available during tvt execution
	hackelec_weight_ = weights[ hack_elec ];

	RotamerTrieBaseCOP trie1 = set.get_trie( hackelec_method );
	RotamerTrieBaseCOP trie2 = ( static_cast< TrieCollection const & >
		( pose.energies().data().get( util::HACKELEC_TRIE_COLLECTION )) ).trie( residue.seqpos() );

	// figure out which trie countPairFunction needs to be used for this set
	TrieCountPairBaseOP cp = get_count_pair_function_trie( pose.residue( set.resid() ), residue, pose, sfxn );
	initialize_count_pair_data_trie( pose.residue( set.resid() ), residue, pose, trie1, trie2 );

	/// now execute the trie vs trie algorithm.
	/// this steps through three rounds of type resolution before finally arriving at the
	/// actual trie_vs_trie method.  The type resolution calls allow the trie-vs-trie algorithm
	/// to be templated with full type knowledge (and therefore be optimized by the compiler for
	/// each variation on the count pair data used and the count pair funtions invoked.
	trie1->trie_vs_path( *trie2, *cp, *this, temp_vector1, temp_vector2 );

	/// add in the energies calculated by the tvt alg.
	for ( Size ii = 1; ii <= set.num_rotamers(); ++ii ) {
		energy_vector[ ii ] += temp_vector1[ ii ];
	}
	//std::cout << "FINISHED evaluate_rotamer_background_energies" << std::endl;

	/*
	//debug
	utility::vector1< Energy > temp_vector3( energy_vector.size(), 0.0f );
	EnergyMap emap;
	for ( Size ii = 1, ii_end = set.num_rotamers(); ii <= ii_end; ++ii ) {
		emap.zero();
		residue_pair_energy( *set.rotamer( ii ), residue, pose, sfxn, emap );
		temp_vector3[ ii ] += weights.dot( emap );
		if ( std::abs( temp_vector1[ ii ] - temp_vector3[ ii ]) > 0.001 ) {
			std::cout << "HackElecE: Residues " << set.resid() << " & " << residue.seqpos() << " rotamers: " << ii << " & bg";
			std::cout << " tvt/reg discrepancy: tvt= " <<  temp_vector1[ ii ] << " reg= " << temp_vector3[ ii ];
			std::cout << " delta: " << temp_vector1[ ii ] - temp_vector3[ ii ] << std::endl;
		}
	}
	//std::cout << "Finished Rotamer BG calcs for residues " << set.resid() << " & " << residue.seqpos() << std::endl;
	*/

}


void
HackElecEnergy::eval_atom_derivative(
	id::AtomID const & atom_id,
	pose::Pose const & pose,
	kinematics::DomainMap const & domain_map,
	ScoreFunction const &,
	EnergyMap const & weights,
	Vector & F1,
	Vector & F2
 	) const
{
	using namespace etable::count_pair;

	// what is my charge?
	Size const pos1( atom_id.rsd() );
	Size const i   ( atom_id.atomno() );
	conformation::Residue const & rsd1( pose.residue( pos1 ) );
	Real const i_charge( rsd1.atomic_charge( i ) );
	int const pos1_map( domain_map( pos1 ) );
	bool const pos1_fixed( pos1_map != 0 );


	if ( i_charge == 0.0 ) return;

	Vector const & i_xyz( rsd1.xyz(i) );

	// cached energies object
	Energies const & energies( pose.energies() );

	// the neighbor/energy links
	EnergyGraph const & energy_graph( energies.energy_graph() );

// 	kinematics::DomainMap const & domain_map( energies.domain_map() );
// 	bool const pos1_fixed( !energies.res_moved( pos1 ) );
// 	assert( pos1_fixed == ( domain_map(pos1) != 0 ) ); // this is probably not generally true but I'm curious

	// loop over *all* nbrs of rsd1 (not just upper or lower)
	for ( graph::Graph::EdgeListConstIter
			iru  = energy_graph.get_node( pos1 )->const_edge_list_begin(),
			irue = energy_graph.get_node( pos1 )->const_edge_list_end();
			iru != irue; ++iru ) {
		Size const pos2( (*iru)->get_other_ind( pos1 ) );

		if ( pos1_fixed && pos1_map == domain_map( pos2 ) ) continue; // fixed wrt one another

		conformation::Residue const & rsd2( pose.residue( pos2 ) );

		assert( pos2 != pos1 );

		if ( exclude_DNA_DNA_ && rsd1.is_DNA() && rsd2.is_DNA() ) continue;
		assert( rsd1.seqpos() != rsd2.seqpos() );
		if ( rsd1.is_bonded( rsd2 ) || rsd1.is_pseudo_bonded( rsd2 )) {
			// generalizing to arbitrary topologies --  assuming crossover of 4
			CountPairFunctionOP cpfxn = CountPairFactory::create_count_pair_function( rsd1, rsd2, CP_CROSSOVER_4 );


			for ( Size j=1, j_end = rsd2.natoms(); j<= j_end; ++j ) {
				Real const j_charge( rsd2.atomic_charge(j) );
				if ( j_charge == 0.0 ) continue;
				Real weight(1.0);
				if ( cpfxn->count( i, j, weight ) ) {
					Vector const & j_xyz( rsd2.xyz(j) );
					Vector const f2( i_xyz - j_xyz );
					Real const dis2( f2.length_squared() );
					Real const dE_dr_over_r = weight * weights[ hack_elec ] *
						eval_dhack_elecE_dr_over_r( dis2, i_charge, j_charge );
					if ( dE_dr_over_r != 0.0 ) {
						Vector const f1( i_xyz.cross( j_xyz ) );
						F1 += dE_dr_over_r * f1;
						F2 += dE_dr_over_r * f2;
					}
				}
			}
		} else {
			// no countpair!
			for ( Size j=1, j_end = rsd2.natoms(); j<= j_end; ++j ) {
				Real const j_charge( rsd2.atomic_charge(j) );
				if ( j_charge == 0.0 ) continue;
				Vector const & j_xyz( rsd2.xyz(j) );
				Vector const f2( i_xyz - j_xyz );
				Real const dis2( f2.length_squared() );
				Real const dE_dr_over_r = weights[ hack_elec ] * eval_dhack_elecE_dr_over_r( dis2, i_charge, j_charge );
				if ( dE_dr_over_r != 0.0 ) {
					Vector const f1( i_xyz.cross( j_xyz ) );
					F1 += dE_dr_over_r * f1;
					F2 += dE_dr_over_r * f2;
				}
			}
		} // are rsd1 and rsd2 bonded?

	} // loop over nbrs of rsd1

}



/// @brief HackElecEnergy distance cutoff
Distance
HackElecEnergy::atomic_interaction_cutoff() const
{
	return max_dis;
}

/// @brief HackElecEnergy
void
HackElecEnergy::indicate_required_context_graphs( utility::vector1< bool > & /* context_graphs_required */ ) const
{
}

/// @brief create a rotamer trie for a particular set, deciding upon the kind of count pair data that
/// needs to be contained by the trie.
///
trie::RotamerTrieBaseOP
HackElecEnergy::create_rotamer_trie(
	pack::rotamer_set::RotamerSet const & rotset,
	pose::Pose const & // will be need to create tries for disulfides
) const
{
	using namespace trie;
	using namespace etable::etrie;

	CPDataCorrespondence cpdata_map( create_cpdata_correspondence_for_rotamerset( rotset ) );
	if ( cpdata_map.n_entries() == 1 || cpdata_map.n_entries() == 0 /* HACK! */ ) {
		ElecAtom at; CountPairData_1_1 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, hydrogen_interaction_cutoff()  );
	} else if ( cpdata_map.n_entries() == 2 ) {
		ElecAtom at; CountPairData_1_2 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, hydrogen_interaction_cutoff() );
	} else if ( cpdata_map.n_entries() == 3 ) {
		ElecAtom at; CountPairData_1_3 cpdat;
		return create_trie( rotset, at, cpdat, cpdata_map, hydrogen_interaction_cutoff() );
	} else {
		std::cerr << "Unsupported number of residue connections in trie construction." << std::endl;
		utility_exit();
		return 0;
	}
}

/// @brief figure out the trie count pair function to use
/// Need to refactor this so that the decision "what kind of count pair behavior should I use" can be decoupled
/// from class instantiation, and therefore shared between the creation of the trie count pair classes and the regular
/// count pair classes
trie::TrieCountPairBaseOP
HackElecEnergy::get_count_pair_function_trie(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const & pose,
	ScoreFunction const & sfxn
) const
{
	conformation::Residue const & res1( pose.residue( set1.resid() ) );
	conformation::Residue const & res2( pose.residue( set2.resid() ) );
	return get_count_pair_function_trie( res1, res2, pose, sfxn );
}


trie::TrieCountPairBaseOP
HackElecEnergy::get_count_pair_function_trie(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	pose::Pose const &,
	ScoreFunction const &
) const
{
	using namespace etable::count_pair;
	using namespace trie;
	using namespace etable::etrie;

	TrieCountPairBaseOP tcpfxn;
	if ( exclude_DNA_DNA_ && res1.is_DNA() && res2.is_DNA() ) {
		tcpfxn = new TrieCountPairNone();
		return tcpfxn;
	}

	/// code needs to be added here to deal with multiple bonds (and psuedubonds!) between residues,
	/// but ultimately, this code is incompatible with designing both disulfides and non-disulfies
	/// at the same residue...
	CPResidueConnectionType connection = res1.is_bonded(res2) ? CP_ONE_BOND : CP_NO_BONDS;
	if ( connection != CP_NO_BONDS )
	{
		tcpfxn = new TrieCountPair1BC4;
	} else {
		tcpfxn = new TrieCountPairAll;
	}
	return tcpfxn;

}

/// @brief figure out for this pair of rotamer tries which peice of
/// count pair data to use.
///
/// seems like this decision is intimately connected with that of deciding
/// which count pair object to use... maybe this initialization code belongs there.
void
HackElecEnergy::initialize_count_pair_data_trie(
	pack::rotamer_set::RotamerSet const & set1,
	pack::rotamer_set::RotamerSet const & set2,
	pose::Pose const &,
	trie::RotamerTrieBaseCOP const & trie1,
	trie::RotamerTrieBaseCOP const & trie2
) const
{
	trie1->point_count_pair_data_at_residue( set2.resid() );
	trie2->point_count_pair_data_at_residue( set1.resid() );
}


void
HackElecEnergy::initialize_count_pair_data_trie(
	conformation::Residue const & res1,
	conformation::Residue const & res2,
	pose::Pose const &, // I will need this guy if the pose ends up holding the chemical connetivity information
	trie::RotamerTrieBaseCOP const & trie1,
	trie::RotamerTrieBaseCOP const & trie2
) const
{
	trie1->point_count_pair_data_at_residue( res2.seqpos() );
	trie2->point_count_pair_data_at_residue( res1.seqpos() );
}

}
}
}
