// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/scoring/methods/HackElecEnergy.hh
/// @brief  Statistically derived rotamer pair potential class declaration
/// @author Phil Bradley


#ifndef INCLUDED_core_scoring_rna_HackElecEnergy_HH
#define INCLUDED_core_scoring_rna_HackElecEnergy_HH

// Unit Headers
#include <core/scoring/hackelec/HackElecEnergy.fwd.hh>

// Package headers
#include <core/scoring/hackelec/ElecAtom.hh>

// Project headers
#include <core/pose/Pose.fwd.hh>
#include <core/scoring/methods/ContextIndependentTwoBodyEnergy.hh>
#include <core/scoring/methods/EnergyMethodOptions.fwd.hh>
#include <core/scoring/trie/RotamerTrieBase.fwd.hh>
#include <core/scoring/trie/RotamerTrie.fwd.hh>
#include <core/scoring/trie/TrieCountPairBase.hh>

// Utility headers


namespace core {
namespace scoring {
namespace hackelec {

///
class HackElecEnergy : public methods::ContextIndependentTwoBodyEnergy {
public:

	///
	HackElecEnergy( methods::EnergyMethodOptions const & options );

	///
	HackElecEnergy( HackElecEnergy const & src );


	/// clone
	virtual
	methods::EnergyMethodOP
	clone() const;

	virtual
	void
	setup_for_derivatives( pose::Pose & pose, ScoreFunction const & ) const;

	virtual
	void
	setup_for_scoring( pose::Pose & pose, ScoreFunction const & ) const;

	///
	virtual
	void
	setup_for_packing( pose::Pose & pose, pack::task::PackerTask const & ) const;

	// Creates a rotamer trie for the input set of rotamers and stores the trie
	// in the rotamer set.
	virtual
	void
	prepare_rotamers_for_packing(
		pose::Pose const & pose,
		pack::rotamer_set::RotamerSet & set ) const;

	// Updates the cached rotamer trie for a residue if it has changed during the course of
	// a repacking
	virtual
	void
	update_residue_for_packing( pose::Pose & pose, Size resid ) const;

	/////////////////////////////////////////////////////////////////////////////
	// scoring
	/////////////////////////////////////////////////////////////////////////////

	virtual
	void
	residue_pair_energy(
		conformation::Residue const & rsd1,
		conformation::Residue const & rsd2,
		pose::Pose const & pose,
		ScoreFunction const &,
		TwoBodyEnergyMap & emap
	) const;

	// was private, but outside world wants to call this one:
	inline
	Real
	eval_atom_atom_hack_elecE(
		Vector const & i_xyz,
		Real const i_charge,
		Vector const & j_xyz,
		Real const j_charge
	) const;


	virtual
	void
	eval_intrares_energy(
		conformation::Residue const &,
		pose::Pose const &,
		ScoreFunction const &,
		EnergyMap &
	) const {}

	virtual
	void
	evaluate_rotamer_pair_energies(
		pack::rotamer_set::RotamerSet const & set1,
		pack::rotamer_set::RotamerSet const & set2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		ObjexxFCL::FArray2D< pack::PackerEnergy > & energy_table
	) const;


	//@brief overrides default rotamer/background energy calculation and uses
	// the trie-vs-trie algorithm instead
	virtual
	void
	evaluate_rotamer_background_energies(
		pack::rotamer_set::RotamerSet const & set,
		conformation::Residue const & residue,
		pose::Pose const & pose,
		ScoreFunction const & sfxn,
		EnergyMap const & weights,
		utility::vector1< pack::PackerEnergy > & energy_vector
	) const;


 	virtual
 	void
 	eval_atom_derivative(
 		id::AtomID const & atom_id,
 		pose::Pose const & pose,
		kinematics::DomainMap const & domain_map,
 		ScoreFunction const &,
 		EnergyMap const & weights,
 		Vector & F1,
 		Vector & F2
 	) const;

	virtual
	bool
	defines_intrares_energy( EnergyMap const & /*weights*/ ) const { return false; }

	virtual
	Distance
	atomic_interaction_cutoff() const;

	virtual
	void indicate_required_context_graphs( utility::vector1< bool > & context_graphs_required ) const;

public:
	 // Hooks for trie algorithms

	///  How close do two heavy atoms have to be such that their hydrogen atoms might interact?
	///  max heavy-to-hydrogen distance ( MAGIC NUMBER!!!! FIX IT ) + atom-pair interaction distance.
	Real
	hydrogen_interaction_cutoff2() const
	{
		return ( hydrogen_interaction_cutoff() )*( hydrogen_interaction_cutoff() );
	}

	Real
	hydrogen_interaction_cutoff() const
	{
		return ( max_dis + 2*1.35 );
	}

	inline
	Energy heavyatom_heavyatom_energy(
		ElecAtom const & at1,
		ElecAtom const & at2,
		DistanceSquared & d2
	) const
	{
		return hackelec_weight_ * eval_atom_atom_hack_elecE( at1.xyz(), at1.charge(), at2.xyz(), at2.charge(), d2  );
	}

	inline
	Energy heavyatom_hydrogenatom_energy(
		ElecAtom const & at1,
		ElecAtom const & at2
	) const
	{
		return hackelec_weight_ * eval_atom_atom_hack_elecE( at1.xyz(), at1.charge(), at2.xyz(), at2.charge()  );
	}

	inline
	Energy hydrogenatom_heavyatom_energy(
		ElecAtom const & at1,
		ElecAtom const & at2
	) const
	{
		return hackelec_weight_ * eval_atom_atom_hack_elecE( at1.xyz(), at1.charge(), at2.xyz(), at2.charge()  );
	}

	inline
	Energy hydrogenatom_hydrogenatom_energy(
		ElecAtom const & at1,
		ElecAtom const & at2
	) const
	{
		return hackelec_weight_ * eval_atom_atom_hack_elecE( at1.xyz(), at1.charge(), at2.xyz(), at2.charge()  );
	}

/// Private methods
private:
	inline
	Real
	eval_atom_atom_hack_elecE(
		Vector const & i_xyz,
		Real const i_charge,
		Vector const & j_xyz,
		Real const j_charge,
		DistanceSquared & d2
	) const;


	inline
	Real
	eval_dhack_elecE_dr_over_r(
		Real const dis2,
		Real const q1,
		Real const q2
	) const;


	trie::RotamerTrieBaseOP
	create_rotamer_trie(
		pack::rotamer_set::RotamerSet const & rotset,
		pose::Pose const & pose
	) const;

	trie::TrieCountPairBaseOP
	get_count_pair_function_trie(
		pack::rotamer_set::RotamerSet const & set1,
		pack::rotamer_set::RotamerSet const & set2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn
	) const;


	trie::TrieCountPairBaseOP
	get_count_pair_function_trie(
		conformation::Residue const & res1,
		conformation::Residue const & res2,
		pose::Pose const & pose,
		ScoreFunction const & sfxn
	) const;

	/// @brief figure out for this pair of rotamer tries which peice of
	/// count pair data to use.  This function should be refactored into trie.functions.hh
	void
	initialize_count_pair_data_trie(
		pack::rotamer_set::RotamerSet const & set1,
		pack::rotamer_set::RotamerSet const & set2,
		pose::Pose const &,
		trie::RotamerTrieBaseCOP const & trie1,
		trie::RotamerTrieBaseCOP const & trie2
	) const;


	void
	initialize_count_pair_data_trie(
		conformation::Residue const & res1,
		conformation::Residue const & res2,
		pose::Pose const &,
		trie::RotamerTrieBaseCOP const & trie1,
		trie::RotamerTrieBaseCOP const & trie2
	) const;

private:

	/////////////////////////////////////////////////////////////////////////////
	// data
	/////////////////////////////////////////////////////////////////////////////


	static Real const max_dis;
	static Real const max_dis2;
	static Real const min_dis;
	static Real const min_dis2;

	bool exclude_DNA_DNA_;

	mutable Real hackelec_weight_; // used during trie-vs-trie algorithm

};

inline
Real
HackElecEnergy::eval_atom_atom_hack_elecE(
	Vector const & i_xyz,
	Real const i_charge,
	Vector const & j_xyz,
	Real const j_charge
) const
{
	static Real const C0( 322.0637 );
	static Real const die( 10.0 ); // dielectric is 10r
	static Real const C1( C0 / die );
	static Real const C2( C1 / max_dis2 );

	Real d2( i_xyz.distance_squared( j_xyz ) );

	Real energy(0.0);
	if ( d2 <= max_dis2 ) {
		if ( d2 < min_dis2 ) d2 = min_dis2;
		energy = i_charge * j_charge * ( C1 / d2 - C2 );
	}
	return energy;
}

inline
Real
HackElecEnergy::eval_atom_atom_hack_elecE(
	Vector const & i_xyz,
	Real const i_charge,
	Vector const & j_xyz,
	Real const j_charge,
	DistanceSquared & d2
) const
{
	static Real const C0( 322.0637 );
	static Real const die( 10.0 ); // dielectric is 10r
	static Real const C1( C0 / die );
	static Real const C2( C1 / max_dis2 );

	d2 = i_xyz.distance_squared( j_xyz );

	Real energy(0.0);
	if ( d2 <= max_dis2 ) {
		if ( d2 < min_dis2 ) d2 = min_dis2;
		energy = i_charge * j_charge * ( C1 / d2 - C2 );
	}
	return energy;
}

}
}
}

#endif
