// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.org/.

/// @file   core/scoring/ScoreFunction.cc
/// @brief  ScoreFunction class definition.
/// @author Stuart G. Mentzer (Stuart_Mentzer@objexx.com)
/// @author Kevin P. Hinshaw (KevinHinshaw@gmail.com)
/// @author Modified by Sergey Lyskov


// Unit headers
#include <core/scoring/symmetry/SymmetricScoreFunction.hh>
#include <core/conformation/symmetry/SymmetricConformation.hh>

// Package headers
#include <core/scoring/EnergiesCacheableDataType.hh>
//#include <core/scoring/ScoringManager.hh>
#include <core/scoring/ScoreFunctionInfo.hh>

#include <core/scoring/methods/EnergyMethod.hh>
#include <core/scoring/methods/EnergyMethodOptions.hh>
#include <core/scoring/methods/ContextIndependentLRTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextIndependentTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentLRTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentTwoBodyEnergy.hh>
#include <core/scoring/methods/ContextIndependentOneBodyEnergy.hh>
#include <core/scoring/methods/ContextDependentOneBodyEnergy.hh>
#include <core/scoring/methods/TwoBodyEnergy.hh>
#include <core/scoring/methods/WholeStructureEnergy.hh>
#include <core/scoring/hbonds/hbonds.hh>
#include <core/scoring/LREnergyContainer.hh>

// // Project headers
#include <core/util/Tracer.hh>
#include <core/pose/Pose.hh>
#include <core/scoring/Energies.hh>
#include <core/util/prof.hh>

// Symmetry extras
#include <core/pack/rotamer_set/RotamerSet.hh> // TMP HACK
#include <core/scoring/hbonds/HBondSet.hh> //TMP HACK
#include <core/scoring/EnvPairPotential.hh>
#include <core/util/datacache/CacheableData.hh> //TMP HACK
//#include <core/scoring/symmetry/NBListCache.hh>
//#include <core/scoring/symmetry/NBListCache.fwd.hh>
#include <core/pose/datacache/CacheableDataType.hh>
#include <core/conformation/symmetry/util.hh>

namespace core {
namespace scoring {
namespace symmetry {

static util::Tracer TR("core.scoring.SymmetricScoreFunction");
///////////////////////////////////////////////////////////////////////////////
SymmetricScoreFunction::SymmetricScoreFunction():
	ScoreFunction()
	{}

///////////////////////////////////////////////////////////////////////////////
ScoreFunctionOP
SymmetricScoreFunction::clone() const
{
	return new SymmetricScoreFunction( *this );
}

///////////////////////////////////////////////////////////////////////////////
SymmetricScoreFunction &
SymmetricScoreFunction::operator=( SymmetricScoreFunction const & src )
{
	ScoreFunction::operator=( src );
	return *this;
}

///////////////////////////////////////////////////////////////////////////////
SymmetricScoreFunction::SymmetricScoreFunction( SymmetricScoreFunction const & src ):
	ScoreFunction( src )
{}

SymmetricScoreFunction::SymmetricScoreFunction( ScoreFunction const & src ):
  ScoreFunction( src )
{}

SymmetricScoreFunction::SymmetricScoreFunction( ScoreFunctionOP src ):
  ScoreFunction( *src )
{}

SymmetricScoreFunction::SymmetricScoreFunction( ScoreFunctionCOP src ):
  ScoreFunction( *src )
{}

///////////////////////////////////////////////////////////////////////////////

// to start out, just thinking fullatom energies
//
// NOTE: no freakin rotamer trials inside scoring!
Real
SymmetricScoreFunction::operator()( pose::Pose & pose ) const
{
	// This is tricky! If we are non-symmetric we want to use the regular score function
	// instead.
	if ( !core::conformation::symmetry::is_symmetric( pose ) ) {
		TR << "Warning!!! Using a symmetric score function on a non-symmetric pose" << std::endl;
		ScoreFunction asym_score( static_cast< ScoreFunction const & >( (*this) ) );
		return ( asym_score )(pose);
	}

	// completely unnecessary temporary hack to force refold if nec. for profiling
	pose.residue( pose.total_residue() );

	PROF_START( util::SCORE );
	//std::cout << "ScoreFunction::operator()\n";

	// notify the pose that we are starting a score evaluation.
	// also tells the cached energies object about the scoring
	// parameters, for the purposes of invalidating cached data
	// if necessary
	//
	// at this point the dof/xyz-moved information will be converted
	// to a domain map. Energy/neighbor links between pair-moved residues
	// will be deleted, and 1d energies of res-moved residues will be
	// cleared.
	//
	// further structure modification will be prevented until scoring is
	// completed
	//
	pose.scoring_begin( *this );
	//std::cout << "ScoreFunction::operator() 1\n";

	if ( pose.energies().total_energy() != 0.0 ) {
		std::cout << "STARTING SCORE NON-ZERO!" << std::endl;
	}

	// ensure that the total_energies are zeroed out -- this happens in Energies.scoring_begin()
	// unneccessary pose.energies().total_energies().clear();
	//std::cout << "ScoreFunction::operator() 2\n";

	// do any setup necessary
	setup_for_scoring( pose );
	//std::cout << "ScoreFunction::operator() 3\n";

	// Make some arrays symmetrical before scoring
	correct_arrays_for_symmetry( pose );


	// evaluate the residue-residue energies that only exist between
	// neighboring residues
	PROF_START( util::NEIGHBOR_ENERGIES );

	eval_twobody_neighbor_energies( pose );

	PROF_STOP ( util::NEIGHBOR_ENERGIES );

	// Ingemar, I put this in.... -Will
	PROF_START( util::LONG_RANGE_ENERGIES );
	eval_long_range_twobody_energies( pose );
	PROF_STOP ( util::LONG_RANGE_ENERGIES );

	// evaluate the onebody energies -- rama, dunbrack, ...
	eval_onebody_energies( pose );

	// give energyfunctions a chance update/finalize energies
	// etable nblist calculation is performed here
	for ( AllMethodsIterator iter=all_energies_begin(),
        iter_end= all_energies_end(); iter != iter_end; ++iter ) {
		(*iter)->finalize_total_energy( pose, *this, pose.energies().finalized_energies() );
	}

	correct_finalize_score( pose );

	pose.energies().total_energies() += pose.energies().finalized_energies();

	// dot the weights with the scores
	pose.energies().total_energy() = pose.energies().total_energies().dot( weights() );
	pose.energies().total_energies()[ total_score ] = pose.energies().total_energy();

	// notify that scoring is over
	pose.scoring_end( *this );

	PROF_STOP ( util::SCORE );

	return pose.energies().total_energy();
}

///////////////////////////////////////////////////////////////////////////////

void
SymmetricScoreFunction::eval_twobody_neighbor_energies(
	pose::Pose & pose
) const
{
	// cached energies object
	Energies & energies( pose.energies() );

	EnergyMap & total_energies( const_cast< EnergyMap & > ( energies.total_energies() ) );

	SymmetricConformation & SymmConf (
		dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	// the neighbor/energy links
	EnergyGraph & energy_graph( energies.energy_graph() );

	// are we using the atom-atom nblist?
	// renaming for true purpose: are we minimizing -- if so,
	// zero the energies stored on edges in the Energy graph, but do
	// not mark the edges as having had their energies computed
	bool const minimizing( energies.use_nblist() );

	TwoBodyEnergyMap tbemap;

	for ( Size i=1, i_end = pose.total_residue(); i<= i_end; ++i ) {
		conformation::Residue const & resl( pose.residue( i ) );
		for ( graph::Graph::EdgeListIter
				iru  = energy_graph.get_node(i)->upper_edge_list_begin(),
				irue = energy_graph.get_node(i)->upper_edge_list_end();
				iru != irue; ++iru ) {
			EnergyEdge & edge( static_cast< EnergyEdge & > (**iru) );

			Size const j( edge.get_second_node_ind() );
			conformation::Residue const & resu( pose.residue( j ) );
			// the pair energies cached in the link
			///TwoBodyEnergyMap & emap( energy_edge->energy_map() );
			tbemap.zero( cd_2b_types() );
			tbemap.zero( ci_2b_types() );

			// the context-dependent guys can't be cached, so they are always reevaluated
			eval_cd_2b( resl, resu, pose, tbemap );

				for ( Size ii = 1; ii <= cd_2b_types().size(); ++ii ) {
					tbemap[ cd_2b_types()[ ii ]] *= symm_info.score_multiply(i,j);
				}

			if ( edge.energies_not_yet_computed() ) {
				// energies not yet computed? <-> (during minimization w/ nblist) moving rsd pair
				/// TEMP zero portions;

				if ( minimizing ) {
					// confirm that this rsd-rsd interaction will be included
					// in the atom pairs on the nblist:
					//assert( ( pose.energies().domain_map(i) !=
					//					pose.energies().domain_map(j) ) ||
					//				( pose.energies().res_moved(i) ) );
					// ensure that these are zeroed, since we will hit them at the
					// end inside the nblist calculation
					// they almost certainly should be, since the energies have not
					// yet been computed...
					eval_ci_2b( resl, resu, pose, tbemap );
					for ( Size ii = 1; ii <= ci_2b_types().size(); ++ii ) {
						tbemap[ ci_2b_types()[ ii ]] *= symm_info.score_multiply(i,j);
					}

					edge.store_active_energies( tbemap );
					// do not mark energies as computed!!!!!!!!!!!!!
				} else {
					eval_ci_2b( resl, resu, pose, tbemap );
					for ( Size ii = 1; ii <= ci_2b_types().size(); ++ii ) {
						tbemap[ ci_2b_types()[ ii ]] *= symm_info.score_multiply(i,j);
					}

					edge.store_active_energies( tbemap );
					edge.mark_energies_computed();
				}
			} else {
				/// Read the CI energies from the edge, as they are still valid;
				for ( Size ii = 1; ii <= ci_2b_types().size(); ++ii ) {
					tbemap[ ci_2b_types()[ ii ]] = edge[ ci_2b_types()[ ii ] ];
				}

				/// Save the freshly computed CD energies on the edge
				edge.store_active_energies( tbemap, cd_2b_types() );
			}

			total_energies.accumulate( tbemap, ci_2b_types() );
			total_energies.accumulate( tbemap, cd_2b_types() );
		} // nbrs of i
	} // i=1,nres

}

void
SymmetricScoreFunction::eval_long_range_twobody_energies( pose::Pose & pose ) const
{

	// find SymmInfo
	SymmetricConformation & SymmConf (
		dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
	SymmetryInfo & symm_info( SymmConf.Symmetry_Info() );

	EnergyMap & total_energies(const_cast< EnergyMap & > (pose.energies().total_energies()));

	for ( CI_LR_2B_MethodIterator iter=ci_lr_2b_methods_begin(),
			iter_end = ci_lr_2b_methods_end(); iter != iter_end; ++iter ) {

		LREnergyContainerOP lrec = pose.energies().nonconst_long_range_container( (*iter)->long_range_type() );
		if ( !lrec || lrec->empty() ) continue; // only score non-emtpy energies.

		// Potentially O(N^2) operation...
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			for ( ResidueNeighborIteratorOP
					rni = lrec->upper_neighbor_iterator_begin( ii ),
					rniend = lrec->upper_neighbor_iterator_end( ii );
					(*rni) != (*rniend); ++(*rni) ) {
				EnergyMap emap;
				if ( ! rni->energy_computed() ) {
					Size jj = rni->upper_neighbor_id();
					(*iter)->residue_pair_energy( pose.residue(ii), pose.residue( jj ), pose, *this, emap );
					emap *= symm_info.score_multiply( ii, jj );
					rni->save_energy( emap );

					/// DANGER DANGER DANGER.  use_nblist() now means "In the process of a minimization". There is
					/// no such thing as a non-neighborlist minimization.  This will confuse people at some point.
					/// We ought to have some "are we minimizing currently" flag who's meaning can be decoupled
					/// from the neighborlist idea.
					if ( ! pose.energies().use_nblist() ) {
						// Do we need to do this symmetrically?
						rni->mark_energy_computed();
					}
				} else {
					rni->retrieve_energy( emap ); // pbmod
				}
				total_energies += emap;
			}
		}
	}

	//fpd CD LR methods should always be computed
	for ( CD_LR_2B_MethodIterator iter = cd_lr_2b_methods_begin(),
			iter_end = cd_lr_2b_methods_end(); iter != iter_end; ++iter ) {

		LREnergyContainerOP lrec
			= pose.energies().nonconst_long_range_container( (*iter)->long_range_type() );

		// Potentially O(N^2) operation...
		for ( Size ii = 1; ii <= pose.total_residue(); ++ii ) {
			for ( ResidueNeighborIteratorOP
					rni = lrec->upper_neighbor_iterator_begin( ii ),
					rniend = lrec->upper_neighbor_iterator_end( ii );
					(*rni) != (*rniend); ++(*rni) ) {

				EnergyMap emap;
				Size jj = rni->upper_neighbor_id();
				(*iter)->residue_pair_energy( pose.residue(ii), pose.residue(jj), pose, *this, emap );
				emap *= symm_info.score_multiply( ii, jj );

				rni->save_energy( emap );
				//rni->mark_energy_computed();

				total_energies += emap;

			}
		}
	}
}


///////////////////////////////////////////////////////////////////////////////
void
SymmetricScoreFunction::eval_onebody_energies( pose::Pose & pose ) const
{
	// context independent onebody energies
	Energies & energies( pose.energies() );
	EnergyMap & totals( energies.total_energies() );

	 // find SymmInfo
	SymmetricConformation & SymmConf (
		dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
	SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	for ( Size i=1; i<= pose.total_residue(); ++i ) {
		if ( !symm_info.fa_is_independent(i) ||
				i > symm_info.num_total_residues_without_pseudo() ) continue;

		EnergyMap & emap( energies.onebody_energies( i ) );

		// 1body intxns ///////////////////////////
		if ( energies.res_moved( i ) ) {
			// have to recalculate
			emap.clear(); // should already have been done when the domain map was internalized
			eval_ci_1b( pose.residue(i), emap );
			emap *= symm_info.score_multiply(i,i);
		}

		emap.zero( cd_1b_types() ); // cant be cached
		EnergyMap cd_1b_emap;
		eval_cd_1b( pose.residue(i), pose, cd_1b_emap );
		cd_1b_emap *= symm_info.score_multiply_factor();
		emap += cd_1b_emap;

		// 2body energy methods are allowed to define 1body intxns ///////////////////////////
		if ( any_intrares_energies() ) {
			// context independent:
			if ( energies.res_moved( i ) ) {
				EnergyMap ci_intrares_emap;
				eval_ci_intrares_energy( pose.residue(i), pose, ci_intrares_emap );
				ci_intrares_emap *= symm_info.score_multiply_factor();
				emap += ci_intrares_emap;
			}
			// context dependent
			emap.zero( cd_2b_types() ); // cant be cached (here only intrares are relevant)
			emap.zero( cd_lr_2b_types() ); // cant be cached (here only intrares are relevant)
			EnergyMap cd_2b_emap = emap;
			eval_cd_intrares_energy( pose.residue(i), pose, emap );
			cd_2b_emap -= emap;
			cd_2b_emap *= -1.0;
			cd_2b_emap *= symm_info.score_multiply_factor();
		}

		totals += emap;

		energies.reset_res_moved( i ); // mark one body energies as having been calculated
		for ( std::vector< Size>::const_iterator
			clone     = symm_info.bb_clones( i ).begin(),
      clone_end = symm_info.bb_clones( i ).end();
      clone != clone_end; ++clone ){
				energies.reset_res_moved( *clone );
		}
		//std::cout << "totals: "<<  i  << totals;
	}
}

void
SymmetricScoreFunction::intersubunit_hbond_energy(
                        pose::Pose & pose,
                        EnergyMap & intersubunit_energy
) const
{

	using EnergiesCacheableDataType::HBOND_SET;

	hbonds::HBondSet const & hbond_set
    ( static_cast< hbonds::HBondSet const & >
      ( pose.energies().data().get( HBOND_SET )));

  SymmetricConformation & SymmConf (
        dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );


  for ( Size i = 1; i <= hbond_set.nhbonds(); ++i ) {
    hbonds::HBond const & hbond( hbond_set.hbond(i) );

	Real sr_bb_hbenergy = 0.0;
  Real lr_bb_hbenergy = 0.0;
  Real bb_sc_hbenergy = 0.0;
  Real sc_hbenergy = 0.0;

			switch ( get_hbond_weight_type( hbond.eval_type() ) ) {
      case hbonds::hbw_SR_BB:
        sr_bb_hbenergy = hbond.energy() * hbond.weight();
        break;
      case hbonds::hbw_LR_BB:
        lr_bb_hbenergy = hbond.energy() * hbond.weight();
        break;
      case hbonds::hbw_BB_SC:
        bb_sc_hbenergy = hbond.energy() * hbond.weight();
        break;
      case hbonds::hbw_SC:
        sc_hbenergy = hbond.energy() * hbond.weight();
        break;
      default:
        std::cout << "Warning: energy from unexpected HB type ignored " <<
          hbond.eval_type() << std::endl;
        break;
      }
			Size factor (0);
	    if ( !( symm_info.fa_is_independent( hbond.don_res() ) &&
				 symm_info.fa_is_independent( hbond.acc_res() )) ) {
				factor = symm_info.score_multiply_factor();
			}

			//if ( symm_info.fa_is_independent( hbond.don_res() ) ) {
				factor = symm_info.score_multiply( hbond.don_res() , hbond.acc_res() );
			//} else {
			//	factor = symm_info.score_multiply( hbond.acc_res() );
			//}

			if ( symm_info.bb_follows( hbond.acc_res() ) == hbond.don_res() ||
         symm_info.bb_follows( hbond.don_res() ) == hbond.acc_res() ) {
				factor = symm_info.score_multiply_factor() - 1;
			}
			intersubunit_energy[ hbond_lr_bb ] += factor*lr_bb_hbenergy;
			intersubunit_energy[ hbond_sr_bb ] += factor*sr_bb_hbenergy;
	}
}

void
SymmetricScoreFunction::symmetrical_allow_hbonds( pose::Pose & pose ) const
{

	using EnergiesCacheableDataType::HBOND_SET;

	hbonds::HBondSetOP hbond_set
			( static_cast< hbonds::HBondSet & >
				( pose.energies().data().get( HBOND_SET )));

	SymmetricConformation & SymmConf (
        dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

  for ( Size i = 1; i <= hbond_set->nhbonds(); ++i ) {
		hbonds::HBond const & hbond( hbond_set->hbond(i) );
			Size acc( hbond.acc_res() );
			Size don( hbond.don_res() );
			if ( symm_info.fa_is_independent( acc ) ) {
				for ( Size clone = 1; clone <= symm_info.num_interfaces(); ++clone ) {
						Size clone_acc = symm_info.bb_clones( acc )[clone];
						hbond_set->set_backbone_backbone_acceptor( clone_acc,
												hbond_set->acc_bbg_in_bb_bb_hbond( acc )  );
				}
			}
			if ( symm_info.fa_is_independent( don ) ) {
        for (Size clone = 1; clone <= symm_info.num_interfaces(); ++clone ) {
            Size clone_don = symm_info.bb_clones( don )[clone];
            hbond_set->set_backbone_backbone_acceptor( clone_don,
                        hbond_set->don_bbg_in_bb_bb_hbond( don )  );
      }
		}
	}
	pose.energies().data().set( HBOND_SET, hbond_set );
}

void
SymmetricScoreFunction::set_symmetric_residue_neighbors_hbonds( pose::Pose & pose ) const
{

	using EnergiesCacheableDataType::HBOND_SET;

	hbonds::HBondSetOP hbond_set
		( static_cast< hbonds::HBondSet &  >
     ( pose.energies().data().get( HBOND_SET )));

  SymmetricConformation & SymmConf (
        dynamic_cast<SymmetricConformation &> ( pose.conformation()) );

  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	  for ( uint res = 1; res <= pose.total_residue(); ++res ) {
    if ( symm_info.get_use_symmetry() ) {
      if ( !symm_info.fa_is_independent( res ) ) {
        int symm_res ( symm_info.bb_follows( res ) );
				int neighbors_symm ( hbond_set->nbrs( symm_res ) );
				hbond_set->set_nbrs( res, neighbors_symm );
			}
		}
  }
	pose.energies().data().set( HBOND_SET, hbond_set );
}

void
SymmetricScoreFunction::set_symmetric_cenlist( pose::Pose & pose ) const
{
//	EnvPairPotential pairpot;
//	pairpot.compute_centroid_environment_symmetric( pose );

	using core::pose::datacache::CacheableDataType::CEN_LIST_INFO;

	CenListInfoOP cenlist( *( static_cast< CenListInfo * >( pose.data().get_ptr( CEN_LIST_INFO )() )));

  SymmetricConformation & SymmConf (
        dynamic_cast<SymmetricConformation &> ( pose.conformation()) );

  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );

	  for ( uint res = 1; res <= pose.total_residue(); ++res ) {
    if ( symm_info.get_use_symmetry() ) {
      if ( !symm_info.fa_is_independent( res ) ) {
        int symm_res ( symm_info.bb_follows( res ) );
				double fcen6_symm ( cenlist->fcen6( symm_res) );
				double fcen10_symm ( cenlist->fcen10( symm_res ) );
				double fcen12_symm ( cenlist->fcen12( symm_res ) );
				cenlist->set_fcen6( res, fcen6_symm );
				cenlist->set_fcen10( res, fcen10_symm );
				cenlist->set_fcen12( res, fcen12_symm );
			}
		}
  }
	pose.data().set( CEN_LIST_INFO, cenlist );
}

void
SymmetricScoreFunction::correct_arrays_for_symmetry( pose::Pose & pose ) const
{
	using core::pose::datacache::CacheableDataType::CEN_LIST_INFO;

	if ( pose.data().has( CEN_LIST_INFO ) ) {
		set_symmetric_cenlist( pose );
	}

	if ( has_nonzero_weight( hbond_lr_bb ) || has_nonzero_weight( hbond_sr_bb )  ||
			 has_nonzero_weight( hbond_bb_sc ) || has_nonzero_weight( hbond_sc ) ) {
		symmetrical_allow_hbonds( pose );
		set_symmetric_residue_neighbors_hbonds( pose );
	}

}


// energy methods that compute scores in 'finalize_total_energy' (incl all whole structure energies)
//    should not be scaled by # subunits
// this method deals with these methods
void
SymmetricScoreFunction::correct_finalize_score( pose::Pose & pose ) const
{
	EnergyMap delta_energy( pose.energies().total_energies() );
	EnergyMap interface_energies, etable_energy;

	SymmetricConformation & SymmConf (
        dynamic_cast<SymmetricConformation &> ( pose.conformation()) );
  SymmetryInfo const & symm_info( SymmConf.Symmetry_Info() );
  Size const factor ( symm_info.score_multiply_factor() - 1 );

	interface_energies[ hbond_lr_bb ] = pose.energies().finalized_energies()[hbond_lr_bb];
	interface_energies[ hbond_sr_bb ] = pose.energies().finalized_energies()[hbond_sr_bb];
	interface_energies[ interchain_contact ] = pose.energies().finalized_energies()[interchain_contact];

	if ( pose.energies().use_nblist()  ) {
		etable_energy[fa_atr] = pose.energies().finalized_energies()[fa_atr];
		etable_energy[fa_rep] = pose.energies().finalized_energies()[fa_rep];
		etable_energy[fa_sol] = pose.energies().finalized_energies()[fa_sol];
		etable_energy[fa_intra_rep] = pose.energies().finalized_energies()[fa_intra_rep];
		etable_energy *= factor;
	}

	delta_energy = pose.energies().finalized_energies();
	delta_energy -= interface_energies;
	delta_energy *= factor;
	delta_energy -= etable_energy;

	if ( has_nonzero_weight( hbond_lr_bb ) || has_nonzero_weight( hbond_sr_bb ) ) {
		EnergyMap new_interface_energy;
		intersubunit_hbond_energy(pose,new_interface_energy);
		delta_energy += new_interface_energy;
	}

	if ( has_nonzero_weight( interchain_contact ) ) {
		EnergyMap interchain_contact_energy;
		interchain_contact_energy[interchain_contact] = pose.energies().finalized_energies()[interchain_contact];
		delta_energy += interchain_contact_energy;
	}

	// whole structure energy correction
	for ( WS_MethodIterator iter=ws_methods_begin(),
			iter_end = ws_methods_end(); iter != iter_end; ++iter ) {
		ScoreTypes type_i = (*iter)->score_types();
		for ( Size j=1; j<=type_i.size(); ++j) {
			EnergyMap deldel_energy;
			deldel_energy[ type_i[j] ] = pose.energies().finalized_energies()[ type_i[j] ]*factor;
			delta_energy -= deldel_energy;
		}
	}

// 	if ( has_nonzero_weight( rg ) ) {
// 		EnergyMap rg_energy;
// 		rg_energy[ rg ] = pose.energies().finalized_energies()[rg]*factor;
// 		delta_energy -= rg_energy;
// 	}
//
// 	// fpd
// 	if ( has_nonzero_weight( chainbreak ) ) {
// 		EnergyMap cb_energy;
// 		cb_energy[ chainbreak ] = pose.energies().finalized_energies()[rg]*factor;
// 		delta_energy -= rg_energy;
// 	}
//
// 	// fpd
// 	if ( has_nonzero_weight( linear_chainbreak ) ) {
// 		EnergyMap lincb_energy;
// 		lincb_energy[ linear_chainbreak ] = pose.energies().finalized_energies()[linear_chainbreak ]*factor;
// 		delta_energy -= lincb_energy;
// 	}
//
// 	// fpd
// 	if ( has_nonzero_weight( overlap_chainbreak ) ) {
// 		EnergyMap ovrcb_energy;
// 		ovrcb_energy[ rg ] = pose.energies().finalized_energies()[rg]*factor;
// 		delta_energy -= ovrcb_energy;
// 	}


	pose.energies().finalized_energies() -= interface_energies;
	pose.energies().finalized_energies() += delta_energy;

}

} // symmetry
} // namespace scoring
} // namespace core
