// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file core/sequence/MultipleSequenceAlignment.cc
/// @brief
/// @author

#include <core/types.hh>
#include <core/sequence/MultipleSequenceAlignment.hh>

#include <numeric/random/random.hh>

#include <utility/vector1.hh>
#include <utility/io/izstream.hh>

namespace core {
namespace sequence {

static core::util::Tracer TR("core.sequence.multiplesequencealignment");
using core::Size;

void MultipleSequenceAlignment::add_sequence( Sequence myseq ) {

	if ( sequences_.size() < 1 ) {
		sequences_.push_back( myseq );
		return;
	}

	if ( myseq.length() < sequences_[1].length() ) {
		core::Size ngaps = sequences_[1].length() - myseq.length();
		// std::cerr << "inserting " << ngaps << " gaps into sequence " << std::endl;
		// std::cerr << myseq << std::endl;
		for ( core::Size i = 1; i <= ngaps; ++i ) {
			// std::cerr << "inserting gap at position " << myseq.size() + 1 << std::endl;
			// std::cerr << myseq << std::endl;
			core::Size random_pos = numeric::random::random_range( 0, (int) myseq.length() );
			myseq.insert_gap( random_pos );
			// std::cerr << myseq << std::endl;
		}
	} else if ( myseq.length() > sequences_[1].length() ) {
		core::Size ngaps = myseq.length() - sequences_[1].length();
		// std::cerr << "inserting " << ngaps << " gaps." << std::endl;
		for ( core::Size s = 1; s <= size(); ++s ) {
			for ( core::Size i = 1; i <= ngaps; ++i ) {
				core::Size random_pos = numeric::random::random_range( 0, (int) sequences_[s].length() );
				sequences_[s].insert_gap( random_pos );
				// std::cerr << sequences_[s];
			}
		}
	}
	sequences_.push_back( myseq );

} // add_sequence

core::Size MultipleSequenceAlignment::size() const {
	return sequences_.size();
}

void MultipleSequenceAlignment::insert_gap_into_sequence( core::Size sequence_index, core::Size position ) {
	runtime_assert( sequence_index <= size() );

	sequences_[sequence_index].insert_gap( position );
	// add a gap onto the end of all of the other sequences.
	for ( core::Size i = 1; i <= size(); ++i ) {
		core::Size random_pos = numeric::random::random_range( 0, (int) sequences_[i].length() );
		if ( i != sequence_index ) sequences_[i].insert_gap( random_pos );
	}

	remove_gapped_columns();
} // insert_gap_into_sequence

void MultipleSequenceAlignment::remove_gapped_columns() {

	for ( core::Size position = 1; position <= sequences_[1].length(); ++position ) {
		bool delete_column( true ); // delete this new position if the entire column is gapped.
		for ( utility::vector1< Sequence >::iterator it = sequences_.begin(), end = sequences_.end();
					it != end; ++it
		) {
			if ( !it->is_gap(position) ) {
				// std::cout << "not deleting column because sequence " << *it
				// 					<< " has no gap at position " << position << std::endl;
				delete_column = false;
			}
		}

		if ( delete_column ) {
			// delete this column from the alignment.
			for ( utility::vector1< Sequence >::iterator it = sequences_.begin(), end = sequences_.end();
						it != end; ++it
			) {
				// std::cout << "deleting column " << position << " from " << *it;
				it->delete_position( position );
			}
		}

		// std::cout << *this << std::endl;
	} // for positions
} // remove_gapped_columns

void MultipleSequenceAlignment::alignment_from_file(
	std::string const & filename
) {
	utility::io::izstream data( filename.c_str() );
	if ( !data ) {
		TR.Error << "ERROR:: Unable to sequence alignment file " << filename << std::endl;
		return;
	}

	core::Size start;
	std::string line, sequence, name;
	while ( getline(data,line) ) {
		if ( line == "" ) continue; // skip blank lines
		std::istringstream line_stream( line );
		line_stream >> name >> start >> sequence;
		Sequence seq( sequence, name, start );
		add_sequence( seq );
	} // while( getline(data,line) )
} // alignment_from_file

} // sequence
} // core

