// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file core/sequence/ScoringSchemeFactory.cc
/// @brief Factory for creating various types of ScoringSchemes for use
/// in sequence alignment.
/// @author James Thompson

// Unit headers
#include <core/sequence/ScoringScheme.hh>
#include <core/sequence/ScoringSchemeFactory.hh>

// Package headers
#include <core/sequence/SimpleScoringScheme.hh>
#include <core/sequence/MatrixScoringScheme.hh>
#include <core/sequence/DPScoringScheme.hh>
#include <core/sequence/L1ScoringScheme.hh>
#include <core/sequence/ProfSimScoringScheme.hh>
#include <core/sequence/CompassScoringScheme.hh>
#include <core/sequence/CompositeScoringScheme.hh>

#include <utility/exit.hh>

using namespace core::sequence;

namespace core {
namespace sequence {

void
ScoringSchemeFactory::add_type( ScoringSchemeOP new_scheme ) {
	std::string const type_name( new_scheme->type() );
	scheme_types_[ type_name ] = new_scheme;
}

ScoringSchemeOP
ScoringSchemeFactory::get_scoring_scheme(
	std::string const & type
) const {
	ScoringSchemeTypes::const_iterator iter = scheme_types_.find( type );
	if ( iter != scheme_types_.end() ) {
		return iter->second->clone();
	} else {
		utility_exit_with_message(
			"ScoringSchemeFactory: unknown ScoringScheme type: " + type
		);
		return NULL;
	}
}

ScoringSchemeFactory::ScoringSchemeFactory(void) {
	// initialization of ScoringSchemes which this factory knows how to
	// instantiate
	add_type( new SimpleScoringScheme() );
	add_type( new MatrixScoringScheme() );
	add_type( new L1ScoringScheme() );
	add_type( new DPScoringScheme() );
	add_type( new ProfSimScoringScheme() );
	add_type( new CompassScoringScheme() );
	add_type( new CompositeScoringScheme() );
}

} // namespace sequence
} // namespace core
