// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file   core/util/datacache/cacheable_observers.cc
/// @brief  a bunch of cacheable observers
///
/// @author Florian Richter ( floric@u.washington.edu)

// Unit headers
#include <core/util/datacache/cacheable_observers.hh>

// Package headers
#include <core/util/datacache/BasicDataCache.hh>


// Project headers
#include <core/conformation/Conformation.hh>
#include <core/conformation/signals/LengthEvent.hh>
#include <core/pose/Pose.hh>


namespace core {
namespace util {
namespace datacache {


/// @brief default constructor
LengthEventCollector::LengthEventCollector() :
	CacheableObserver()
{
	length_events_.clear();
}


/// @brief copy constructor
LengthEventCollector::LengthEventCollector( LengthEventCollector const & rval ) :
	CacheableObserver( rval )
{
	copy_length_events( rval.length_events_ );
}

LengthEventCollector::~LengthEventCollector() {
	detach_from();
}


/// @brief copy assignment
LengthEventCollector &
LengthEventCollector::operator =( LengthEventCollector const & rval ) {
	if ( this != &rval ) {
		CacheableObserver::operator =( rval );

		copy_length_events( rval.length_events_ );
	}

	return *this;
}

utility::vector1< core::conformation::signals::LengthEvent > const &
LengthEventCollector::events(){
	return length_events_;
}


pose::datacache::CacheableObserverOP
LengthEventCollector::create()
{
	return new LengthEventCollector();
}

pose::datacache::CacheableObserverOP
LengthEventCollector::clone()
{
	return new LengthEventCollector( *this );
}


void
LengthEventCollector::copy_length_events(
	utility::vector1< core::conformation::signals::LengthEvent > const & events
){

	using namespace core::conformation::signals;
	length_events_.clear();

	for( utility::vector1< LengthEvent >::const_iterator len_it = events.begin();
			 len_it != events.end(); ++len_it ){

		length_events_.push_back( LengthEvent( *len_it ) );
	}

}

/// @details all this class does is keep track of length events
void
LengthEventCollector::on_length_change( conformation::signals::LengthEvent const & event ){

	length_events_.push_back( conformation::signals::LengthEvent( event ) );
}

void
LengthEventCollector::attach_impl( pose::Pose & pose ){

	length_event_link_ = pose.conformation().attach_length_obs( &LengthEventCollector::on_length_change, this );

}

void
LengthEventCollector::detach_impl(){

	length_event_link_.invalidate();

}


} // namespace datacache
} // namespace pose
} // namespace core
