// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

#include <core/util/prof.hh>
#include <core/util/Tracer.hh>
#include <ObjexxFCL/ObjexxFCL.hh>
#include <ObjexxFCL/formatted.io.hh>
#include <ObjexxFCL/string.functions.hh>
#include <string>

// Commented by inclean daemon #include <cstdlib>
// Commented by inclean daemon #include <cstdio>
// Commented by inclean daemon #include <iostream>

namespace core {
namespace util {

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

utility::vector1< std::string > tag2string;

utility::vector1< clock_t > start_clock( n_prof_tags, 0 );
utility::vector1< double > total_clock( n_prof_tags, 0 );
utility::vector1< int > calls( n_prof_tags, 0 );
utility::vector1< int > bad_calls( n_prof_tags, 0 );

clock_t const SHRINK_FACTOR( 2 );

double const clock_factor( ( (double) SHRINK_FACTOR * 100.0 ) /
													 CLOCKS_PER_SEC );

//bool profiling( true );


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
setup_tag2string()
{
	tag2string.clear();
	for ( int i=1; i<= n_prof_tags; ++i ) {
		tag2string.push_back( ObjexxFCL::string_of(i) );
	}

	// now fill in
	tag2string[ TEST1 ] = "CONFORMATION_COPY";
	tag2string[ TEST2 ] = "FIND_NEIGHBORS";
	tag2string[ TEST3 ] = "COMPUTE_GUNN";
	tag2string[ TEST4 ] = "SMOOTH_MOVE";
	tag2string[ ATOM_TREE_UPDATE_INTERNAL_COORDS ] = "ATOM_TREE_UPDATE_INTERNAL_COORDS";
	tag2string[ ATOM_TREE_UPDATE_XYZ_COORDS ] = "ATOM_TREE_UPDATE_XYZ_COORDS";
	tag2string[ ROTAMER_TRIALS ] = "ROTAMER_TRIALS";
	tag2string[ PACK_ROTAMERS ] = "PACK_ROTAMERS";
	tag2string[ UPDATE_RESIDUE_NEIGHBORS ] = "UPDATE_RESIDUE_NEIGHBORS";
	tag2string[ UPDATE_RESIDUE_TORSIONS ] = "UPDATE_RESIDUE_TORSIONS";
	tag2string[ UPDATE_RESIDUE_COORDINATES ] = "UPDATE_RESIDUE_COORDINATES";
	tag2string[ SETUP_NBLIST ] = "SETUP_NBLIST";
	tag2string[ SCORE ] = "SCORE";
	tag2string[ POSE_COPY ] = "POSE_COPY";
	tag2string[ ENERGY_GRAPH_COPY ] = "ENERGY_GRAPH_COPY";
	tag2string[ CONSTRAINT_SCORE ] = "CONSTRAINT_SCORE";
	tag2string[ CCD_CLOSE ] = "CCD_CLOSE";
	tag2string[ FUNC ] = "FUNC";
	tag2string[ DFUNC ] = "DFUNC";
	tag2string[ GET_ENERGIES ] = "GET_ENERGIES";
	tag2string[ SIMANNEALING ] = "SIMANNEALING";
	tag2string[ MC_ACCEPT ] = "MC_ACCEPT";
	tag2string[ INSERT_FRAGS ] = "INSERT_FRAGS";
	tag2string[ GB_GET_ALL_BORN_RADII ] = "GB_GET_ALL_BORN_RADII";
	tag2string[ GB_SETUP_FOR_PACKING ] = "GB_SETUP_FOR_PACKING";
	tag2string[ GEN_BORN_ROTAMER_PAIR_ENERGIES ] = "GEN_BORN_ROTAMER_PAIR_ENERGIES";
	tag2string[ GEN_BORN_ROTAMER_BACKGROUND_ENERGIES ] = "GEN_BORN_ROTAMER_BACKGROUND_ENERGIES";
	tag2string[ NEIGHBOR_ENERGIES ] = "NEIGHBOR_ENERGIES";
	tag2string[ LONG_RANGE_ENERGIES ] = "LONG_RANGE_ENERGIES";
	tag2string[ MINMOVER_APPLY ] = "MINMOVER_APPLY";
	tag2string[ BACKRUB_MOVER ] = "BACKRUB_MOVER";
	tag2string[ FIND_SUGAR_AND_SUITE_FRAGS_I ] = "FIND_SUGAR_AND_SUITE_FRAGS_I";
	tag2string[ FIND_SUGAR_AND_SUITE_FRAGS_II ] = "FIND_SUGAR_AND_SUITE_FRAGS_II";
	tag2string[ MAKE_BASE_PAIR_MOVE ] = "MAKE_BASE_PAIR_MOVE";
	tag2string[ MAKE_BASE_STEP_MOVE ] = "MAKE_BASE_STEP_MOVE";
	tag2string[ TOTAL ] = "TOTAL";

	// abinitio debugging tags
	tag2string[ ABINITIO ] = "ABINITIO";
	tag2string[ STAGE1 ] = "STAGE1";
	tag2string[ STAGE2 ] = "STAGE2";
	tag2string[ STAGE3 ] = "STAGE3";
	tag2string[ STAGE4 ] = "STAGE4";
	tag2string[ STAGE5 ] = "STAGE5";
	tag2string[ FRAGMENT_MOVER ] = "FRAGMENT_MOVER";
	tag2string[ RG ] = "RG";
	tag2string[ SEQUENCE_COMPARISON ] = "SEQUENCE_COMPARISON";
	tag2string[ KDTREE_CONSTRUCT] = "KDTREE_CONSTRUCT";
	tag2string[ KDTREE_SEARCH] = "KDTREE_SEARCH";
	tag2string[ CONSTRUCT_DISTANCE_MATRIX] = "CONSTRUCT_DISTANCE_MATRIX";

	tag2string[ JD2 ] = "JD2";
	tag2string[ JD2_OUTPUT ] = "JD2_OUTPUT";
	tag2string[ JD2_SILENT_OUTPUTTER ] = "JD2_SILENT_OUTPUTTER";
	tag2string[ JD2_INIT_MOVER ] ="JD2_INIT_MOVER";
	tag2string[ ARCHIVE_SYNC_BATCHES ] = "ARCHIVE_SYNC_BATCHES";
	tag2string[	ARCHIVE_JOBSCOMPLETE ] = "ARCHIVE_JOBSCOMPLETE";
	tag2string[ ARCHIVE_CRITICAL_JOBSCOMPLETE ] = "ARCHIVE_CRITICAL_JOBSCOMPLETE";
	tag2string[ ARCHIVE_READ_DECOYS ] = "ARCHIVE_READ_DECOYS";
	tag2string[ ARCHIVE_GEN_BATCH ] = "ARCHIVE_GEN_BATCH";
	tag2string[ ARCHIVE_BLOCK_FILE ] = "ARCHIVE_BLOCK_FILE";
	tag2string[ ARCHIVE_FILL_POSE ] = "ARCHIVE_FILL_POSE";
	tag2string[ ARCHIVE_SCORE_POSE ] = "ARCHIVE_SCORE_POSE";
	tag2string[ ARCHIVE_EVALUATORS ] = "ARCHIVE_EVALUATORS";
	tag2string[ CA_RMSD_EVALUATION ] = "CA_RMSD_EVALUATION";
	tag2string[ TRUNCATED_SCORE_EVALUATOR ] ="TRUNCATED_SCORE_EVALUATOR";


	tag2string[ SAVE_ARCHIVE ] = "SAVE_ARCHIVE";

	tag2string[ ARCHIVE_EVAL_DECOYS ] = "ARCHIVE_EVAL_DECOYS";
	tag2string[ SILENT_READ_TAG_TEST ] = "SILENT_READ_TAG_TEST";
	tag2string[ MPI_FILE_BUF ] = "MPI_FILE_BUF";
	tag2string[ MPI_JD2_WAITS_FOR_ARCHIVE ] ="MPI_JD2_WAITS_FOR_ARCHIVE";
	tag2string[ MPI_NOTIFY_ARCHIVE ] = "MPI_NOTIFY_ARCHIVE";

  // Fragmentpicker stuff
  tag2string[ FRAGMENTPICKING_PROFILE_SCORE ] = "FRAGMENTPICKING_PROFILE_SCORE";
  tag2string[ FRAGMENTPICKING_PROFILE_CAHING ] = "FRAGMENTPICKING_PROFILE_CAHING";
  tag2string[ FRAGMENTPICKING_SECONDARY_SCORE ] = "FRAGMENTPICKING_SECONDARY_SCORE";
  tag2string[ FRAGMENTPICKING_READ_VALL ] = "FRAGMENTPICKING_READ_VALL";
  tag2string[ FRAGMENTPICKING ] = "FRAGMENTPICKING";
  tag2string[ FRAGMENTPICKING_CANDIDATES_COLLECTING ] = "FRAGMENTPICKING_CANDIDATES_COLLECTING";
  tag2string[ FRAGMENTPICKING_ATOMPAIR_SCORE ] = "FRAGMENTPICKING_ATOMPAIR_SCORE";
  tag2string[ FRAGMENTPICKING_PHIPSI_SCORE ] = "FRAGMENTPICKING_PHIPSI_SCORE";
  tag2string[ FRAGMENTPICKING_DIHEDRALCONSTR_SCORE ] = "FRAGMENTPICKING_DIHEDRALCONSTR_SCORE";
}


////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
void
prof_show()
{
	using namespace ObjexxFCL;
	//using ObjexxFCL::F;
	//using ObjexxFCL::I;
	Tracer tt( "core.util.prof", t_info, true /*muted by default*/ );

	// clocks are shown in the unit of 0.01 sec
	//if ( !profiling ) return;
	static bool init( false );
	if( !init ) {
		init = true;
		setup_tag2string();
	}

	PROF_STOP( util::TOTAL );

	tt << "\n========================================\n";
	tt << "========================================\n";
	tt << "============ PROFILE INFO ==============\n";
	tt << "========================================\n";
	tt << "========================================\n";
	tt << A(12,"clock") << ' ' << A(9,"ncalls") << ' ' << A(9,"bad_calls") << ' ' << A(12,"c/call") << ' ' <<
		"tag" << '\n';
	for ( int i=1; i<= n_prof_tags; ++i ) {
		ProfTag const tag( static_cast< ProfTag >( i ) );
		double const t( total_clock[ tag ] );
		int const ncalls( calls[tag] );
		int const bcalls( bad_calls[tag] );
		double const clocks_per_call( ncalls != 0 ? t/ncalls : 0.0 );
		tt << F(12,2,t) << ' ' << I(9,ncalls) << ' ' << I(9,bcalls)
							<< ' ' << F(12,3, clocks_per_call ) << ' ' << tag2string[tag] << '\n';
	}
	tt << "========================================\n";
	tt << "========================================\n";
	tt << "========================================" << std::endl;
	PROF_START( TOTAL );
}

/////////////////////////////////////////////////////////////////////////////
void
prof_reset()
{
	// check for options?
	//static bool init( false );
	//if ( !init ) {
	//	init = true;
	//	profiling = truefalseoption("profile");
	//}
	//if ( force_profiling ) profiling = true;
	//if ( !profiling ) return;

	start_clock.clear();
	total_clock.clear();
	calls.clear();
	bad_calls.clear();

	start_clock.resize( n_prof_tags, 0 );
	total_clock.resize( n_prof_tags, 0 );
	calls.resize( n_prof_tags, 0 );
	bad_calls.resize( n_prof_tags, 0 );

	PROF_START( TOTAL );
}

} // util
} // core
