// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file
/// @brief
/// @author

#ifndef INCLUDED_core_util_prof_HH
#define INCLUDED_core_util_prof_HH

#include <utility/vector1.hh>
#include <string>
#include <time.h>

// Commented by inclean daemon #include <ctime>


namespace core {
namespace util {

/**
	 not intended for profiling inside tight loops
	 the clock() routine currently being used has fairly crappy
	 resolution and it will introduce some small overhead with the
	 function calls and the if-check even if not using -profile on
	 the command line

	 you can wrap it around large-ish chunks of code, like fullatom_energy
	 or rotamer_trials...


	 A simple setup for timing code fragments. Probably not optimal timing
	 functions -- I'm open to suggestions.

	 looks like (see eg fullatom_energy or scorefxn)

	 PROF_START( prof::TAG );

	 <function call>

	 PROF_STOP( prof::TAG );

	 where TAG is in the enum "Prof_tag" below (feel free to add new ones)
	 also add to tag2string if you want friendly output.

	 PROF_STOP checks the time and increments the total time assigned to TAG


	 2. later on, in your simulation code you can do:

	 prof_reset();

	 -- miscellaneous simulation --

	 prof_show();

	 The final call to prof::show() will display the time usage measured
	 by all the PROF_* calls between reset() and show()

**/

#ifdef NO_PROF
#define PROF_START(expr)
#define PROF_STOP(expr)
#else
#define PROF_START(expr) ( prof_start_function_body( expr) )
#define PROF_STOP(expr) ( prof_stop_function_body( expr ) )
#endif


///
enum ProfTag {
	TEST1 = 1,
	TEST2,
	TEST3,
	TEST4,
	TEST5,
	ATOM_TREE_UPDATE_INTERNAL_COORDS,
	ATOM_TREE_UPDATE_XYZ_COORDS,
	UPDATE_RESIDUE_TORSIONS,
	UPDATE_RESIDUE_COORDINATES,
	ROTAMER_TRIALS,
	PACK_ROTAMERS,
	UPDATE_RESIDUE_NEIGHBORS,
	SETUP_NBLIST,
	CONSTRAINT_SCORE,
	SCORE,
	POSE_COPY,
	ENERGY_GRAPH_COPY,
	CCD_CLOSE,
	FUNC,
	DFUNC,
	GET_ENERGIES,
	SIMANNEALING,
	INSERT_FRAGS,
	MC_ACCEPT,
	GB_SETUP_FOR_PACKING,
	GB_GET_ALL_BORN_RADII,
	GEN_BORN_ROTAMER_PAIR_ENERGIES,
	GEN_BORN_ROTAMER_BACKGROUND_ENERGIES,
	NEIGHBOR_ENERGIES,
	LONG_RANGE_ENERGIES,
	ABINITIO,
	STAGE1,
	STAGE2,
	STAGE3,
	STAGE4,
	STAGE5,
	FRAGMENT_MOVER,
	MINMOVER_APPLY,
	BACKRUB_MOVER,
	FIND_SUGAR_AND_SUITE_FRAGS_I,
	FIND_SUGAR_AND_SUITE_FRAGS_II,
	MAKE_BASE_PAIR_MOVE,
	MAKE_BASE_STEP_MOVE,
	RG,
	SEQUENCE_COMPARISON,
	KDTREE_CONSTRUCT,
	KDTREE_SEARCH,
	CONSTRUCT_DISTANCE_MATRIX,
	DALPHABALL,
	DALPHABALL_DERIV,
	MPI_FILE_BUF,
	JD2,
	JD2_OUTPUT,
	JD2_SILENT_OUTPUTTER,
	JD2_INIT_MOVER,
	ARCHIVE_SYNC_BATCHES,
	ARCHIVE_JOBSCOMPLETE,
	ARCHIVE_BLOCK_FILE,
 	SAVE_ARCHIVE,
	ARCHIVE_CRITICAL_JOBSCOMPLETE,
	ARCHIVE_GEN_BATCH,
	ARCHIVE_READ_DECOYS,
	ARCHIVE_EVAL_DECOYS,
	ARCHIVE_FILL_POSE,
	ARCHIVE_SCORE_POSE,
	ARCHIVE_EVALUATORS,
	MPI_JD2_WAITS_FOR_ARCHIVE,
	MPI_NOTIFY_ARCHIVE,
	SILENT_READ_TAG_TEST,
	CA_RMSD_EVALUATION,
	TRUNCATED_SCORE_EVALUATOR,

  FRAGMENTPICKING_PROFILE_SCORE,
  FRAGMENTPICKING_PROFILE_CAHING,
  FRAGMENTPICKING_SECONDARY_SCORE,
  FRAGMENTPICKING_READ_VALL,
  FRAGMENTPICKING,
  FRAGMENTPICKING_CANDIDATES_COLLECTING,
  FRAGMENTPICKING_ATOMPAIR_SCORE,
  FRAGMENTPICKING_PHIPSI_SCORE,
  FRAGMENTPICKING_DIHEDRALCONSTR_SCORE,
	TOTAL, // keep these two last
	n_prof_tags = TOTAL
};


extern utility::vector1< std::string > tag2string;
extern utility::vector1< clock_t > start_clock;
extern utility::vector1< double > total_clock;
extern utility::vector1< int > calls;
extern utility::vector1< int > bad_calls;

extern double const clock_factor;
extern clock_t const SHRINK_FACTOR; // prevent overflow
	//extern bool profiling;

///
inline
void
prof_start_function_body( ProfTag const tag )
{
	//if (!profiling) return;
	start_clock[ tag ] = clock() / SHRINK_FACTOR;
	++calls[ tag ];
}


///
inline
void
prof_stop_function_body( ProfTag const tag )
{
	//if (!profiling) return;

	clock_t const current( clock() / SHRINK_FACTOR );
	clock_t const start( start_clock[tag] );

	if ( current >= start ) {
		total_clock[ tag ] += clock_factor * ( current - start );
	} else {
		--calls[ tag ];
		++bad_calls[ tag ];
	}
}


void
prof_reset();// bool const force_profiling = false );


void
prof_show();

} // util
} // core


#endif
