// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
 //////////////////////////////////////////////
 /// @begin
 ///
 /// @file core/optimization/SVD_Solver.cc
 ///
 /// @brief SVD solver class
 ///`
 /// @detailed Solve over-determined set of linear equation to minimize ||A x - b||^2, using Singular Value Decomposition (SVD) method.
 ///
 /// @param
 /// Specify the size of the problem in the constructor (M is the number of equations, N is the number of parameters to fit)
 /// M MUST be larger or equal than N.
 /// Use the set_* functions to set the data vector b and the matrix A.
 /// Use the run_* functions in the correct order to solve your system (run_decomp_svd, then run_solve_svd)
 /// You can score the result with run_score_svd_on_matrix
 /// You can retrieve your solution with get_svd_solution.
 ///
 /// @return
 /// The score of the fitting : sqrt( ||A x - b||^2 ) with run_score_svd_on_matrix();
 /// The fited vector x with get_svd_solution();
 ///
 /// @remarks
 /// Calls in a wrong order of the functions will abort the program (i.e. if you try to solve the problem before you set a matrix A)
 /// Once the matrix is decomposed, you can change the vector b and solve Ax=b with the new vector. (That's why those 2 functions are separated)
 /// The matrix A is necessary to calculate the score (argument of run_score_svd_on_matrix), but the matrix A is not stored within
 /// the SVD_solver object, so make sure you have it available when scoring (this is done on purpose for speed up)
 /// Is it possible to speed up calculations by using FArraynD.index() call? ObjexxFCL doc is not really clear.
 ///
 /// @references
 ///
 /// @authorsv Christophe Schmitz & Srivatsan Raman
 ///
 /// @last_modified April 2009
 ////////////////////////////////////////////////

// Unit headers
#include <numeric/SVD/SVD_Solver.hh>

// Package headers

// Project headers

// Utility headers
#include <utility/vector1.hh>
#include <utility/exit.hh>

// Numeric headers

// Objexx headers
#include <ObjexxFCL/Fmath.hh>

// C++ headers
#include <iostream>


namespace numeric{
namespace SVD{

using namespace core;

SVD_Solver::~SVD_Solver(){
}

SVD_Solver &
SVD_Solver::operator=(SVD_Solver const & other){
	if ((M_ != other.M_) || (N_ != other.N_)){
		utility_exit_with_message( "You can't call the = operator on SVD_Solver object of different size" );
	}
	if ( this != &other ) {
		fstyle_b_ = other.fstyle_b_;
		fstyle_A_decomp_ = other.fstyle_A_decomp_;
		fstyle_v_ = other.fstyle_v_;
		fstyle_x_ = other.fstyle_x_;
		fstyle_w_ = other.fstyle_w_;
		fstyle_tmp_ = other.fstyle_tmp_;

		b_is_set_ = other.b_is_set_;
		A_is_set_ = other.A_is_set_;
		A_is_decomp_ = other.A_is_decomp_;
		x_is_solved_ = other.x_is_solved_;
	}
	return *this;
}

SVD_Solver:: SVD_Solver(SVD_Solver const & other):
	M_(other.M_), N_(other.N_)
{
  fstyle_b_ = other.fstyle_b_;
  fstyle_A_decomp_ = other.fstyle_A_decomp_;
  fstyle_v_ = other.fstyle_v_;
  fstyle_x_ = other.fstyle_x_;
  fstyle_w_ = other.fstyle_w_;
  fstyle_tmp_ = other.fstyle_tmp_;

	b_is_set_ = other.b_is_set_;
	A_is_set_ = other.A_is_set_;
	A_is_decomp_ = other.A_is_decomp_;
	x_is_solved_ = other.x_is_solved_;
}

SVD_Solver::SVD_Solver():
	M_(0), N_(0)
{
	utility_exit_with_message( "You shouldn't call the empty constructor for SVD_Solver class" );
}

///////////////////////////////////////////////
/// @brief M is the size of vector b (experimental data); N is the size of the vector to optimize M >= N
///////////////////////////////////////////////
SVD_Solver::SVD_Solver(core::Size const M, core::Size const N):
	M_(M), N_(N)
{

	if(N_ >= M_){
		utility_exit_with_message("First parameter of SVD_Solver constructor MUST be larger than the second parameter");
	}

	fstyle_A_decomp_.dimension(M, N);
	fstyle_b_.dimension(M);
	fstyle_v_.dimension(N, N);
	fstyle_x_.dimension(N);
	fstyle_w_.dimension(N);
	fstyle_tmp_.dimension(N);

	b_is_set_ = false;
	A_is_set_ = false;
	A_is_decomp_ = false;
	x_is_solved_ = false;
}

///////////////////////////////////////////////
/// @brief To minimize ||Ax - b||^2, you need to set the vector b
///////////////////////////////////////////////
void
SVD_Solver::set_vector_b(utility::vector1<core::Real>  const & b){
	core::Size i;
	if (M_ != b.size()){
		utility_exit_with_message("Size dimension differs when trying to set vector b in SVD_solver class");
	}

	for (i = 1; i <= M_; ++i){
		fstyle_b_(i) = b[i];
	}

	b_is_set_ = true;
	x_is_solved_ = false;
}

///////////////////////////////////////////////
/// @brief To minimize ||Ax - b||^2, you need to set the vector b
///////////////////////////////////////////////
void
SVD_Solver::set_vector_b(FArray1D< core::Real > const & b){
	core::Size i;
	if (M_ != b.size()){
		utility_exit_with_message("Size dimension differs when trying to set vector b in SVD_solver class");
	}

	for (i = 1; i <= M_; ++i){
		fstyle_b_(i) = b(i);
	}

	b_is_set_ = true;
	x_is_solved_ = false;
}

///////////////////////////////////////////////
/// @brief To minimize ||Ax - b||^2, you need to set the matrix A
///////////////////////////////////////////////
void
SVD_Solver::set_matrix_A(utility::vector1< utility::vector1<core::Real> >  const & A){
	core::Size i, j;
	if (M_ != A.size()){
		utility_exit_with_message("Size dimension differs when trying to set the matrix A in SVD_solver class");
	}

	for (i = 1; i <= M_; ++i){
		if (N_ != A[i].size()){
			utility_exit_with_message( "Size dimension differs when trying to set the matrix A in SVD_solver class");
		}
		for (j = 1; j <= N_; ++j){
			fstyle_A_decomp_(i, j) = A[i][j];
		}
	}

	A_is_set_ = true;
	A_is_decomp_ = false;
	x_is_solved_ = false;
}


///////////////////////////////////////////////
/// @brief To minimize ||Ax - b||^2, you need to set the matrix A
///////////////////////////////////////////////
void
SVD_Solver::set_matrix_A( FArray2D< core::Real > const & A){
	core::Size i, j;

	if (M_*N_ != A.size()){
		utility_exit_with_message( "Size dimension differs when trying to set the matrix A in SVD_solver class");
	}

	for (i = 1; i <= M_; ++i){
		for (j = 1; j <= N_; ++j){
			fstyle_A_decomp_(i, j) = A(i, j);
		}
	}

	A_is_set_ = true;
	A_is_decomp_ = false;
	x_is_solved_ = false;
}

///////////////////////////////////////////////
/// @brief Decomposition of the matrix A. Can be called only when A is set.
/// You can't decompose twice in a row without reseting the matrix A.
///////////////////////////////////////////////
void
SVD_Solver::run_decomp_svd(){
	if(!A_is_set_){
		utility_exit_with_message("The matrix A is not ready to be decomposed in SVD_solver class");
	}
	svdcmp();
	A_is_decomp_ = true;
}

///////////////////////////////////////////////
/// @brief Minimize ||Ax - b||^2. Can be called only when b is set and A is decomposed.
/// You are allowed to update the vector b and minimize again.
///////////////////////////////////////////////
void
SVD_Solver::run_solve_svd(){
	if((!A_is_decomp_) || (!b_is_set_)){
		utility_exit_with_message("SVD_solver object is not in a state to solve Ax = b");
	}
	svbksb();
	x_is_solved_ = true;

}

///////////////////////////////////////////////
/// @brief Return the optimized vector x. Can be called only when ||Ax-b||^2 has been minimized.
///////////////////////////////////////////////
FArray1D< core::Real > const &
SVD_Solver::get_svd_solution() const{
	if(!x_is_solved_){
		utility_exit_with_message("SVD_solver object has not yet solved Ax = b");
	}
	return(fstyle_x_);
}

///////////////////////////////////////////////
/// @brief Return the score SQRT(||Ax-b||^2). Can be called only when ||Ax-b||^2 has been minimized.
/// You need the give the original A matrix as argument.
///////////////////////////////////////////////
core::Real
SVD_Solver::run_score_svd_on_matrix(utility::vector1< utility::vector1<core::Real> > const & cppstyle_A) const{

	core::Size i, j;
	core::Real score, temp;

	if(!x_is_solved_){
		utility_exit_with_message("SVD_solver object is not in a state to score ||Ax = b||^2");
	}
	score = 0;
	for(i = 1; i <= M_; ++i){
		temp = 0;
		for(j = 1; j <= N_; ++j){
			temp += cppstyle_A[i][j] * fstyle_x_(j);
		}
		score += ( temp - fstyle_b_(i) ) * ( temp - fstyle_b_(i) ) ;
	}
	return(sqrt(score));
}

///////////////////////////////////////////////
/// @brief Return the score SQRT(||Ax-b||^2). Can be called only when ||Ax-b||^2 has been minimized.
/// You need the give the original A matrix as argument.
///////////////////////////////////////////////
core::Real
SVD_Solver::run_score_svd_on_matrix(FArray2D< core::Real > const & A) const{

	core::Size i, j;
	core::Real score, temp;

	if(!x_is_solved_){
		utility_exit_with_message("SVD_solver object is not in a state to score ||Ax = b||^2");
	}
	score = 0;
	for(i = 1; i <= M_; ++i){
		temp = 0;
		for(j = 1; j <= N_; ++j){
			temp += A(i, j) * fstyle_x_(j);
		}
		score += ( temp - fstyle_b_(i) ) * ( temp - fstyle_b_(i) ) ;
	}
	return(sqrt(score));
}

///////////////////////////////////////////////
/// @brief Attempt to speed up calculation of the cost without actually solving Ax = b
/// The routine works, but is not faster (nor slower)
///////////////////////////////////////////////
core::Real
SVD_Solver::run_score_svd_without_solving(){

	core::Size i, j;
	core::Real score, temp;

	if((!A_is_decomp_) || (!b_is_set_)){
		utility_exit_with_message("SVD_Solver object not in state to call run_score_svd_without_solving");
	}

	for(i = 1; i <= N_; ++i){
		fstyle_tmp_(i) = 0;
		for(j = 1; j <= M_; j++){
			fstyle_tmp_(i) += fstyle_A_decomp_(j, i) *  fstyle_b_(j);
		}
	}

	score = 0;
	for(i = 1; i <= M_; ++i){
		temp = 0;
		for(j = 1; j <= N_; ++j){
			temp += fstyle_A_decomp_(i, j) * fstyle_tmp_(j);
		}
		temp -= fstyle_b_(i);
		score += 	temp * temp;
	}

	return(sqrt(score));
}

////////////////////////////////////////////////////////////////
// copy past from ResidualDipolarCouplingEnergy.cc of Srivatsan Raman
////////////////////////////////////////////////////////////////
core::Real
SVD_Solver::pythag(Real const & a, Real const & b) const {

	Real pythag;

	Real absa = std::abs(a);
	Real absb = std::abs(b);
	if ( absa > absb ) {
		Real const ratio = absb/absa;
		pythag = absa * std::sqrt( 1.0 + ( ratio * ratio ) );
	} else {
		if ( absb == 0.0 ) {
			pythag = 0.0;
		} else {
			Real const ratio = absa/absb;
			pythag = absb * std::sqrt( 1.0 + ( ratio * ratio ) );
		}
	}
	return pythag;
}


////////////////////////////////////////////////////////////////
// Copy past from ResidualDipolarCouplingEnergy.cc of Srivatsan Raman
// Can this be optimized with FArraynD.index() call?
// Is it somehhow possible to score quicker without calculating the x vector?
////////////////////////////////////////////////////////////////
void
SVD_Solver::svbksb(){
	Real s;

	for ( Size j = 1; j <= N_; ++j ) {
		s = 0.0;
		if ( fstyle_w_(j) != 0.0 ) {
			for ( Size i = 1; i <= M_; ++i ) {
				s += fstyle_A_decomp_(i,j) * fstyle_b_(i);
			}
			s /= fstyle_w_(j);
		}
		fstyle_tmp_(j) = s;
	}
	for ( Size j = 1; j <= N_; ++j ) {
		s = 0.0;
		for ( Size jj = 1; jj <= N_; ++jj ) {
			s += fstyle_v_(j,jj) * fstyle_tmp_(jj);
		}
		fstyle_x_(j) = s;
	}
}

////////////////////////////////////////////////////////////////
// copy past from ResidualDipolarCouplingEnergy.cc of Srivatsan Raman
// Can this be optimized with FArraynD.index() call?
////////////////////////////////////////////////////////////////
void
SVD_Solver::svdcmp(){

//U    USES pythag
	Size i,its,j,jj,k,l,nm;
	Real anorm, c, f, g, h, s, scale, x, y, z;
	g = 0.0;
	scale = 0.0;
	anorm = 0.0;
	l = 0;
	nm = 0;
	for ( i = 1; i <= N_; ++i ) {
		l = i+1;
		fstyle_tmp_(i) = scale*g;
		g = 0.0;
		s = 0.0;
		scale = 0.0;
		if ( i <= M_ ) {
			for ( k = i; k <= M_; ++k ) {
				scale += std::abs(fstyle_A_decomp_(k,i));
			}
			if ( scale != 0.0 ) {
				for ( k = i; k <= M_; ++k ) {
					fstyle_A_decomp_(k,i) /= scale;
					s += fstyle_A_decomp_(k,i)*fstyle_A_decomp_(k,i);
				}
				f = fstyle_A_decomp_(i,i);
				g = -sign(std::sqrt(s),f);
				h = f*g-s;
				fstyle_A_decomp_(i,i) = f-g;
				for ( j = l; j <= N_; ++j ) {
					s = 0.0;
					for ( k = i; k <= M_; ++k ) {
						s += fstyle_A_decomp_(k,i)*fstyle_A_decomp_(k,j);
					}
					f = s/h;
					for ( k = i; k <= M_; ++k ) {
						fstyle_A_decomp_(k,j) += f*fstyle_A_decomp_(k,i);
					}
				}
				for ( k = i; k <= M_; ++k ) {
					fstyle_A_decomp_(k,i) *= scale;
				}
			}
		}
		fstyle_w_(i) = scale *g;
		g = 0.0;
		s = 0.0;
		scale = 0.0;
		if ( (i <= M_) && (i != N_) ) {
			for ( k = l; k <= N_; ++k ) {
				scale += std::abs(fstyle_A_decomp_(i,k));
			}
			if ( scale != 0.0 ) {
				for ( k = l; k <= N_; ++k ) {
					fstyle_A_decomp_(i,k) /= scale;
					s += fstyle_A_decomp_(i,k)*fstyle_A_decomp_(i,k);
				}
				f = fstyle_A_decomp_(i,l);
				g = -sign(std::sqrt(s),f);
				h = f*g-s;
				fstyle_A_decomp_(i,l) = f-g;
				for ( k = l; k <= N_; ++k ) {
					fstyle_tmp_(k) = fstyle_A_decomp_(i,k)/h;
				}
				for ( j = l; j <= M_; ++j ) {
					s = 0.0;
					for ( k = l; k <= N_; ++k ) {
						s += fstyle_A_decomp_(j,k)*fstyle_A_decomp_(i,k);
					}
					for ( k = l; k <= N_; ++k ) {
						fstyle_A_decomp_(j,k) += s*fstyle_tmp_(k);
					}
				}
				for ( k = l; k <= N_; ++k ) {
					fstyle_A_decomp_(i,k) *= scale;
				}
			}
		}
		anorm = std::max(anorm,(std::abs(fstyle_w_(i))+std::abs(fstyle_tmp_(i))));
	}
	for ( i = N_; i >= 1; --i ) {
		if ( i < N_ ) {
			if ( g != 0.0 ) {
				for ( j = l; j <= N_; ++j ) {
					fstyle_v_(j,i) = (fstyle_A_decomp_(i,j)/fstyle_A_decomp_(i,l))/g;
				}
				for ( j = l; j <= N_; ++j ) {
					s = 0.0;
					for ( k = l; k <= N_; ++k ) {
						s += fstyle_A_decomp_(i,k)*fstyle_v_(k,j);
					}
					for ( k = l; k <= N_; ++k ) {
						fstyle_v_(k,j) += s*fstyle_v_(k,i);
					}
				}
			}
			for ( j = l; j <= N_; ++j ) {
				fstyle_v_(i,j) = 0.0;
				fstyle_v_(j,i) = 0.0;
			}
		}
		fstyle_v_(i,i) = 1.0;
		g = fstyle_tmp_(i);
		l = i;
	}
	for ( i = std::min(M_,N_); i >= 1; --i ) {
		l = i+1;
		g = fstyle_w_(i);
		for ( j = l; j <= N_; ++j ) {
			fstyle_A_decomp_(i,j) = 0.0;
		}
		if ( g != 0.0 ) {
			g = 1.0/g;
			for ( j = l; j <= N_; ++j ) {
				s = 0.0;
				for ( k = l; k <= M_; ++k ) {
					s += fstyle_A_decomp_(k,i)*fstyle_A_decomp_(k,j);
				}
				f = (s/fstyle_A_decomp_(i,i))*g;
				for ( k = i; k <= M_; ++k ) {
					fstyle_A_decomp_(k,j) += f*fstyle_A_decomp_(k,i);
				}
			}
			for ( j = i; j <= M_; ++j ) {
				fstyle_A_decomp_(j,i) *= g;
			}
		} else {
			for ( j = i; j <= M_; ++j ) {
				fstyle_A_decomp_(j,i) = 0.0;
			}
		}
		fstyle_A_decomp_(i,i) += 1.0;
	}
	for ( k = N_; k >= 1; --k ) {
		for ( its = 1; its <= 30; ++its ) {
			for ( l = k; l >= 1; --l ) {
				nm = l-1;
				if ( (std::abs(fstyle_tmp_(l))+anorm) == anorm ) goto L2;
				if ( (std::abs(fstyle_w_(nm))+anorm) == anorm ) goto L1;
			}
L1:
			c = 0.0;
			s = 1.0;
			for ( i = l; i <= k; ++i ) {
				f = s*fstyle_tmp_(i);
				fstyle_tmp_(i) *= c;
				if ( (std::abs(f)+anorm) == anorm ) goto L2;
				g = fstyle_w_(i);
				h = pythag(f,g);
				fstyle_w_(i) = h;
				h = 1.0/h;
				c = (g*h);
				s = -(f*h);
				for ( j = 1; j <= M_; ++j ) {
					y = fstyle_A_decomp_(j,nm);
					z = fstyle_A_decomp_(j,i);
					fstyle_A_decomp_(j,nm) = (y*c)+(z*s);
					fstyle_A_decomp_(j,i) = -(y*s)+(z*c);
				}
			}
L2:
			z = fstyle_w_(k);
			if ( l == k ) {
				if ( z < 0.0 ) {
					fstyle_w_(k) = -z;
					for ( j = 1; j <= N_; ++j ) {
						fstyle_v_(j,k) = -fstyle_v_(j,k);
					}
				}
				goto L3;
			}
			if ( its == 30) utility_exit_with_message("no convergence in svdcmp \n" );
			x = fstyle_w_(l);
			nm = k-1;
			y = fstyle_w_(nm);
			g = fstyle_tmp_(nm);
			h = fstyle_tmp_(k);
			f = ((y-z)*(y+z)+(g-h)*(g+h))/(2.0*h*y);
			g = pythag(f,1.0);
			f = ((x-z)*(x+z)+h*((y/(f+sign(g,f)))-h))/x;
			c = 1.0;
			s = 1.0;
			for ( j = l; j <= nm; ++j ) {
				i = j+1;
				g = fstyle_tmp_(i);
				y = fstyle_w_(i);
				h = s*g;
				g *= c;
				z = pythag(f,h);
				fstyle_tmp_(j) = z;
				c = f/z;
				s = h/z;
				f = (x*c)+(g*s);
				g = -(x*s)+(g*c);
				h = y*s;
				y *= c;
				for ( jj = 1; jj <= N_; ++jj ) {
					x = fstyle_v_(jj,j);
					z = fstyle_v_(jj,i);
					fstyle_v_(jj,j) = (x*c)+(z*s);
					fstyle_v_(jj,i) = -(x*s)+(z*c);
				}
				z = pythag(f,h);
				fstyle_w_(j) = z;
				if ( z != 0.0 ) {
					z = 1.0/z;
					c = f*z;
					s = h*z;
				}
				f = (c*g)+(s*y);
				x = -(s*g)+(c*y);
				for ( jj = 1; jj <= M_; ++jj ) {
					y = fstyle_A_decomp_(jj,j);
					z = fstyle_A_decomp_(jj,i);
					fstyle_A_decomp_(jj,j) = (y*c)+(z*s);
					fstyle_A_decomp_(jj,i) = -(y*s)+(z*c);
				}
			}
			fstyle_tmp_(l) = 0.0;
			fstyle_tmp_(k) = f;
			fstyle_w_(k) = x;
		}
L3:;
	}
}

}//namespace SVD
}//namespace numeric
