// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// This file is part of the Rosetta software suite and is made available under license.
// The Rosetta software is developed by the contributing members of the Rosetta Commons consortium.
// (C) 199x-2009 Rosetta Commons participating institutions and developers.
// For more information, see http://www.rosettacommons.
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.
//////////////////////////////////////////////
/// @begin
///
/// @file core/optimization/SVD_Solver.hh
///
/// @brief SVD solver class
///
/// @detailed Solve over-determined set of linear equation to minimize ||A x - b||^2, using Singular Value Decomposition (SVD) method.
///
/// @param
/// Specify the size of the problem in the constructor (M is the number of equations, N is the number of parameters to fit)
/// M MUST be larger or equal than N.
/// Use the set_* functions to set the data vector b and the matrix A.
/// Use the run_* functions in the correct order to solve your system (run_decomp_svd, then run_solve_svd)
/// You can score the result with run_score_svd_on_matrix
/// You can retrieve your solution with get_svd_solution.
///
/// @return
/// The score of the fitting : sqrt( ||A x - b||^2 ) with run_score_svd_on_matrix();
/// The fitted vector x with get_svd_solution();
///
/// @remarks
/// Calls in a wrong order of the functions will abort the program (i.e. if you try to solve the problem before you set a matrix A)
/// Once the matrix is decomposed, you can change the vector b and solve Ax=b with the new vector. (That's why those 2 functions are separated)
/// The matrix A is necessary to calculate the score (argument of run_score_svd_on_matrix), but the matrix A is not stored within
/// the SVD_solver object, so make sure you have it available when scoring (this is done on purpose for speed up)
/// Is it possible to speed up calculations by using FArraynD.index() call? ObjexxFCL doc is not really clear.
///
/// @references
///
/// @authorsv Christophe Schmitz
///
/// @last_modified April 2009
////////////////////////////////////////////////

#ifndef INCLUDED_numeric_SVD_SVD_Solver_HH
#define INCLUDED_numeric_SVD_SVD_Solver_HH

// Package headers

// Project headers
#include <core/types.hh>

// Utility headers
#include <utility/vector1.fwd.hh>
// Numeric headers

// Objexx headers
#include <ObjexxFCL/FArray2D.hh>
#include <ObjexxFCL/FArray1D.hh>

// C++ headers


namespace numeric{
namespace SVD{

using namespace ObjexxFCL;

class SVD_Solver {

private:

	FArray1D< core::Real > fstyle_b_; //The vector b you want to use to solve Ax = b
	FArray2D< core::Real > fstyle_A_decomp_;
	FArray2D< core::Real > fstyle_v_;
	FArray1D< core::Real > fstyle_x_; //The solution x
	FArray1D< core::Real > fstyle_w_;
	FArray1D< core::Real > fstyle_tmp_;

	core::Size const M_; //M_ is the size of vector b (data)
	core::Size const N_; //N_ is the size of the parameter vector to optimize

	//A few states used to make sure the order of calls are correct
	bool b_is_set_;
	bool A_is_set_;
	bool A_is_decomp_;
	bool x_is_solved_;

public:
	SVD_Solver(core::Size const M, core::Size const N);

	~SVD_Solver();

	SVD_Solver();

	SVD_Solver &
	operator=(SVD_Solver const & other);

	SVD_Solver(SVD_Solver const & other);

	void
	set_vector_b( utility::vector1<core::Real>  const & b );

	void
	set_matrix_A( utility::vector1< utility::vector1<core::Real> >  const & A);

	void
	set_vector_b( FArray1D< core::Real > const & b );

	void
	set_matrix_A( FArray2D< core::Real > const & A );

	core::Real
	run_score_svd_on_matrix(utility::vector1< utility::vector1<core::Real> > const & cppstyle_A) const;

	core::Real
	run_score_svd_on_matrix(FArray2D< core::Real > const & A) const;

	core::Real
	run_score_svd_without_solving();

	void
	run_decomp_svd();

	void
	run_solve_svd();

	FArray1D< core::Real > const &
	get_svd_solution() const;

private:

	core::Real
	pythag(core::Real const & a, core::Real const & b) const;

	void
	svdcmp();

	void
	svbksb();

};

}//namespace SVD
}//namespace numeric


#endif
