// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file numeric/kdtree/kdtree.hh
/// @brief utility functions for kd-tree. See kdtree.hh for more information.
/// @author James Thompson
//
#ifndef INCLUDED_numeric_kdtree_util_HH
#define INCLUDED_numeric_kdtree_util_HH

#include <numeric/types.hh>
#include <utility/vector1.hh>

#include <numeric/kdtree/KDTree.fwd.hh>
#include <numeric/kdtree/KDNode.fwd.hh>
#include <numeric/kdtree/KDPoint.fwd.hh>
#include <numeric/kdtree/HyperRectangle.fwd.hh>

#include <numeric/kdtree/KDPointList.hh>

#include <numeric/kdtree/WrappedReal.hh>

#include <utility/pointer/ReferenceCount.hh>
#include <utility/pointer/ReferenceCount.fwd.hh>

#include <iostream>
#include <string>

namespace numeric {
namespace kdtree {

/// @brief Returns a KDPointList of the N nearest neighbors from the KDTree to
/// the given input point.
KDPointList
nearest_neighbors(
	KDTree & tree,
	utility::vector1< numeric::Real > const & pt,
	Size const wanted
);

/// @brief Searches the KDtree for the nearest neigbor to a given input point,
/// returns nearest neighbor and distance-squared to nearest neigbor by
/// reference.
void
nearest_neighbor(
	KDTree & tree,
	utility::vector1< numeric::Real > const & pt,
	// returns:
	KDNodeOP & nearest,
	numeric::Real & dist_sq
);


/// @brief Function for constructing a KDTree. Returns a KDNodeOP that
/// represents the root of the tree. Points need to be sorted as the
/// tree is being constructed, so the reference to the points is non-const.
KDNodeOP construct_kd_tree(
	utility::vector1< KDPointOP > & points,
	numeric::Size depth,
	KDTree & tree
);

/// distance metrics for real-valued points

/// @brief Returns the square of the Euclidean distance between the two points
/// vec1 and vec2.
numeric::Real sq_vec_distance(
	utility::vector1< numeric::Real > const & vec1,
	utility::vector1< numeric::Real > const & vec2
);

/// @brief Returns the Euclidean distance between the two points vec1 and vec2.
numeric::Real vec_distance(
	utility::vector1< numeric::Real > const & vec1,
	utility::vector1< numeric::Real > const & vec2
);

/// @brief Transforms the list of points given into percentiles using
/// a linear mapping from the input space to percentile-space for each
/// variable.
/// @detailed For each variable X in row R, replaces X with the quantity
/// ( X - min(R) ) / ( max(R) - min(R) ). Runs in O(N) time.
void transform_percentile(
	utility::vector1< utility::vector1< numeric::Real > > & points
);

void transform_percentile(
	utility::vector1< utility::vector1< numeric::Real > > & points,
	HyperRectangleOP bounds
);

void transform_percentile_single_pt(
	utility::vector1< numeric::Real > & point,
	HyperRectangleOP bounds
);

/// @brief Makes a vector of KDPoints.
utility::vector1< KDPointOP > make_points(
	utility::vector1< utility::vector1< numeric::Real > > const & points
);

HyperRectangleOP get_percentile_bounds(
	utility::vector1< utility::vector1< numeric::Real > > & points
);

/// @brief Makes a vector1 of KDPoints, associating the nth entry in data
/// with the nth entry in points.
utility::vector1< KDPointOP > make_points(
	utility::vector1< utility::vector1< numeric::Real > > const & points,
	utility::vector1< utility::pointer::ReferenceCountOP > const & data
);

/// @brief output operator for vector1< Real >
std::ostream & operator<< (
	std::ostream & out,
	const utility::vector1< numeric::Real > & points
);

/// @brief returns true if the given hyper-rectangle intersects with the given
/// hypersphere.
bool hr_intersects_hs(
	HyperRectangle hr,
	utility::vector1< numeric::Real > const & pt,
	numeric::Real r
);

// Recursive functions that do things to kd-trees, not intended for public
// use because they're a little awkward.

/// @brief returns the nearest neighbor to the given point.
/// @detailed Parameters are (in order):
/// - current: the base of the tree
/// - pt: the point that is being searched against the tree
/// - bounds: hyper-rectangle in k-space that bounds all points in the tree
/// - max_dist_sq: maximum squared distance that we care about.
///
/// - nearest neighbor (returned by reference)
/// - squared distance to the nearest neighbor
void
nearest_neighbor(
	KDNodeOP & current,
	utility::vector1< numeric::Real > const & pt,
	HyperRectangle & bounds,
	numeric::Real max_dist_sq,

	// returns:
	KDNodeOP & nearest,
	numeric::Real & dist_sq
);

/// @brief Recursive function definition for search for a list of the N nearest
/// neighbors, where N is defined as a member variable of the KDPointList
/// object.
void
nearest_neighbors(
	KDNodeOP & current,
	utility::vector1< numeric::Real > const & pt,
	HyperRectangle & bounds,
	numeric::Real max_dist_sq,

	// returns:
	KDPointList & neighbors
);

} // kdtree
} // numeric

#endif
