// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file devel/ProteinInterfaceDesign/DockDesign.cc
/// @author Sarel Fleishman (sarelf@u.washington.edu)

// Project Headers
#include <protocols/moves/PackRotamersMover.hh>
//#include <protocols/moves/ResidueMover.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/docking/RestrictTaskForDocking.hh>
#include <core/pose/Pose.hh>
#include <protocols/moves/MoverStatus.hh>

#include <core/conformation/Interface.hh>
#include <core/kinematics/Jump.hh>
#include <core/scoring/rms_util.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/io/database/open.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/util/Tracer.hh>

#include <core/options/util.hh>

#include <protocols/filters/Filter.hh>
#include <protocols/moves/DataMap.hh>
#include <utility/Tag/Tag.hh>

// JD2 headers
#include <protocols/jd2/JobDistributor.hh>
#include <protocols/jd2/Job.hh>
#include <core/options/keys/jd2.OptionKeys.gen.hh>

// Utility Headers
#include <utility/vector1.hh>

// Unit Headers
#include <protocols/ProteinInterfaceDesign/DockDesign.hh>

// C++ headers
#include <map>
#include <string>
#include <fstream>

namespace protocols {
namespace ProteinInterfaceDesign {

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.DockDesign" );
static core::util::Tracer TR_report( "protocols.ProteinInterfaceDesign.DockDesign.REPORT" );

typedef core::Real Real;
typedef core::pose::Pose Pose;

using namespace protocols::ProteinInterfaceDesign;
using namespace core;
using namespace std;

/// @detailed Takes care of the docking, design and filtering moves. pre_cycle and pose_cycle can
/// be setup in derived classes to setup variables before and after these cycles.
void
DockDesign::apply( Pose & pose )
{
//	runtime_assert( docking_mover_set_ );
	runtime_assert( movers_.size() );

	protocols::moves::Mover::set_last_move_status( protocols::moves::FAIL_RETRY );

	pose.update_residue_neighbors();
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it ) {
		(*mover_it).first->set_native_pose( get_native_pose() );
		(*mover_it).first->apply( pose );
		// collect Mover info: jd2 JobDistributor passes this info to Job,
		// and JobOutputters may then write this info to output files
		info().insert( info().end(), mover_it->first->info().begin(), mover_it->first->info().end() );
		pose.update_residue_neighbors();
		moves::MoverStatus status( (*mover_it).first->get_last_move_status() );
		bool const pass( status==protocols::moves::MS_SUCCESS  && (*mover_it).second->apply( pose ) );
		if( pass ) {
			if( (unsigned int) ( mover_it - movers_.begin() + 1) == movers_.size() ) { // return successfully only on last mover
				protocols::moves::Mover::set_last_move_status( protocols::moves::MS_SUCCESS ); // tell jobdistributor to save pose
				TR<<"setting status to success"<<std::endl;

				// Adds filter values to job info.  Only used in JD2.
				if ( core::options::option[ core::options::OptionKeys::jd2::dd_parser ] ){
					using protocols::jd2::JobDistributor;
					using protocols::jd2::InnerJob;
					protocols::jd2::JobOP job_me( JobDistributor::get_instance()->current_job() );
					FArray1D_bool superpos ( pose.total_residue(), true );
					core::Real const rms( core::scoring::rmsd_with_super_subset( *(job_me->inner_job()->get_pose()), pose, superpos, core::scoring::is_protein_CA ) );
					job_me->add_string_real_pair("rms", rms );

					for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
							 mover_it!=movers_.end(); ++mover_it ) {
						core::Real const filter_value( (*mover_it).second->report_sm( pose ) );
						if( filter_value > -9999 )
							job_me->add_string_real_pair((*mover_it).second->get_user_defined_name(), filter_value);
					}
				}
				report_all( pose );
				// rescore the pose with either score12 or a user-specified scorefunction. this ensures that all output files end up with scores.
				core::scoring::ScoreFunctionOP scorefxn = core::scoring::getScoreFunction();
				(*scorefxn)(pose);
				return;
			}
		}
		else{ // fail
			if( status != protocols::moves::MS_SUCCESS )
				protocols::moves::Mover::set_last_move_status( status );
			return;
		}
	}
}

void
DockDesign::report_all( Pose const & pose ) const {
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it )
		(*mover_it).second->report( TR_report, pose );
	TR_report.flush();
}

void
DockDesign::report_all_sm( std::map< std::string, core::Real > & score_map, Pose const & pose ) const {
	for( utility::vector1< mover_filter_pair >::const_iterator mover_it = movers_.begin();
		 mover_it!=movers_.end(); ++mover_it ) {
		 core::Real const filter_value( (*mover_it).second->report_sm( pose ) );
		 if( filter_value >= -9999 )
			score_map[ (*mover_it).second->get_user_defined_name() ] = filter_value;
	}
}

void
DockDesign::parse_my_tag( TagPtr const tag,
	protocols::moves::DataMap &,
	protocols::filters::Filters_map const &filters,
	protocols::moves::Movers_map const &movers,
	core::pose::Pose const & )
{
	using namespace protocols::moves;
	using namespace utility::Tag;

	TR<<"DockDesign mover with the following movers and filters\n";
	std::vector< TagPtr > const dd_tags( tag->getTags() );
	for( std::vector< TagPtr >::const_iterator dd_it=dd_tags.begin(); dd_it!=dd_tags.end(); ++dd_it ) {
		TagPtr const tag_ptr = *dd_it;
		std::string const mover_name( tag_ptr->getOption<string>( "mover_name", "null" ) );
		std::string const filter_name( tag_ptr->getOption<string>( "filter_name", "true_filter" ));

		std::map< std::string const, MoverOP >::const_iterator find_mover( movers.find( mover_name ) );
		std::map< std::string const, protocols::filters::FilterCOP >::const_iterator find_filter( filters.find( filter_name ));
		if( find_mover == movers.end() ) {
			TR.Error<<"mover not found in map. skipping:\n"<<tag_ptr<<std::endl;
			runtime_assert( find_mover != movers.end() );
			continue;
		}
		if( find_filter == filters.end() ) {
			TR.Error<<"filter not found in map. skipping:\n"<<tag_ptr<<std::endl;
			runtime_assert( find_filter != filters.end() );
			continue;
		}
		add_mover( find_mover->second, find_filter->second );
		TR << "added mover \"" << mover_name << "\" with filter \"" << filter_name << "\"\n";
	}
	TR.flush();
}

} //ProteinInterfaceDesign
} //protocols

