// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/movers/DesignMinimizeHbonds.hh
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

#ifndef INCLUDED_protocols_ProteinInterfaceDesign_movers_DesignMinimizeHbonds_HH
#define INCLUDED_protocols_ProteinInterfaceDesign_movers_DesignMinimizeHbonds_HH

#include <protocols/ProteinInterfaceDesign/movers/DesignRepackMover.hh>
#include <core/types.hh>
#include <core/pose/Pose.fwd.hh>
#include <utility/Tag/Tag.fwd.hh>
#include <protocols/filters/Filter.fwd.hh>
#include <protocols/moves/Mover.fwd.hh>
#include <core/scoring/ScoreFunction.fwd.hh>

namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

/// @brief used to design a protein to hbond preferentially to a set of target residues on the partner.
/// Hbonds involving backbone or sidechain on the target can be counted, and whether to design donors or
/// acceptors can also be defined.
class DesignMinimizeHbonds : public DesignRepackMover
{
public:
	typedef core::scoring::ScoreFunctionOP ScoreFunctionOP;
	typedef core::scoring::ScoreFunctionCOP ScoreFunctionCOP;
	typedef core::scoring::ScoreFunction ScoreFunction;
	typedef core::pose::Pose Pose;
public:
		DesignMinimizeHbonds() : DesignRepackMover( "DesignMinimizeHbonds" ) { min_rb_set_ = min_bb_set_ = min_sc_set_ = false; optimize_foldtree_ = true;
		automatic_repacking_definition_ = true; }
	DesignMinimizeHbonds( ScoreFunctionCOP scorefxn_repack, ScoreFunctionCOP scorefxn_minimize,
						  utility::vector1< core::Size > const target_residues, bool const donors, bool const acceptors,
						  bool const bb_hbond, bool const sc_hbond, core::Real const hbond_energy_threshold,
						  core::Real interface_distance_cutoff=8.0,
						  bool const repack_partner1=true, bool const repack_partner2=false, bool const repack_non_ala = true ) : DesignRepackMover( "DesignMinimizeHbonds" )
	{
		if( scorefxn_repack ) scorefxn_repack_ = new ScoreFunction( *scorefxn_repack );
		if( scorefxn_minimize ) scorefxn_minimize_ = new ScoreFunction( *scorefxn_minimize );
		target_residues_ = target_residues;
		repack_partner1_ = repack_partner1;
		repack_partner2_ = repack_partner2;
		donors_ = donors; acceptors_ = acceptors;
		bb_hbond_ = bb_hbond;
		sc_hbond_ = sc_hbond;
		hbond_energy_threshold_ = hbond_energy_threshold;
		interface_distance_cutoff_ = interface_distance_cutoff;
		repack_non_ala_ = repack_non_ala;
		// only requires donors_ || acceptors_ if doing sc hbonding
		runtime_assert( ( donors_ || acceptors_ ) || ( bb_hbond_ && !sc_hbond_ ) );
		runtime_assert( bb_hbond || sc_hbond );
		runtime_assert( hbond_energy_threshold_ <= 0 );
		runtime_assert (interface_distance_cutoff_ >= 0 );
	}
	DesignMinimizeHbonds( ScoreFunctionCOP scorefxn_repack, ScoreFunctionCOP scorefxn_minimize,
						  core::Size const target_residue, bool const donors, bool const acceptors,
						  bool const bb_hbond, bool const sc_hbond, core::Real const hbond_energy_threshold,
						  core::Real interface_distance_cutoff=8.0,
						  bool const repack_partner1=true, bool const repack_partner2=false, bool const repack_non_ala=true ) : DesignRepackMover( "DesignMinimizeHbonds" )
	{
		  if( scorefxn_repack ) scorefxn_repack_ = new ScoreFunction( *scorefxn_repack );
		  if( scorefxn_minimize ) scorefxn_minimize_ = new ScoreFunction( *scorefxn_minimize );
		  target_residues_.push_back( target_residue );
		  bb_hbond_ = bb_hbond;
		  sc_hbond_ = sc_hbond;
		  hbond_energy_threshold_ = hbond_energy_threshold;
		  repack_partner1_ = repack_partner1;
		  repack_partner2_ = repack_partner2;
		  interface_distance_cutoff_ = interface_distance_cutoff;
		  donors_ = donors; acceptors_ = acceptors;
			repack_non_ala_ = repack_non_ala;
	}
	DesignMinimizeHbonds( DesignMinimizeHbonds const & init ) : DesignRepackMover( init ) {
		hbond_energy_threshold_ = init.hbond_energy_threshold_;
		donors_ = init.donors_;
		acceptors_ = init.acceptors_;
		bb_hbond_ = init.bb_hbond_;
		sc_hbond_ = init.sc_hbond_;
	}
	void apply( Pose & pose );
	protocols::moves::MoverOP clone() const {
		return( protocols::moves::MoverOP( new DesignMinimizeHbonds( *this ) ) );
	}
	protocols::moves::MoverOP fresh_instance() const { return protocols::moves::MoverOP( new DesignMinimizeHbonds ); }
	void parse_my_tag( utility::Tag::TagPtr const tag, protocols::moves::DataMap &, protocols::filters::Filters_map const &, protocols::moves::Movers_map const &, core::pose::Pose const & );
	virtual ~DesignMinimizeHbonds() {};
private:
	bool donors_, acceptors_;
	bool bb_hbond_, sc_hbond_;
	core::Real hbond_energy_threshold_, interface_distance_cutoff_;
};

} // movers
} // ProteinInterfaceDesign
} // protocols


#endif /*INCLUDED_protocols_ProteinInterfaceDesign_movers_DesignMinimizeHbonds_HH*/
