// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.


/// @file protocols/ProteinInterfaceDesign/movers/HotspotHasherMover.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

#include <protocols/ProteinInterfaceDesign/movers/HotspotHasherMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/IDMover_cc_headers.hh>
#include <protocols/hotspot_hashing/HotspotStubSet.hh>
#include <protocols/hotspot_hashing/HotspotStub.hh>
#include <core/options/keys/out.OptionKeys.gen.hh>
#include <utility/file/file_sys_util.hh>

namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace core;
using namespace std;
using namespace core::scoring;
using namespace protocols::moves;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.HotspotHasherMover" );

void HotspotHasherMover::apply( core::pose::Pose & pose ) {

	// finding a request for ALL adds the main 18 aa's to
	if( std::find( resnames_.begin(), resnames_.end(), "ALL" ) != resnames_.end() ) {
		resnames_.push_back( "ALA" );
		resnames_.push_back( "ARG" );
		resnames_.push_back( "ASN" );
		resnames_.push_back( "ASP" );
		resnames_.push_back( "GLU" );
		resnames_.push_back( "GLN" );
		resnames_.push_back( "HIS" );
		resnames_.push_back( "ILE" );
		resnames_.push_back( "LEU" );
		resnames_.push_back( "LYS" );
		resnames_.push_back( "MET" );
		resnames_.push_back( "PHE" );
		resnames_.push_back( "PRO" );
		resnames_.push_back( "SER" );
		resnames_.push_back( "THR" );
		resnames_.push_back( "TRP" );
		resnames_.push_back( "TYR" );
		resnames_.push_back( "VAL" );
	}

	protocols::hotspot_hashing::HotspotStubSet stubset;

	// read existing hashes
	if ( utility::file::file_exists( hashin_fname_ ) ) {
		stubset.read( hashin_fname_ );
		TR << "Found hash file " << hashin_fname_ << std::endl;
	}
	if ( utility::file::file_exists( hashout_fname_ ) ) { // useful for interrupted runs
		stubset.read( hashout_fname_ );
		TR << "Found hash file " << hashout_fname_ << std::endl;
	}


	// for each residue requested
	for( std::vector< std::string >::const_iterator it=resnames_.begin() ; it!=resnames_.end(); ++it ) {
		std::string resname = *it;

		TR << "Hash contains " << stubset.size(resname) << " " << resname << " stubs." << std::endl;

		// check to see if we've already finished our hash
		core::Size stubs_left = n_stubs_;
		stubs_left -= stubset.size( resname );
		if( stubs_left <= 0 )
		{
			// perform a scorecut
			if ( core::options::option[ core::options::OptionKeys::out::scorecut ].user() )
			{
				Real score_cutoff = core::options::option[ core::options::OptionKeys::out::scorecut ]();
				std::stringstream i;
				i.str("");
				i << score_cutoff;
				stubset.clear();
				stubset.read( hashout_fname_ );
				protocols::hotspot_hashing::HotspotStubSetOP cut_stubs = stubset.subset( score_cutoff );
				std::string newfname = i.str() + "cut_" + hashout_fname_;
				cut_stubs->write_all( newfname );
			}
			return;
		}

		// do hashing in 10-stub cycles to minimize file i/o
		Size n_per(10);

		Size n_cycles = n_stubs_ / n_per;
		// make sure we do at least one cycle
		if( n_cycles <= 0 ) n_cycles = 1;
		// PERFORM HASHING
		for( Size i = 1; i <= n_cycles; ++i )
		{
			stubset.clear();
			stubset.score_threshold( score_threshold_ );
			TR << "Finding " << n_per*i << "/" << n_stubs_ << " " << resname << " stubs" ;
			if( target_resnum_ ) {
				TR << " " << target_distance_ << "A from " << target_resnum_ << std::endl;
				stubset.fill( pose, scorefxn_, target_resnum_, target_distance_, resname, n_per );
			}
			else {
				TR << "." << std::endl;
				stubset.fill( pose, scorefxn_, resname, n_per );
			}
			stubset.write_all( hashout_fname_ );
		}
	} // for each residue

	// perform a scorecut
	if ( core::options::option[ core::options::OptionKeys::out::scorecut ].user() )
	{
		Real score_cutoff = core::options::option[ core::options::OptionKeys::out::scorecut ]();
		std::stringstream i;
		i.str("");
		i << score_cutoff;
		stubset.clear();
		stubset.read( hashout_fname_ );
		protocols::hotspot_hashing::HotspotStubSetOP cut_stubs = stubset.subset( score_cutoff );
		std::string newfname = i.str() + "cut_" + hashout_fname_;
		cut_stubs->write_all( newfname );
	}
} // HotspotHasherMover::apply


void
HotspotHasherMover::parse_my_tag( TagPtr const tag, DataMap & data, protocols::filters::Filters_map const & filters, Movers_map const &, core::pose::Pose const & pose )
{

	std::string const scorefxn( tag->getOption<string>( "scorefxn", "score12" ));
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn)) );

	n_stubs_ = tag->getOption<core::Size>( "nstubs", 1000 );

	// target residue
	// target_resnum gets set below with residues
	target_resnum_ = 0;
	if( tag->hasOption( "target_residue_pdb_num" ) || tag->hasOption( "target_residue_res_num" ) ) {
		target_resnum_ = get_resnum( tag, pose, "target_residue_" );
	}

	target_distance_ = tag->getOption<core::Real>( "target_distance", 15.0 );

	score_threshold_ = tag->getOption<core::Real>( "threshold", -1.0 );

	// hash in/out
	hashin_fname_ = tag->getOption<std::string>( "in", "");
	hashout_fname_ = tag->getOption<std::string>( "out", "hash.stubs");

	// filter
	std::string const hotspot_filter_name( tag->getOption<std::string>( "hotspot_filter", "true_filter" ) );
	std::map< std::string const, protocols::filters::FilterCOP >::const_iterator find_filter( filters.find( hotspot_filter_name ));
	bool const filter_found( find_filter != filters.end() );
	if( filter_found )
		hotspot_filter_ = find_filter->second->clone();
	else {
		TR<<"***WARNING WARNING! Filter defined for HotspotHasher not found in filter_list!!!! Defaulting to truefilter***"<<std::endl;
		hotspot_filter_ = new protocols::filters::TrueFilter;
	}
	runtime_assert( filter_found );

	// residues
	std::vector< TagPtr > const hasher_tags( tag->getTags() );
	for( std::vector< TagPtr >::const_iterator hash_it=hasher_tags.begin(); hash_it!=hasher_tags.end(); ++hash_it ) {
		TagPtr const hash_tag_ptr = *hash_it;
		std::string tag_name = hash_tag_ptr->getName();
		if( tag_name == "residue" ) {
			std::string resname( hash_tag_ptr->getOption< std::string >( "type", "" ) );
			resnames_.push_back( resname );
		}
	}
	runtime_assert( resnames_.size() > 0 );

	TR<<"hashing mover finding residues: ";
	for( std::vector< std::string >::const_iterator it=resnames_.begin() ; it!=resnames_.end(); ++it ) TR<<*it<<" ";
	if( target_resnum_ ) TR << target_distance_ << "A away from residue " << target_resnum_ << std::endl;
	TR<<std::endl;
} // HotspotHasherMover::parse_my_tag

} //movers
} //ProteinInterfaceDesign
} //protocols
