// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/DesignRepackMover.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

#include <protocols/ProteinInterfaceDesign/movers/DesignRepackMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/InterfaceRecapitulationMover.hh>

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <protocols/ProteinInterfaceDesign/design_utils.hh>

#include <core/pack/task/TaskFactory.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreTypeManager.hh>
#include <core/scoring/ScoreType.hh>
#include <protocols/moves/Mover.hh>
#include <utility/Tag/Tag.hh>

// Utility Headers

// Unit Headers
#include <protocols/ProteinInterfaceDesign/util.hh>

// C++ headers
#include <map>
#include <string>
using namespace core;

// option key includes

using namespace core::scoring;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.InterfaceRecapitulationMover" );


namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace protocols::moves;

void
InterfaceRecapitulationMover::apply( core::pose::Pose & pose ){
	core::pose::Pose const saved_pose( pose );
	design_mover_->apply( pose );
	core::pack::task::PackerTaskCOP task( design_mover_->task() );

	core::Size designable_positions( 0 );
	for( Size i( 1 ); i<= pose.total_residue(); ++i )
		if( task->being_designed( i ) ) ++designable_positions;
	ReportSequenceDifferences rsd( core::scoring::ScoreFunctionFactory::create_score_function( STANDARD_WTS, SCORE12_PATCH ) );
	rsd.calculate( saved_pose, pose );
	std::map< core::Size, std::string > const res_names1( rsd.res_name1() );
	core::Size const mutated( res_names1.size() );
	core::Real const rate( (core::Real) mutated / designable_positions );
	TR<<"Your design mover mutated "<<mutated<<" positions out of "<<designable_positions<<" designable positions. Sequence recovery is: "<<rate<<std::endl;
}

void
InterfaceRecapitulationMover::parse_my_tag( utility::Tag::TagPtr const tag, protocols::moves::DataMap &, protocols::filters::Filters_map const &, protocols::moves::Movers_map const & movers, core::pose::Pose const & ){
	std::string const mover_name( tag->getOption<std::string>( "mover_name" ) );
	std::map< std::string const, MoverOP >::const_iterator find_mover( movers.find( mover_name ));
	bool const stub_mover_found( find_mover != movers.end() );
	if( stub_mover_found ){
		design_mover_ = dynamic_cast< DesignRepackMover * >( find_mover->second() );
		if( !design_mover_ ){
			TR<<"dynamic cast failed in tag "<<tag<<". Make sure that the mover is derived from DesignRepackMover"<<std::endl;
			runtime_assert( design_mover_ );
		}
	}
}

} //movers
} //ProteinInterfaceDesign
} //protocols
