// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/movers/RepackMinimize.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

#include <protocols/ProteinInterfaceDesign/movers/RepackMinimize.hh>
#include <protocols/ProteinInterfaceDesign/movers/IDMover_cc_headers.hh>
#include <protocols/ProteinInterfaceDesign/design_utils.hh>
#include <core/pack/pack_rotamers.hh>

namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using namespace core;
using namespace std;
using namespace core::scoring;
using namespace protocols::moves;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.RepackMinimize" );

/// @details designs interface residues and minimizes the pose.
/// If minimization parameters have not been set by the user the default minimization behaviour is as follows:
/// + minimize bb/sc for all residues that are repacked.
/// + minimize bb/sc of residues that are +-1 from target residues
/// + minimize bb for all residues (minimizing over only interface residues causes large motions during minimization)
/// + minimize rb jump
/// Note that some confusion may arise from the use of repack1_ and repack2_
/// The behaviour of these variables is such that if they're false, then the
/// relevant partner is not designed but may be repacked. Turning off design_
/// precludes design across the whole system only allowing repack. At one point
/// the names should be rethought...
void
RepackMinimize::apply( pose::Pose & pose )
{
	allowed_aas_[ chemical::aa_cys ] = false;
	allowed_aas_[ chemical::aa_gly ] = false;
	allowed_aas_[ chemical::aa_pro ] = false;

	setup_packer_and_movemap( pose );
	using namespace core::scoring;

	pack::pack_rotamers( pose, *scorefxn_repack_, task_ );

	MinimizeInterface( pose, scorefxn_minimize_, curr_min_bb_, curr_min_sc_, curr_min_rb_, optimize_foldtree_, target_residues_ );
	pose.update_residue_neighbors();
	(*scorefxn_minimize_)( pose );
	/// Now handled automatically.  scorefxn_minimize_->accumulate_residue_total_energies( pose );
}


void
RepackMinimize::parse_my_tag( TagPtr const tag, DataMap & data, protocols::filters::Filters_map const &filters, Movers_map const & movers, core::pose::Pose const & pose )
{
	TR<<"repack minimize mover with the following parameters:\n";
	DesignRepackMover::parse_my_tag( tag, data, filters, movers, pose );
}

} //movers
} //ProteinInterfaceDesign
} //protocols
