// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file protocols/ProteinInterfaceDesign/movers/ddG.cc
/// @brief implementation of the ddG class for computing interface delta dGs
/// @author Sarel Fleishman (sarelf@u.washington.edu)


#include <core/types.hh>
#include <core/conformation/Residue.hh>
#include <core/conformation/Interface.hh>
//#include <protocols/moves/ResidueMover.hh>
#include <core/chemical/ResidueType.hh>
#include <core/chemical/ChemicalManager.hh>

#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>

#include <core/pack/pack_rotamers.hh>
#include <core/pack/task/PackerTask.hh>
#include <core/pack/task/TaskFactory.hh>

#include <core/kinematics/MoveMap.hh>

#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>

#include <core/pose/Pose.hh>
#include <core/options/util.hh>
#include <core/init.hh>

#include <core/io/pdb/pose_io.hh>

#include <numeric/xyzVector.hh>
#include <numeric/random/random.hh>

#include <protocols/ProteinInterfaceDesign/movers/DesignRepackMover.hh>
#include <protocols/ProteinInterfaceDesign/movers/ddG.hh>

#include <protocols/moves/PackRotamersMover.hh>
#include <protocols/docking/RestrictTaskForDocking.hh>

// C++ headers
#include <map>

#include <core/util/Tracer.hh>

namespace protocols {
namespace ProteinInterfaceDesign {
namespace movers {

using core::util::T;
using core::util::Error;
using core::util::Warning;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.movers.ddG" );

using namespace core;
using namespace protocols::ProteinInterfaceDesign;
using namespace core::scoring;

/// @details a private function for storing energy values
void
ddG::fill_energy_vector( pose::Pose const & pose, std::map< ScoreType, core::Real > & energy_map )
{
	using namespace core::scoring;
	for ( int i=1; i<= n_score_types; ++i ) {
		if ( (*scorefxn_)[ ScoreType(i) ] != 0.0 && ScoreType(i) != pro_close ) {
			energy_map.insert( std::make_pair( ScoreType( i ), (*scorefxn_)[ ScoreType(i)] * pose.energies().total_energies()[ ScoreType( i ) ] ) );
		}
	}
}

/// @details output the ddG values
void
ddG::report_ddG( std::ostream & out ) const
{
	out << "-----------------------------------------\n";
	out << " Scores                       Wghtd.Score\n";
	out << "-----------------------------------------\n";
	std::map< ScoreType, Real >::const_iterator unbound_it=unbound_energies_.begin();
	for( std::map< ScoreType, Real >::const_iterator bound_it=bound_energies_.begin();
			 bound_it!=bound_energies_.end();
			 ++bound_it ) {
			 if( std::abs( unbound_it->second ) > 0.001 || std::abs( bound_it->second ) > 0.001 )
				out << ' ' << LJ( 24, bound_it->first ) << ' ' << F( 9,3, bound_it->second - unbound_it->second )<<'\n';
		++unbound_it;
	}
	out << "-----------------------------------------\n";
	out << "Sum ddg: "<< sum_ddG()<<'\n';
}

/// @details returns the total ddG
Real
ddG::sum_ddG() const
{
	Real sum_energy(0.0);

	std::map< ScoreType, Real >::const_iterator unbound_it=unbound_energies_.begin();
	for( std::map< ScoreType, Real >::const_iterator bound_it=bound_energies_.begin();
			 bound_it!=bound_energies_.end();
			 ++bound_it ) {
		sum_energy += bound_it->second - unbound_it->second;
		++unbound_it;
	}
	return sum_energy;
}

/// @details compute the energy of the repacked complex in the bound and unbound states
void
ddG::calculate( pose::Pose const & pose_in )
{
	using namespace pack;
	using namespace protocols::moves;

	pose::Pose pose = pose_in;

	setup_packer_and_movemap( pose );
	pack::pack_rotamers( pose, *scorefxn_, task_ );
	(*scorefxn_)( pose );
	fill_energy_vector( pose, bound_energies_ );

	RigidBodyTransMoverOP translate( new RigidBodyTransMover( pose, rb_jump_ ) );
	translate->step_size( 1000.0 );
	translate->apply( pose );
	pack::pack_rotamers( pose, *scorefxn_, task_ );
	(*scorefxn_)( pose );
	fill_energy_vector( pose, unbound_energies_ );
}

} //movers
} //ProteinInterfaceDesign
} //protocols
