// -*- mode:c++;tab-width:2;indent-tabs-mode:t;show-trailing-whitespace:t;rm-trailing-spaces:t -*-
// vi: set ts=2 noet:
//
// (c) Copyright Rosetta Commons Member Institutions.
// (c) This file is part of the Rosetta software suite and is made available under license.
// (c) The Rosetta software is developed by the contributing members of the Rosetta Commons.
// (c) For more information, see http://www.rosettacommons.org. Questions about this can be
// (c) addressed to University of Washington UW TechTransfer, email: license@u.washington.edu.

/// @file devel/ProteinInterfaceDesign/dock_design_filters.cc
/// @brief
/// @author Sarel Fleishman (sarelf@u.washington.edu), Jacob Corn (jecorn@u.washington.edu)

// Project Headers
#include <core/types.hh>
#include <core/pose/Pose.hh>
#include <core/pose/PDBInfo.hh>

#include <core/conformation/Interface.hh>
#include <core/optimization/AtomTreeMinimizer.hh>
#include <core/optimization/MinimizerOptions.hh>
#include <core/kinematics/MoveMap.hh>
#include <core/pack/task/TaskFactory.hh>
#include <core/scoring/ScoreFunction.hh>
#include <core/scoring/ScoreFunctionFactory.hh>
#include <core/scoring/ScoreTypeManager.hh>
#include <core/scoring/ScoreType.hh>
#include <protocols/docking/DockingProtocol.hh>
#include <protocols/moves/ReturnSidechainMover.hh>
#include <core/chemical/ChemicalManager.hh>
//#include <protocols/moves/ResidueMover.hh>
#include <protocols/moves/PeptideStapleMover.hh>
#include <protocols/toolbox/PoseMetricCalculators/SasaCalculator.hh>
#include <core/util/MetricValue.hh>
#include <protocols/toolbox/PoseMetricCalculators/InterfaceSasaDefinitionCalculator.hh>
#include <numeric/random/random.hh>
#include <core/pose/PDBPoseMap.hh>
#include <utility/Tag/Tag.hh>
#include <protocols/moves/Mover.hh>
#include <protocols/moves/DataMap.hh>

// Utility Headers

// Unit Headers
#include <protocols/ProteinInterfaceDesign/dock_design_filters.hh>
#include <protocols/ProteinInterfaceDesign/movers/ddG.hh>
#include <protocols/ProteinInterfaceDesign/design_utils.hh>
#include <protocols/ProteinInterfaceDesign/util.hh>

// C++ headers
#include <map>
#include <string>
using namespace core;
using namespace core::scoring;

static core::util::Tracer TR( "protocols.ProteinInterfaceDesign.dock_design_filters" );

namespace protocols {
namespace ProteinInterfaceDesign {
using namespace protocols::filters;

using namespace protocols::moves;

using namespace utility::Tag;
using namespace std;

void
ResiduesInInterfaceFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	residues_in_interface_threshold_ = tag->getOption<core::Size>( "residues", 20 );
	rb_jump_ = tag->getOption<core::Size>( "jump_number", 1 );

	TR<<"residues in interface filter over jump number " << rb_jump_ << " with threshold "<<residues_in_interface_threshold_<<std::endl;
}

void
ScoreTypeFilter::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	using namespace core::scoring;

	std::string const scorefxn_name( tag->getOption<string>( "scorefxn", "score12" ) );
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name )) );
	score_type_ = core::scoring::score_type_from_name( tag->getOption<string>( "score_type" ) );
	score_type_threshold_ = tag->getOption<core::Real>( "threshold" );

	TR<<"ScoreType filter for score_type "<<score_type_<<" with threshold "<<score_type_threshold_<<std::endl;
}

void
InterfaceSasaFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	lower_threshold_ = tag->getOption<core::Real>( "threshold", 800 );

	TR<<"SasaFilter with lower threshold of "<<lower_threshold_<<" Ang^2"<<std::endl;
}

void
AlaScan::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	distance_threshold_ = tag->getOption<core::Real>( "interface_distance_cutoff", 8.0 );
	chain1_ = tag->getOption< bool >( "partner1", 0 );
	chain2_ = tag->getOption< bool >( "partner2", 1 );
	runtime_assert( chain1_ || chain2_ );
	std::string const scorefxn_name( tag->getOption< string >( "scorefxn", "score12" ));
	repeats_ = tag->getOption< core::Size >( "repeats", 1 );
	using namespace core::scoring;
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name )) );
	TR<<"AlaScan with distance threshold of "<<distance_threshold_<<" Ang "<<". partner1="<<chain1_<<". partner2="<<chain2_<<" using "<<repeats_<<" repeats."<<std::endl;
}

void
NeighborTypeFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
  residue_types_.assign( chemical::num_canonical_aas, false );
	std::vector< TagPtr > const neighbor_type_tags( tag->getTags() );
	for( std::vector< TagPtr >::const_iterator nt_it=neighbor_type_tags.begin(); nt_it!=neighbor_type_tags.end(); ++nt_it ) {
    TagPtr const nt_tag_ptr = *nt_it;
    if( nt_tag_ptr->getName() == "Neighbor" ) {
			std::string const type( nt_tag_ptr->getOption<string>( "type" ) );
			residue_types_[ chemical::aa_from_name( type ) ] = true;
    }
	}
	target_residue_ = get_resnum( tag, pose );
	distance_threshold_ = tag->getOption<core::Real>( "distance", 8.0 );

	TR<<"NeighborTypeFilter with distance threshold of "<<distance_threshold_<<" around residue "<<target_residue_<<std::endl;
}

void
ResidueBurialFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
	target_residue_ = get_resnum( tag, pose );
	distance_threshold_ = tag->getOption<core::Real>( "distance", 8.0 );
	neighbors_ = tag->getOption<core::Size>( "neighbors", 1 );

	TR<<"ResidueBurialFilter with distance threshold of "<<distance_threshold_<<" around residue "<<target_residue_<<" with "<<neighbors_<<" neighbors."<<std::endl;
}

void
ResidueDistanceFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
	res1_ = get_resnum( tag, pose, "res1_" );
	res2_ = get_resnum( tag, pose, "res2_" );
	distance_threshold_ = tag->getOption<core::Real>( "distance", 8.0 );

	TR<<"ResidueDistanceFilter with distance threshold of "<<distance_threshold_<<" between residues "<<res1_<<" and "<<res2_<<std::endl;
}

void
DdgFilter::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const & , Movers_map const & , core::pose::Pose const & )
{
	using namespace core::scoring;

	std::string const scorefxn_name( tag->getOption<string>( "scorefxn", "score12" ) );
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name )) );
	ddg_threshold_ = tag->getOption<core::Real>( "threshold", -15 );
	rb_jump_ = tag->getOption< core::Size >( "jump", 1 );

  TR<<"ddg filter with threshold "<< ddg_threshold_<<" and scorefxn "<<scorefxn_name<<" over jump "<<rb_jump_<<std::endl;
}

void
HbondsToResidueFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
	partners_ = tag->getOption<core::Size>( "partners" );
	energy_cutoff_ = tag->getOption<core::Real>( "energy_cutoff", -0.5 );
	backbone_ = tag->getOption<bool>( "backbone", 0 );
	sidechain_ = tag->getOption<bool>( "sidechain", 1 );
	resnum_ = get_resnum( tag, pose );

	TR<<"Hbonds to residue filter for resnum "<<resnum_<<" with "<<partners_<<" hbonding partners"<<std::endl;
}

void
EnergyPerResidueFilter::parse_my_tag( TagPtr const tag, DataMap & data, Filters_map const &, Movers_map const &, core::pose::Pose const & pose )
{
	using namespace core::scoring;

	std::string const scorefxn_name( tag->getOption<string>( "scorefxn", "score12" ) );
	scorefxn_ = new ScoreFunction( *(data.get< ScoreFunction * >( "scorefxns", scorefxn_name ) ));
	score_type_ = core::scoring::score_type_from_name( tag->getOption<string>( "score_type", "total_score" ) );
	threshold_ = tag->getOption<core::Real>( "energy_cutoff", 0.0 );
	whole_interface_ = tag->getOption<bool>( "whole_interface" , 0 );
	rb_jump_ = tag->getOption<core::Size>( "jump_number", 1 );
	interface_distance_cutoff_ = tag->getOption<core::Real>( "interface_distance_cutoff" , 8.0 );

	if (whole_interface_==1 ) {
		resnum_ = 1;
		TR<<"energies for all interface residues with a distance cutoff of "
		<< interface_distance_cutoff_ << " A will be calculated \n"
		<< "jump_number is set to "<< rb_jump_
		<< "\n and scorefxn " <<scorefxn_name <<" will be used" <<std::endl;
	}
	else {
		resnum_ = get_resnum( tag, pose );
		TR<<"EnergyPerResidueFilter for residue "<<resnum_<<" of score_type "<<score_type_<<" with cutoff "<<threshold_<<std::endl;
		}
}

void
BuriedUnsatHbondFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	jump_num_ = tag->getOption<core::Size>( "jump_number", 1 );
	upper_threshold_ = tag->getOption<core::Size>( "cutoff", 20 );

	TR<<"Buried Unsatisfied Hbond filter over jump number " << jump_num_ << " with cutoff " << upper_threshold_ << std::endl;
}

void
TerminusDistanceFilter::parse_my_tag( TagPtr const tag, DataMap &, Filters_map const &, Movers_map const &, core::pose::Pose const & )
{
	jump_num_ = tag->getOption<core::Size>( "jump_number", 1 );
	distance_ = tag->getOption<core::Size>( "distance", 5 );

	TR<<"Distance From Terminus filter over jump number " << jump_num_ << " with cutoff " << distance_ << std::endl;
}


} // ProteinInterfaceDesign
} // devel
